/* Copyright 2025-2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: Zlib
 */
#define TEXTRENDER_FT2_IMPL

#ifndef TEXTRENDER_FT2_ALLOCATOR
#define TEXTRENDER_FT2_ALLOCATOR  g_allocator
#endif

#define VECTOR_DEF_ALLOC  TEXTRENDER_FT2_ALLOCATOR

#include "textrender_ft2.h"
#include "bisect.h"
#include "alloc.h"
#include "logging.h"
#include "vector.h"

// This macros convert a freetype fixed point number and converted to an
// interger using the floor or ceil operation.
#define FT_FLOOR(X) ((X & -64) / 64)
#define FT_CEIL(X)	(((X + 63) & -64) / 64)

// Global library object
// According to the documentations, multithreaded applications are fine as long as
// font open (FT_New_Face) and close (FT_Done_Face) are done single-threadly.
FT_Library ft2_library;
unsigned ft2_lib_init = 0;

void textrender_ft2_glyph_destroy(TextRenderGlyph* G)
{
	alloc_free(TEXTRENDER_FT2_ALLOCATOR, G);
}

TextRenderGlyph* textrender_ft2_glyph_create(TextRenderFT2* obj,
	unsigned cp)
{
	int r;

	FT_Activate_Size(obj->ft_size);

	FT_UInt glyph_index = FT_Get_Char_Index(obj->ft_face, cp);
	if (glyph_index == 0 && cp != 0) goto error;  //missing

	// Load glyph
	int load_flags = FT_LOAD_TARGET_MONO;
	if (obj->cfg.bold > 0 || obj->cfg.slanted > 0)
		load_flags |= FT_LOAD_NO_BITMAP;
	r = FT_Load_Glyph(obj->ft_face, glyph_index, load_flags);
	if (r) {
		log_error("FT_Load_Glyph: %d", r);
		goto error;
	}

	FT_GlyphSlot slot = obj->ft_face->glyph;	 //shortcut

	// Style: bold
	if (obj->cfg.bold > 0) {
		FT_Pos strength = obj->cfg.bold;
		if (slot->format == FT_GLYPH_FORMAT_OUTLINE)
			FT_Outline_Embolden(&slot->outline, strength);
		else
			FT_Bitmap_Embolden(slot->library, &slot->bitmap, strength, strength);
	}

	// Style: slanted ("italic")
	if (obj->cfg.slanted > 0 && slot->format == FT_GLYPH_FORMAT_OUTLINE) {
		int factor = obj->cfg.slanted;
		FT_Matrix shear;
		shear.xx = 1 << 16;
		shear.xy = (int) (factor * (1 << 16) / 1000);
		shear.yx = 0;
		shear.yy = 1 << 16;
		FT_Outline_Transform(&slot->outline, &shear);
	}

	// Style: rotation
	//...

	// Style: stroke
/*	if( style & FSF_stroke ){
		FT_Fixed radius = 0.02 * ( ptsize * dpi / 72 );
		FT_Glyph glyph;
		FT_Stroker stroker;
		FT_Get_Glyph( slot, &glyph );
		FT_Stroker_Set( stroker, radius,
						FT_STROKER_LINECAP_ROUND,
						FT_STROKER_LINEJOIN_ROUND,
						0 );
		FT_Glyph_Stroke( &slot, stroker, 1 );
		FT_Stroker_Done( stroker );
		FT_Done_Glyph( glyph );
	}*/

	// Render glyph
	r = FT_Render_Glyph(slot, FT_RENDER_MODE_NORMAL);	//256 levels of gray
	if (r) {
		log_error("FT_Render_Glyph: %d", r);
		goto error;
	}

	if (!( slot->bitmap.pixel_mode == FT_PIXEL_MODE_GRAY
		&& slot->bitmap.num_grays == 256
		&& slot->bitmap.palette == 0
		&& slot->bitmap.pitch == (int)slot->bitmap.width ))
	{
		log_error("ft2 glyph format invalid");
		goto error;
	}

	// Save glyph
	size_t bm_sz = slot->bitmap.pitch * slot->bitmap.rows;
	TextRenderGlyph * G = alloc_alloc(TEXTRENDER_FT2_ALLOCATOR,
		sizeof(TextRenderGlyph) + bm_sz);

	G->bitmap_width  = slot->bitmap.pitch;
	G->bitmap_height = slot->bitmap.rows;
	G->bitmap_left   = slot->bitmap_left;
	G->bitmap_top    = slot->bitmap_top;
	G->advance_x     = slot->advance.x >> 6;
	G->glyph_index   = glyph_index;
	G->bitmap_data   = (void*)(G+1); 
	memcpy(G->bitmap_data, slot->bitmap.buffer, bm_sz);

	log_debug2("ft2 glyph retrieved: %u %u %dx%d",
		cp, G->glyph_index, G->bitmap_width, G->bitmap_height);
	return G;

error:
	return obj->glyph_missing;
}

void textrender_ft2_close(TextRenderFT2* obj)
{
	// Glyphs cache free
	vec_forr(obj->glyph_cache, i) {
		if (obj->glyph_cache[i].glyph == obj->glyph_missing) continue;
		textrender_ft2_glyph_destroy(obj->glyph_cache[i].glyph);
	}
	vec_free(obj->glyph_cache);
	
	vec_forr(obj->glyph_ascii, i) {
		if (obj->glyph_ascii[i] == obj->glyph_missing) continue;
		textrender_ft2_glyph_destroy(obj->glyph_ascii[i]);
	}
	vec_free(obj->glyph_ascii);

	textrender_ft2_glyph_destroy(obj->glyph_missing);

	// Face free
	if (!obj->ft_face) return;

	FT_Done_Face(obj->ft_face);
	obj->ft_face = NULL;
	obj->ft_size = NULL;

	// Library free
	if (ft2_lib_init == 1) {
		FT_Done_FreeType(ft2_library);
		ft2_library = NULL;
	}
	ft2_lib_init--;
}

int textrender_ft2_open(TextRenderFT2* obj)
{
	int R=1, r;

	if (obj->ft_face)
		textrender_ft2_close(obj);

	if (!(obj->cfg.path && obj->cfg.path[0])) return 0; 

	// Library initialization
	if (ft2_lib_init == 0) {
		r = FT_Init_FreeType(&ft2_library);
		if (r) ERROR_LOG(-1, "FreeType initialization failed: %d", r);
	}
	ft2_lib_init++;

	// Open font file
	log_debug("Loading font '%s'", obj->cfg.path);
	r = FT_New_Face(ft2_library, obj->cfg.path, obj->cfg.index, &obj->ft_face);
	if (r) ERROR_LOG(-1, "Could not open font '%s': %d", obj->cfg.path, r);

	r = FT_New_Size(obj->ft_face, &obj->ft_size);
	if (r) ERROR_LOG(-1, "FT_New_Size: %d", r);
	
	FT_Activate_Size(obj->ft_size);

	r = FT_Set_Pixel_Sizes(obj->ft_face, 0, obj->cfg.size);
	if (r) ERROR_LOG(-1, "FT_Set_Char_Size: %d", r);

	//TODO: configure style ?
	//TTF_SetFontStyle(font, outline);
	//TTF_SetFontOutline(font, outline);
	//TTF_SetFontHinting(font, hinting);
	//TTF_SetFontKerning(font, (flags & OF_NoKerning) ? 0 : 1);

	// Initiate glyph cache
	obj->glyph_missing = textrender_ft2_glyph_create(obj, 0);
	if (!obj->glyph_missing) ERROR_LOG(-1, "textrender ft2 missing glyph create");

	obj->ascii_cp_first = 0x20;
	obj->ascii_cp_last  = 0x7e;
	vec_resize(obj->glyph_ascii, obj->ascii_cp_last - obj->ascii_cp_first + 1);
	vec_for(obj->glyph_ascii, i, 0) {
		unsigned cp = obj->ascii_cp_first + i;
		obj->glyph_ascii[i] = textrender_ft2_glyph_create(obj, cp);
	}

	if (log_level_check(LOG_LVL_DEBUG)) {
		int w=0, h=0;
		vec_for(obj->glyph_ascii, i, 0) {
			MAXSET(w, obj->glyph_ascii[i]->bitmap_width);
			MAXSET(h, obj->glyph_ascii[i]->bitmap_height);
		}
		log_debug("font ascii glyph max size: %dx%d", w, h);
	}

	// Calculate metrics
	FT_Fixed scale = obj->ft_face->size->metrics.y_scale;
	obj->B.metrics.ascent = FT_CEIL(FT_MulFix(obj->ft_face->ascender, scale));
	obj->B.metrics.descent = -FT_CEIL(FT_MulFix(obj->ft_face->descender, scale));		//negative
	obj->B.metrics.height = obj->B.metrics.ascent + obj->B.metrics.descent + /* baseline */ 1;
	obj->B.metrics.lineskip = FT_CEIL(FT_MulFix(obj->ft_face->height, scale));
	//obj->B.metrics.underline_offset = FT_FLOOR(FT_MulFix(obj->ft_face->underline_position, scale));
	//obj->B.metrics.underline_height = FT_FLOOR(FT_MulFix(obj->ft_face->underline_thickness, scale));
	textrender_metrics_update(&obj->B);

end:
	if (R<0) textrender_ft2_close(obj);
	return R;
}

const TextRenderGlyph* textrender_ft2_glyph_get(TextRenderFT2* obj, unsigned cp)
{
	if (cp < obj->ascii_cp_first)
		return obj->glyph_missing;
	
	if (cp <= obj->ascii_cp_last)
		return obj->glyph_ascii[ cp - obj->ascii_cp_first ];

	BISECT_RIGHT_DECL(found, idx, 0, vec_count(obj->glyph_cache),
		obj->glyph_cache[i_].cp - cp);
	if (!found) {
		TextRenderGlyph * glyph = textrender_ft2_glyph_create(obj, cp);
		struct TextRenderFT2GlyphCacheItem gci = {cp, glyph};
		vec_insert(obj->glyph_cache, idx, 1, &gci);
	}
	return obj->glyph_cache[idx].glyph;
}

ImgPoint textrender_ft2_kerning_get(TextRenderFT2* obj, unsigned glyph_left,
	unsigned glyph_right)
{
	FT_Vector kerning;
	FT_Get_Kerning(obj->ft_face, glyph_left, glyph_right, FT_KERNING_DEFAULT,
		&kerning);
	return (ImgPoint){ kerning.x >> 6, kerning.y >> 6 };
}

void textrender_ft2_destroy(TextRenderFT2* obj)
{
	textrender_ft2_close(obj);
	alloc_free(TEXTRENDER_FT2_ALLOCATOR, obj);
}

const TextRenderClass textrender_class_ft2 = {
	(const TextRenderGlyph* (*)(TextRender*, unsigned)) textrender_ft2_glyph_get,
	(ImgPoint (*)(TextRender*, unsigned, unsigned)) textrender_ft2_kerning_get,
	(void (*)(TextRender*)) textrender_ft2_destroy,
};

TextRender* textrender_ft2_create_cfg(const TextRenderFT2Config* cfg)
{
	TextRenderFT2* tr = alloc_new(TEXTRENDER_FT2_ALLOCATOR, TextRenderFT2, 1);
	tr->B.cls = &textrender_class_ft2;
	tr->cfg = *cfg;
	int r = textrender_ft2_open(tr);
	if (r < 0) {
		alloc_free(TEXTRENDER_FT2_ALLOCATOR, tr);
		tr = NULL;
	}
	return &tr->B;
}
