/* Copyright 2025-2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: Zlib
 *
 * Renders text to an image.
 */
#pragma once
#include "image.h"

typedef enum {
	// Disable alpha blending (transparency). ~40% runtime reduction.
	TR_RF_FAST_BLIT			= 0x0001,
	// Disable UTF-8 decoding. ~3ns per character
	TR_RF_NO_UTF8			= 0x0002,
	// Kerning is the adjustment of the relative positions of certain pairs of
	// characters (e.g. "AV"). With freetype2, it may take ~30ns per character.
	TR_RF_NO_KERNING		= 0x0004,

	TR_RF_ALIGN_RIGHT		= 0x0100,
	TR_RF_ALIGN_HCENTER		= 0x0200,
	TR_RF_ALIGN_BOTTOM		= 0x0400,
	TR_RF_ALIGN_VCENTER		= 0x0800,
	TR_RF_ALIGN_CENTER		= TR_RF_ALIGN_HCENTER | TR_RF_ALIGN_VCENTER,
	TR_RF_MASK_ALIGN		= 0x0F00,
} textrender_render_flags_t;

typedef struct TextRenderGlyph {
	unsigned char *	bitmap_data;
	int				bitmap_width;
	int				bitmap_height;
	int				bitmap_left;
	int				bitmap_top;
	int				advance_x;
	unsigned		glyph_index;
} TextRenderGlyph;

typedef struct TextRender TextRender;

typedef struct TextRenderClass {
	const TextRenderGlyph* (*glyph_get)(TextRender*, unsigned);
	ImgPoint (*kerning_get)(TextRender*, unsigned, unsigned);
	void (*destroy)(TextRender*);
} TextRenderClass;

struct TextRender {
	const TextRenderClass* cls;
	
	struct TextRenderMetrics {
		unsigned	height;
		unsigned	ascent;
		unsigned	descent;
		unsigned	lineskip;
		unsigned	mean_width;		//mean character width
		unsigned	num_width;		//width of a number
		unsigned	m_width;		//width of an 'm'
		unsigned	tab_width;
	} metrics;
	
	struct TextRenderConfig {
		int rflags;  // Flags always used by _render and _size_get
	} c;
};

// Use the class-specific constructor to create a new object

void textrender_destroy(TextRender* obj);

// Obtain the rendered size of a text (in pixels).
int textrender_size_get(TextRender* obj, ImgPoint* size,
	const char* text, const char* text_end, int flags);

// Render text to an image.
// If non-null, dstrect will be filled with the bounding box.
int textrender_render(TextRender* obj, Image* img, ImgRect* dstrect,
	const char* text, const char* text_end, const ImgColor col, int flags);

// Render text to a new image.
int textrender_render_new(TextRender* obj, Image* img, ImgFormat fmt,
	const char* text, const char* text_end,
	const ImgColor col_fg, const ImgColor col_bg, int flags);

// Updates the metrics.
// Mostly for internal use.
void textrender_metrics_update(TextRender* obj);

/* "C" text renderer using an embedded font.
 */
typedef struct TextRenderC {
	TextRender B;
	const TextRenderGlyph* glyphs;
	unsigned cp_first, cp_last;
} TextRenderC;

extern const TextRenderClass textrender_class_c;
