/* Copyright 2024, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: Zlib
 */
#define STRUCTIO_IMPLEMENTATION
#include "structio_ubjson.h"

/*
	UBJSON Encoder
*/
static inline
int stio_ubjson_write_open_string(StioStream* sio, const Any* value,
	AnyBaseType* pctx_type);

static inline
int stio_ubjson_write_value(StioStream* sio, const Any* value,
	AnyBaseType* pctx_type);

static inline
int stio_ubjson_write_chunk_array(StioStream* sio, const Any* value)
{
	assert( value->t == ANY_T_ARRAY );
	assert( value->p.ap != NULL );
	assert( value->len != ANYT_LENGTH_INDEF );

	const Any* ac = value->p.ap;
	for (uint32_t i=0, e=value->len; i<e; ++i)
	{
		TRYR( stio_ubjson_write_value(sio, ac, NULL) );
		ac++;
	}

	return STIO_R_OK;
}

static inline
int stio_ubjson_write_chunk_map(StioStream* sio, const Any* value)
{
	assert( value->t == ANY_T_MAP );
	assert( value->p.ap != NULL );
	assert( value->len != ANYT_LENGTH_INDEF );

	const Any* ac = value->p.ap;
	for (uint32_t i=0, e=value->len; i<e; ++i)
	{
		if (ac->t != ANY_T_STRING) return STIO_E_VALUE;
		TRYR( stio_ubjson_write_open_string(sio, ac, NULL) );
		ac++;
		TRYR( stio_ubjson_write_value(sio, ac, NULL) );
		ac++;
	}

	return STIO_R_OK;
}

static inline
int stio_ubjson_write_chunk_string(StioStream* sio, const Any* value)
{
	assert( value->t == ANY_T_STRING );
	assert( value->p.p != NULL );
	assert( value->len != ANYT_LENGTH_INDEF );
	TRYR( stream_write_chk(sio->s, value->len, value->p.p) );
	return STIO_R_OK;
}

static inline
int stio_ubjson_write_chunk_vector(StioStream* sio, const Any* value,
	AnyBaseType ctx_type)
{
	assert( anyb_pointer_is(value->t) );
	assert( value->p.p != NULL );
	assert( value->len != ANYT_LENGTH_INDEF );

	AnyBaseType stype = anyb_pointer_deref(value->t);
	const size_t sstep = anyb_size(stype);

	if (ctx_type == ANY_T_ARRAY)
	{
		Any v = { .t=stype };
		const unsigned char* ac = value->p.p;
		for (uint32_t i=0, e=value->len; i<e; ++i)
		{
			memcpy(&v.p, ac, sstep);
			TRYR( stio_ubjson_write_value(sio, &v, NULL) );
			ac += sstep;
		}
	}
	else
	{
		bool swap = !big_endian_is() && ~sio->cflags & STIO_SF_NO_BYTE_SWAP;
		TRYR( stio_write_chunk_vector(sio, value, ctx_type, swap) );
	}

	return STIO_R_OK;
}

#define UBJSON_INT_PUT(_VAR_) do { \
	if (INT8_MIN <= (_VAR_) && (_VAR_) <= INT8_MAX) { \
		TRYR( stream_char_put(sio->s, 'i') ); \
		TRYR( stream_char_put(sio->s, (_VAR_)) ); \
	} \
	else if (INT16_MIN <= (_VAR_) && (_VAR_) <= INT16_MAX) { \
		TRYR( stream_char_put(sio->s, 'I') ); \
		TRYR( stio_stream_write_be16(sio, &(_VAR_)) ); \
	} \
	else if (INT32_MIN <= (_VAR_) && (_VAR_) <= INT32_MAX) { \
		TRYR( stream_char_put(sio->s, 'l') ); \
		TRYR( stio_stream_write_be32(sio, &(_VAR_)) ); \
	} \
	else { \
		TRYR( stream_char_put(sio->s, 'L') ); \
		TRYR( stio_stream_write_be64(sio, &(_VAR_)) ); \
	} \
} while(0)

#define UBJSON_UINT_PUT(_VAR_) do { \
	if ((_VAR_) <= INT8_MAX) { \
		TRYR( stream_char_put(sio->s, 'i') ); \
		TRYR( stream_char_put(sio->s, (_VAR_)) ); \
	} \
	else if ((_VAR_) <= INT16_MAX) { \
		TRYR( stream_char_put(sio->s, 'I') ); \
		TRYR( stio_stream_write_be16(sio, &(_VAR_)) ); \
	} \
	else if ((_VAR_) <= INT32_MAX) { \
		TRYR( stream_char_put(sio->s, 'l') ); \
		TRYR( stio_stream_write_be32(sio, &(_VAR_)) ); \
	} \
	else { \
		TRYR( stream_char_put(sio->s, 'L') ); \
		TRYR( stio_stream_write_be64(sio, &(_VAR_)) ); \
	} \
} while(0)

static inline
int stio_ubjson_write_open_string(StioStream* sio, const Any* value,
	AnyBaseType* pctx_type)
{
	if (value->len == ANYT_LENGTH_INDEF) return STIO_E_VALUE;
	UBJSON_UINT_PUT(value->len);

	if (value->len == 0) return STIO_R_OK;
	else if (value->p.p) return stio_ubjson_write_chunk_string(sio, value);
	else if (pctx_type) { *pctx_type=ANY_T_STRING; return STIO_R_CTX_BEGIN; }
	else return STIO_E_NESTING;
}

static inline
int stio_ubjson_write_open_array(StioStream* sio, const Any* value,
	AnyBaseType* pctx_type)
{
	if (value->len != ANYT_LENGTH_INDEF) {
		TRYR( stream_char_put(sio->s, '#') );
		UBJSON_UINT_PUT(value->len);
	}

	if (value->len == 0) return STIO_R_OK;
	else if (value->p.p && value->len != ANYT_LENGTH_INDEF)
		return stio_ubjson_write_chunk_array(sio, value);
	else if (pctx_type) { *pctx_type=ANY_T_ARRAY; return STIO_R_CTX_BEGIN; }
	else return STIO_E_NESTING;
}

static inline
int stio_ubjson_write_open_map(StioStream* sio, const Any* value,
	AnyBaseType* pctx_type)
{
	if (value->len != ANYT_LENGTH_INDEF) {
		TRYR( stream_char_put(sio->s, '#') );
		UBJSON_UINT_PUT(value->len);
	}

	if (value->len == 0) return STIO_R_OK;
	else if (value->p.p && value->len != ANYT_LENGTH_INDEF)
		return stio_ubjson_write_chunk_map(sio, value);
	else if (pctx_type) { *pctx_type=ANY_T_MAP; return STIO_R_CTX_BEGIN; }
	else return STIO_E_NESTING;
}

static inline
int stio_ubjson_write_open_vector(StioStream* sio, const Any* value,
	AnyBaseType* pctx_type)
{
	if (value->len == ANYT_LENGTH_INDEF) return STIO_E_VALUE;

	AnyBaseType ctx_type = value->t;
	TRYR( stream_char_put(sio->s, '$') );
	switch (value->t) {
		//ANY_T_CHARP = ANY_T_STRING, unreachable
	case ANY_T_CHARP:		TRYR( stream_char_put(sio->s, 'C') );	break;
	case ANY_T_UINT8P:		TRYR( stream_char_put(sio->s, 'U') );	break;
	case ANY_T_INT8P:		TRYR( stream_char_put(sio->s, 'i') );	break;
	case ANY_T_INT16P:		TRYR( stream_char_put(sio->s, 'I') );	break;
	case ANY_T_UINT16P:		ctx_type = ANY_T_INT32P; //passthrough
	case ANY_T_INT32P:		TRYR( stream_char_put(sio->s, 'l') );	break;
	case ANY_T_UINT32P:		ctx_type = ANY_T_INT64P; //passthrough
	case ANY_T_INT64P:		TRYR( stream_char_put(sio->s, 'L') );	break;
	case ANY_T_FLOAT32P:	TRYR( stream_char_put(sio->s, 'd') );	break;
	case ANY_T_FLOAT64P:	TRYR( stream_char_put(sio->s, 'D') );	break;
	default:
		return STIO_E_VALUE;
	}

	TRYR( stream_char_put(sio->s, '#') );
	UBJSON_UINT_PUT(value->len);

	if (value->len == 0) return STIO_R_OK;
	else if (value->p.p)
		return stio_ubjson_write_chunk_vector(sio, value, ctx_type);
	else if (pctx_type) { *pctx_type=ctx_type; return STIO_R_CTX_BEGIN; }
	else return STIO_E_NESTING;
}

static inline
int stio_ubjson_write_value(StioStream* sio, const Any* value,
	AnyBaseType* pctx_type)
{
	DebugLog("stio_ubjson_write_value %d", value->t);
	TRYR( stream_write_prep(sio->s, 9) );

	switch (value->t) {
	case ANY_T_NULL:
		TRYR( stream_char_put(sio->s, 'Z') );
		return STIO_R_OK;

	case ANY_T_BOOL:
		TRYR( stream_char_put(sio->s,  value->p.b ? 'T' : 'F' ) );
		return STIO_R_OK;

	case ANY_T_CHAR:
		TRYR( stream_char_put(sio->s, 'C') );
		TRYR( stream_char_put(sio->s, value->p.c) );
		return STIO_R_OK;

	case ANY_T_UINT8:
		TRYR( stream_char_put(sio->s, 'U') );
		TRYR( stream_char_put(sio->s, value->p.u8) );
		return STIO_R_OK;

	case ANY_T_UINT16:	UBJSON_UINT_PUT(value->p.u16);	return STIO_R_OK;
	case ANY_T_UINT32:	UBJSON_UINT_PUT(value->p.u32);	return STIO_R_OK;
	case ANY_T_UINT64:	UBJSON_UINT_PUT(value->p.u64);	return STIO_R_OK;
	case ANY_T_INT8:	UBJSON_INT_PUT(value->p.i8);	return STIO_R_OK;
	case ANY_T_INT16:	UBJSON_INT_PUT(value->p.i16);	return STIO_R_OK;
	case ANY_T_INT32:	UBJSON_INT_PUT(value->p.i32);	return STIO_R_OK;
	case ANY_T_INT64:	UBJSON_INT_PUT(value->p.i64);	return STIO_R_OK;

	case ANY_T_FLOAT32:
		TRYR( stream_char_put(sio->s, 'd') );
		TRYR( stio_stream_write_be32(sio, &value->p.f32) );
		return STIO_R_OK;

	case ANY_T_FLOAT64: {
		float f32 = value->p.f64;
		if (f32 == value->p.f64) {
			TRYR( stream_char_put(sio->s, 'd') );
			TRYR( stio_stream_write_be32(sio, &f32) );
		} else {
			TRYR( stream_char_put(sio->s, 'D') );
			TRYR( stio_stream_write_be64(sio, &value->p.f64) );
		}
		} return STIO_R_OK;

	case ANY_T_STRING:
		TRYR( stream_char_put(sio->s, 'S') );
		return stio_ubjson_write_open_string(sio, value, pctx_type);

	case ANY_T_ARRAY:
		TRYR( stream_char_put(sio->s, '[') );
		return stio_ubjson_write_open_array(sio, value, pctx_type);

	case ANY_T_MAP:
		TRYR( stream_char_put(sio->s, '{') );
		return stio_ubjson_write_open_map(sio, value, pctx_type);

	//case ANY_T_CHARP = ANY_T_STRING
	case ANY_T_UINT8P:
	case ANY_T_INT8P:
	case ANY_T_INT16P:
	case ANY_T_UINT16P:
	case ANY_T_INT32P:
	case ANY_T_UINT32P:
	case ANY_T_INT64P:
	case ANY_T_FLOAT32P:
	case ANY_T_FLOAT64P:
		TRYR( stream_char_put(sio->s, '[') );
		return stio_ubjson_write_open_vector(sio, value, pctx_type);

	default:
		return STIO_E_VALUE;
	}
}

static inline
int stio_ubjson_write_value_cast(StioStream* sio,
	AnyBaseType type, const Any* value)
{
	if (!anyb_scalar_is(value->t)) return STIO_E_VALUE;

	const size_t sz = anyb_size(type);
	TRYR( stream_write_prep(sio->s, sz) );

	if (type == value->t)
	{
		TRYR( stio_stream_write_be(sio, sz, &value->p) );
	}
	else
	{
		AnyPayload p;
		anyp_cast(type, &p, value->t, &value->p);
		TRYR( stio_stream_write_be(sio, sz, &p) );
	}

	return STIO_R_OK;
}

static inline
int stio_ubjson_write_ctx_begin(StioStream* sio, StioItem* itm,
	AnyBaseType ctx_type)
{
	DebugLog("stio_ubjson_write_ctx_begin %d", ctx_type);
	itm->npend = itm->value.len;
	itm->value = (Any){ .t=ctx_type };
	return STIO_R_CTX_BEGIN;
}

int stio_ubjson_write(StioStream* sio, StioCtx* ctx, StioItem* itm)
{
	int r;
	DebugLog("stio_ubjson_write %d", itm->type);
	switch (itm->type) {
	case STIO_T_NULL: //no-op
		TRYR( stream_write_prep(sio->s, 1) );
		TRYR( stream_char_put(sio->s, 'N') );
		return STIO_R_OK;

	case STIO_T_VALUE:
		if (ctx && anyb_pointer_is(ctx->vtype))
		{
			TRYR( r = stio_ubjson_write_value_cast(sio,
				anyb_pointer_deref(ctx->vtype), &itm->value) );

			if (ctx->npend != STIO_LENGTH_INDEF) {
				if (ctx->npend < 1) return STIO_E_OVERFLOW;
				ctx->npend--;
			}
			return r;
		}
		else
		{
			AnyBaseType ctx_type = 0;
			TRYR( r = stio_ubjson_write_value(sio, &itm->value, &ctx_type) );
			if (r == STIO_R_CTX_BEGIN)
				stio_ubjson_write_ctx_begin(sio, itm, ctx_type);

			if (ctx) {
				if (ctx->npend != STIO_LENGTH_INDEF) {
					if (ctx->npend < 1) return STIO_E_OVERFLOW;
					ctx->npend--;
				}
				ctx->sflags &= ~STIO_IF_KEY_DONE;
			}
			return r;
		}

	case STIO_T_KEY: {
		if (!ctx) return STIO_E_CONTEXT;
		if (ctx->vtype != ANY_T_MAP) return STIO_E_CONTEXT;
		if (ctx->sflags & STIO_IF_KEY_DONE) return STIO_E_CONTEXT;
		if (itm->value.t != ANY_T_STRING) return STIO_E_VALUE;

		TRYR( stream_write_prep(sio->s, 9) );

		AnyBaseType ctx_type = 0;
		TRYR( r = stio_ubjson_write_open_string(sio, &itm->value, &ctx_type) );
		if (r == STIO_R_CTX_BEGIN)
			stio_ubjson_write_ctx_begin(sio, itm, ctx_type);


		ctx->sflags |= STIO_IF_KEY_DONE;
		return r;
		}

	case STIO_T_CHUNK: {
		if (!ctx) return STIO_E_CONTEXT;
		if (anyb_scalar_is(itm->value.t)) return STIO_E_VALUE;
		if (!itm->value.p.p) return STIO_E_VALUE;
		if (itm->value.len == ANYT_LENGTH_INDEF) return STIO_E_VALUE;
		if (itm->value.len > ctx->npend) return STIO_E_OVERFLOW;

		int r;
		if (itm->value.t == ANY_T_MAP) {
			if (ctx->vtype != ANY_T_MAP) return STIO_E_CONTEXT;
			if (ctx->sflags & STIO_IF_KEY_DONE) return STIO_E_CONTEXT;
			r = stio_ubjson_write_chunk_map(sio, &itm->value);
		}
		else if (itm->value.t == ANY_T_ARRAY) {
			if (ctx->vtype != ANY_T_ARRAY) return STIO_E_CONTEXT;
			r = stio_ubjson_write_chunk_array(sio, &itm->value);
		}
		else if (itm->value.t == ANY_T_STRING) {
			if (ctx->vtype != ANY_T_STRING) return STIO_E_CONTEXT;
			r = stio_ubjson_write_chunk_string(sio, &itm->value);
		}
		else if (anyb_pointer_is(itm->value.t)) {
			r = stio_ubjson_write_chunk_vector(sio, &itm->value, ctx->vtype);
		}
		else
			return STIO_E_VALUE;

		if (r < 0) return r;
		if (ctx->npend != STIO_LENGTH_INDEF) {
			assert(ctx->npend >= itm->value.len);
			ctx->npend -= itm->value.len;
		}
		return r;
		}

	case STIO_T_END:
		if (!ctx) return STIO_E_CONTEXT;
		if (ctx->npend == STIO_LENGTH_INDEF) {
			TRYR( stream_write_prep(sio->s, 1) );
			if (ctx->vtype == ANY_T_ARRAY)	TRYR( stream_char_put(sio->s, ']') );
			else if (ctx->vtype == ANY_T_MAP)	TRYR( stream_char_put(sio->s, '}') );
			else return STIO_E_CONTEXT;
			ctx->npend = 0;
		}
		else if (ctx->npend) return STIO_E_UNDERFLOW;
		return STIO_R_CTX_END;

	default:
		return STIO_E_TYPE;
	}
}

/*
	UBJSON Decoder
*/
static inline
int stio_ubjson_read_value(StioStream* sio, StioCtx* ctx,
	Any* value, StioItem* itm);

static inline
int stio_ubjson_read_chunk_string(StioStream* sio, Any* value, uint32_t* plen)
{
	DebugLog("stio_ubjson_read_chunk_string");
	if (value->t == ANY_T_VOIDP)
		value->t = ANY_T_STRING;

	if (value->t != ANY_T_STRING && value->t != ANY_T_UINT8P)
		return STIO_E_VALUE;

	if (value->len > *plen)
		value->len = *plen;

	if (value->p.p) {
		size_t rr = stream_read(sio->s, value->len, value->p.p);
		if (rr < value->len) value->len = rr;
	} else {
		TRYR( stream_seek(sio->s, value->len, SEEK_CUR) );
	}

	assert(*plen != STIO_LENGTH_INDEF);
	*plen -= value->len;
	return *plen ? STIO_R_OK : STIO_R_CTX_END;
}

static inline
int stio_ubjson_read_chunk_vector(StioStream* sio, Any* value,
	uint32_t* plen, AnyBaseType type)
{
	DebugLog("stio_ubjson_read_chunk_vector");
	assert( anyb_pointer_is(type) );

	AnyBaseType stype = anyb_pointer_deref(type);
	const size_t sstep = anyb_size(stype);

	//TODO: repeated in stio_read_chunk_vector
	if (value->t == ANY_T_VOIDP) {
		value->t = type;
		value->len /= sstep;
	}

	if (!anyb_pointer_is(value->t))
		return STIO_E_VALUE;

	if (value->len > *plen)
		value->len = *plen;
	
	bool swap = !big_endian_is() && ~sio->cflags & STIO_SF_NO_BYTE_SWAP;
	TRYR( stio_read_chunk_vector(sio, value, type, *plen, swap) );

	assert(*plen != STIO_LENGTH_INDEF);
	*plen -= value->len;

	return *plen ? STIO_R_OK : STIO_R_CTX_END;
}

static inline
int stio_ubjson_read_chunk_array(StioStream* sio, StioCtx* ctx, Any* value)
{
	DebugLog("stio_ubjson_read_chunk_array");
	if (value->t == ANY_T_VOIDP) {
		value->t = ANY_T_ARRAY;
		value->len /= sizeof(Any);
	}

	uint32_t i=0, e=value->len;
	if (e > ctx->npend) e = ctx->npend;

	if (value->t == ANY_T_ARRAY)
	{
		Any v={0};
		Any* ac;
		unsigned step;
		if (value->p.ap) { ac = value->p.ap; step = 1; }
		else { ac = &v; step = 0; }

		for (; i<e; ++i)
		{
			int r = stio_ubjson_read_value(sio, ctx, ac, NULL);
			if (r == STIO_E_NESTING) break;
			if (r < 0) return r;
			if (r == STIO_R_CTX_END) { ctx->npend=0; break; }
			ac+=step;
		}
	}
	else if (anyb_pointer_is(value->t))
	{
		AnyBaseType vtype = anyb_pointer_deref(value->t);
		const size_t vstep = anyb_size(vtype);
		unsigned char* ac = value->p.p;
		for (; i<e; ++i)
		{
			Any v;
			int r = stio_ubjson_read_value(sio, ctx, &v, NULL);
			if (r == STIO_E_NESTING) break;
			if (r < 0) return r;
			if (r == STIO_R_CTX_END) { ctx->npend=0; break; }
			if (ac && vstep) {
				anyp_cast(vtype, ac, v.t, &v.p);
				ac += vstep;
			}
		}
	}
	else
		return STIO_E_VALUE;

	value->len = i;
	if (ctx->npend) {
		if (ctx->npend != STIO_LENGTH_INDEF)
			ctx->npend -= i;
	}
	return (ctx->npend) ? STIO_R_OK : STIO_R_CTX_END;
}

static inline
int stio_ubjson_read_chunk_map(StioStream* sio, StioCtx* ctx, Any* value)
{
	DebugLog("stio_ubjson_read_chunk_map");
	return STIO_E_VALUE;
}

static inline
int stio_ubjson_read_ptype(StioStream* sio, AnyBaseType* ptype)
{
	int c;
	TRYR( c = stream_char_get(sio->s) );
	switch (c) {
	case 'C':  *ptype = ANY_T_CHARP;  return STIO_R_OK;
	case 'i':  *ptype = ANY_T_INT8P;  return STIO_R_OK;
	case 'U':  *ptype = ANY_T_UINT8P;  return STIO_R_OK;
	case 'I':  *ptype = ANY_T_INT16P;  return STIO_R_OK;
	case 'l':  *ptype = ANY_T_INT32P;  return STIO_R_OK;
	case 'L':  *ptype = ANY_T_INT64P;  return STIO_R_OK;
	case 'd':  *ptype = ANY_T_FLOAT32P;  return STIO_R_OK;
	case 'D':  *ptype = ANY_T_FLOAT64P;  return STIO_R_OK;
	}
	return STIO_E_DATA;
}

static inline
int stio_ubjson_read_length(StioStream* sio, uint32_t* len)
{
	int c;
	TRYR( c = stream_char_get(sio->s) );
	switch (c) {
	case 'i': {
		int8_t i8;
		TRYR( stream_read_var(sio->s, i8) );
		*len = i8;
		} break;
	case 'U': {
		uint8_t u8;
		TRYR( stream_read_var(sio->s, u8) );
		*len = u8;
		} break;
	case 'I': {
		int16_t i16;
		TRYR( stio_stream_read_var_be(sio, i16) );
		*len = i16;
		} break;
	case 'l': {
		int32_t i32;
		TRYR( stio_stream_read_var_be(sio, i32) );
		*len = i32;
		} break;
	case 'L': {
		int64_t i64;
		TRYR( stio_stream_read_var_be(sio, i64) );
		*len = i64;
		} break;
	default:
		return STIO_E_DATA;
	}
	return STIO_R_OK;
}

static inline
int stio_ubjson_read_open_finish(StioStream* sio, StioItem* itm,
	AnyBaseType type, uint32_t len)
{
	itm->npend = len;

	if (itm->npend == 0) {
		itm->value = (Any){ .t=type };
		return STIO_R_OK;
	}
	else if (itm->value.p.p && itm->value.len > 0)
	{
		if (itm->value.t == ANY_T_VOIDP) itm->value.t = type;
		if (itm->value.t != type) goto ctx_new;

		int r;
		if (type == ANY_T_STRING) {
			r = stio_ubjson_read_chunk_string(sio, &itm->value, &itm->npend);
		}
		else if (type == ANY_T_ARRAY) {
			if (itm->value.t != ANY_T_ARRAY) goto ctx_new;
			StioCtx ctx={ .vtype=itm->value.t, .npend=itm->npend };
			r = stio_ubjson_read_chunk_array(sio, &ctx, &itm->value);
			itm->npend = ctx.npend;
		}
		else if (type == ANY_T_MAP) {
			if (itm->value.t != ANY_T_MAP) goto ctx_new;
			StioCtx ctx={ .vtype=itm->value.t, .npend=itm->npend };
			r = stio_ubjson_read_chunk_map(sio, &ctx, &itm->value);
			itm->npend = ctx.npend;
		}
		else { //vector
			r = stio_ubjson_read_chunk_vector(sio, &itm->value, &itm->npend,
				type);
		}

		if (r == STIO_E_VALUE) goto ctx_new;
		if (r < 0) return r;
		if (itm->npend == 0) return STIO_R_OK;

		return STIO_R_CTX_BEGIN;
	}

ctx_new:
	itm->value = (Any){ .t=type };
	return STIO_R_CTX_BEGIN;
}

static inline
int stio_ubjson_read_open_string(StioStream* sio, StioItem* itm)
{
	uint32_t len;
	TRYR( stio_ubjson_read_length(sio, &len) );
	DebugLog("stio_ubjson_read_open_string %d", (int)len);
	return stio_ubjson_read_open_finish(sio, itm, ANY_T_STRING, len);
}

static inline
int stio_ubjson_read_open_array(StioStream* sio, StioItem* itm)
{
	AnyBaseType type=ANY_T_ARRAY;
	uint32_t len=STIO_LENGTH_INDEF;

	int c;
	TRYR( c = stream_char_get(sio->s) );
	if (c == '$') {
		TRYR( stio_ubjson_read_ptype(sio, &type) );
		TRYR( c = stream_char_get(sio->s) );
		if (c != '#') return STIO_E_DATA;
	}
	if (c == '#') {
		TRYR( stio_ubjson_read_length(sio, &len) );
	}
	else {
		stream_unget(sio->s, 1);
	}

	DebugLog("stio_ubjson_read_open_array %d %d", type, (int)len);
	return stio_ubjson_read_open_finish(sio, itm, type, len);
}

static inline
int stio_ubjson_read_open_map(StioStream* sio, StioItem* itm)
{
	AnyBaseType type=ANY_T_MAP;
	uint32_t len=STIO_LENGTH_INDEF;

	int c;
	TRYR( c = stream_char_get(sio->s) );
	if (c == '$') return STIO_E_NOT_IMPL;
	if (c == '#') {
		TRYR( stio_ubjson_read_length(sio, &len) );
	}
	else {
		stream_unget(sio->s, 1);
	}

	DebugLog("stio_ubjson_read_open_map %d", (int)len);
	return stio_ubjson_read_open_finish(sio, itm, type, len);
}

static inline
int stio_ubjson_read_value(StioStream* sio, StioCtx* ctx,
	Any* value, StioItem* itm)
{
	int c;
	AnyPayload p;

	do {
		TRYR( c = stream_char_get(sio->s) );
	} while (c == 'N');

	DebugLog("stio_ubjson_read_value %02x", c);
	switch (c) {
	case 'Z':  if (value) *value = any_null();  return STIO_R_OK;
	case 'T':  if (value) *value = any_bool(true);  return STIO_R_OK;
	case 'F':  if (value) *value = any_bool(false);  return STIO_R_OK;

	case 'C':
		TRYR( stream_read_var(sio->s, p.c) );
		if (value) *value = any_char(p.c);
		return STIO_R_OK;
	case 'i':
		TRYR( stream_read_var(sio->s, p.i8) );
		if (value) *value = any_int8(p.i8);
		return STIO_R_OK;
	case 'U':
		TRYR( stream_read_var(sio->s, p.u8) );
		if (value) *value = any_uint8(p.u8);
		return STIO_R_OK;
	case 'I':
		TRYR( stio_stream_read_var_be(sio, p.i16) );
		if (value) *value = any_int16(p.i16);
		return STIO_R_OK;
	case 'l':
		TRYR( stio_stream_read_var_be(sio, p.i32) );
		if (value) *value = any_int32(p.i32);
		return STIO_R_OK;
	case 'L':
		TRYR( stio_stream_read_var_be(sio, p.i64) );
		if (value) *value = any_int64(p.i64);
		return STIO_R_OK;

	case 'd':
		TRYR( stio_stream_read_var_be(sio, p.f32) );
		if (value) *value = any_float32(p.f32);
		return STIO_R_OK;
	case 'D':
		TRYR( stio_stream_read_var_be(sio, p.f64) );
		if (value) *value = any_float64(p.f64);
		return STIO_R_OK;

	case 'H':
		return STIO_E_NOT_IMPL;

	case ']':
		if (!ctx || ctx->vtype != ANY_T_ARRAY ||
				ctx->npend != STIO_LENGTH_INDEF)
			return STIO_E_DATA;
		return STIO_R_CTX_END;
	case '}':
		if (!ctx || ctx->vtype != ANY_T_MAP ||
				ctx->npend != STIO_LENGTH_INDEF)
			return STIO_E_DATA;
		return STIO_R_CTX_END;

	case 'S': //UTF-8 string
		if (!itm) { stream_unget(sio->s, 1); return STIO_E_NESTING; }
		return stio_ubjson_read_open_string(sio, itm);

	case '[': //array
		if (!itm) { stream_unget(sio->s, 1); return STIO_E_NESTING; }
		return stio_ubjson_read_open_array(sio, itm);

	case '{': //map
		if (!itm) { stream_unget(sio->s, 1); return STIO_E_NESTING; }
		return stio_ubjson_read_open_map(sio, itm);
	}

	return STIO_E_DATA;
}

static inline
int stio_ubjson_read_value_cast(StioStream* sio,
	Any* value, AnyBaseType type)
{
	const size_t sz = anyb_size(type);
	TRYR( stio_stream_read_be(sio, sz, &value->p) );
	value->t = type;
	value->len = 0;
	value->cls = 0;
	return STIO_R_OK;
}

static inline
int stio_ubjson_read_end(StioStream* sio, StioCtx* ctx, StioItem* itm)
{
	DebugLog("stio_ubjson_read_end");
	ctx->npend = 0;
	itm->type = STIO_T_END;
	return STIO_R_CTX_END;
}

int stio_ubjson_read(StioStream* sio, StioCtx* ctx, StioItem* itm)
{
	if (!ctx)
	{
		if (itm->type == STIO_T_CHUNK) return STIO_E_TYPE;

		TRYR( stream_read_prep(sio->s, 1) );
		itm->type = STIO_T_VALUE;
		return stio_ubjson_read_value(sio, NULL, &itm->value, itm);
	}
	else if (ctx && ctx->npend == 0)
	{
		return stio_ubjson_read_end(sio, ctx, itm);
	}

	TRYR( stream_read_prep(sio->s, 1) );
	if (ctx->vtype == ANY_T_ARRAY)
	{
		if (itm->type == STIO_T_CHUNK)
		{
			return stio_ubjson_read_chunk_array(sio, ctx, &itm->value);
		}
		else
		{
			itm->type = STIO_T_VALUE;
			int r = stio_ubjson_read_value(sio, ctx, &itm->value, itm);
			if (r < 0) return r;
			if (r == STIO_R_CTX_END)
				return stio_ubjson_read_end(sio, ctx, itm);
			if (ctx->npend != STIO_LENGTH_INDEF)
				ctx->npend--;
			return r;
		}
	}
	else if (ctx->vtype == ANY_T_MAP)
	{
		int c;
		TRYR( c = stream_char_get(sio->s) );
		if (c == '}')
			return stio_ubjson_read_end(sio, ctx, itm);
		else
			stream_unget(sio->s, 1);

		if (itm->type == STIO_T_CHUNK)
		{
			return stio_ubjson_read_chunk_map(sio, ctx, &itm->value);
		}
		else if (~ctx->sflags & STIO_IF_KEY_DONE)
		{
			itm->type = STIO_T_KEY;
			int r = stio_ubjson_read_open_string(sio, itm);
			if (r < 0) return r;
			ctx->sflags |= STIO_IF_KEY_DONE;
			return r;
		}
		else
		{
			itm->type = STIO_T_VALUE;
			int r = stio_ubjson_read_value(sio, ctx, &itm->value, itm);
			if (r < 0) return r;
			ctx->sflags &= ~STIO_IF_KEY_DONE;
			assert(r != STIO_R_CTX_END);
			if (ctx->npend != STIO_LENGTH_INDEF)
				ctx->npend--;
			return r;
		}
	}
	else if (ctx->vtype == ANY_T_STRING || ctx->vtype == ANY_T_UINT8P)
	{
		assert(ctx->npend != STIO_LENGTH_INDEF);
		if (itm->type == STIO_T_CHUNK)
		{
			return stio_ubjson_read_chunk_string(sio, &itm->value, &ctx->npend);
		}
		else
		{
			itm->type = STIO_T_VALUE;
			int r = stio_ubjson_read_value_cast(sio, &itm->value, ANY_T_CHAR);
			if (r < 0) return r;
			ctx->npend--;
			return r;
		}
	}
	else if (anyb_pointer_is(ctx->vtype))
	{
		assert(ctx->npend != STIO_LENGTH_INDEF);
		if (itm->type == STIO_T_CHUNK)
		{
			return stio_ubjson_read_chunk_vector(sio, &itm->value, &ctx->npend,
				ctx->vtype);
		}
		else
		{
			itm->type = STIO_T_VALUE;
			AnyBaseType t = anyb_pointer_deref(ctx->vtype);
			int r = stio_ubjson_read_value_cast(sio, &itm->value, t);
			if (r < 0) return r;
			ctx->npend--;
			return r;
		}
	}
	return STIO_E_CONTEXT;
}


const StioClass stio_class_ubjson = {
	stio_ubjson_read,
	stio_ubjson_write,
	"ubjson"
};
