/* Copyright 2024, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: Zlib
 */
#define STRUCTIO_IMPLEMENTATION
#include "structio_cbor.h"

/*
	CBOR encoder
*/
static inline
int stio_cbor_write_value(StioStream* sio, StioItem* itm, const Any* value);

static inline
int stio_cbor_write_chunk_string(StioStream* sio,
	AnyBaseType ctx_type, const Any* value)
{
	assert( anyb_size( anyb_pointer_deref(ctx_type) ) == 1 );
	assert( anyb_size( anyb_pointer_deref(value->t) ) == 1 );
	TRYR( stream_write_chk(sio->s, value->len, value->p.p) );
	return STIO_R_OK;
}

static inline
int stio_cbor_write_chunk_vector(StioStream* sio,
	AnyBaseType ctx_type, const Any* value)
{
	assert( sio->cflags & STIO_SF_CBOR_TYPED_ARRAYS );
	assert( anyb_pointer_is(ctx_type) );
	TRYR( stio_write_chunk_vector(sio, value, ctx_type, false) );
	return STIO_R_OK;
}

static inline
int stio_cbor_write_chunk_array(StioStream* sio,
	AnyBaseType ctx_type, const Any* value)
{
	assert( ctx_type == ANY_T_ARRAY || anyb_pointer_is(ctx_type) );

	if (value->t == ANY_T_ARRAY)
	{
		const Any* ac = value->p.ap;
		for (uint32_t i=0, e=value->len; i<e; ++i)
		{
			TRYR( stio_cbor_write_value(sio, NULL, ac) );
			ac++;
		}
	}
	else if (anyb_pointer_is(value->t))
	{
		AnyBaseType vtype = anyb_pointer_deref(value->t);
		const size_t vstep = anyb_size(vtype);

		Any v = { .t=vtype };
		const unsigned char* ac = value->p.p;
		for (uint32_t i=0, e=value->len; i<e; ++i)
		{
			memcpy(&v.p, ac, vstep);
			ac += vstep;
			TRYR( stio_cbor_write_value(sio, NULL, &v) );
		}
	}
	else
		return STIO_E_VALUE;

	return STIO_R_OK;
}

static inline
int stio_cbor_write_chunk_map(StioStream* sio,
	AnyBaseType ctx_type, const Any* value)
{
	assert( ctx_type == ANY_T_MAP );
	assert( value->t == ANY_T_MAP );

	const Any* ac = value->p.ap;
	for (uint32_t i=0, e=value->len; i<e; ++i)
	{
		TRYR( stio_cbor_write_value(sio, NULL, ac) );
		ac++;
		TRYR( stio_cbor_write_value(sio, NULL, ac) );
		ac++;
	}

	return STIO_R_OK;
}

static inline
int stio_cbor_write_chunk(StioStream* sio, StioCtx* ctx, const Any* value)
{
	DebugLog("stio_cbor_write_chunk %d", value->t);
	if (ctx->npend < value->len) return STIO_E_OVERFLOW;
	if (!value->p.p) return STIO_E_VALUE;

	int r = STIO_E_VALUE;
	if (value->t == ANY_T_STRING || value->t == ANY_T_UINT8P)
	{
		if (ctx->vtype != ANY_T_STRING || ctx->vtype != ANY_T_UINT8P)
			return STIO_E_CONTEXT;
		r = stio_cbor_write_chunk_string(sio, ctx->vtype, value);
	}
	else if (value->t == ANY_T_ARRAY)
	{
		if (ctx->vtype != ANY_T_ARRAY) return STIO_E_CONTEXT;
		r = stio_cbor_write_chunk_array(sio, ctx->vtype, value);
	}
	else if (value->t == ANY_T_MAP)
	{
		if (ctx->vtype != ANY_T_MAP) return STIO_E_CONTEXT;
		if (ctx->sflags & STIO_IF_KEY_DONE) return STIO_E_CONTEXT;
		r = stio_cbor_write_chunk_map(sio, ctx->vtype, value);
	}
	else if (anyb_pointer_is(value->t))
	{
		if (ctx->sflags & STIO_IF_DIRECT_ACCESS)
		{
			r = stio_cbor_write_chunk_vector(sio, ctx->vtype, value);
		}
		else
		{
			if (ctx->vtype != ANY_T_ARRAY && !anyb_pointer_is(ctx->vtype))
				return STIO_E_CONTEXT;
			r = stio_cbor_write_chunk_array(sio, ctx->vtype, value);
		}
	}
	else
		return STIO_E_VALUE;

	if (r < 0) return r;

	if (ctx->npend != STIO_LENGTH_INDEF) {
		assert(ctx->npend >= value->len);
		ctx->npend -= value->len;
	}
	return r;
}

static inline
int stio_cbor_write_uint(StioStream* sio, uint8_t mt, uint64_t n)
{
	uint8_t *end, *cur=stream_buffer_get(sio->s, (void**)&end);
	assert(end-cur >= 9);

	if (n <= 0x17) {
		*cur++ = mt | n;
	}
	else if (n <= 0xff) {
		*cur++ = mt | 0x18;
		*cur++ = n;
	}
	else if (n <= 0xffff) {
		*cur++ = mt | 0x19;
		stio_copy_be(sio, 2, cur, &n);
		cur += 2;
	}
	else if (n <= 0xffffffff) {
		*cur++ = mt | 0x1a;
		stio_copy_be(sio, 4, cur, &n);
		cur += 4;
	}
	else {
		*cur++ = mt | 0x1a;
		stio_copy_be(sio, 8, cur, &n);
		cur += 8;
	}

	stream_commit(sio->s, cur);
	return STIO_R_OK;
}

static inline
int stio_cbor_write_sint(StioStream* sio, int64_t n)
{
	if (n < 0)	return stio_cbor_write_uint(sio, 0x20, -n-1);
	else		return stio_cbor_write_uint(sio, 0, n);
}

static inline
int stio_cbor_write_tag(StioStream* sio, uint64_t n)
{
	//TODO: process and filter ?
	return stio_cbor_write_uint(sio, 0xc0, n);
}

static inline
int stio_cbor_write_ctx_begin(StioStream* sio, StioItem* itm)
{
	DebugLog("stio_cbor_write_ctx_begin %d", itm->value.t);
	itm->npend = itm->value.len;
	itm->value = (Any){ .t=itm->value.t };
	return STIO_R_CTX_BEGIN;
}

static inline
int stio_cbor_write_open(StioStream* sio, StioItem* itm, const Any* value,
	uint8_t mt)
{
	unsigned cbor_len = value->len;
	unsigned itm_flags = 0;
	if (sio->cflags & STIO_SF_CBOR_TYPED_ARRAYS &&
		value->len != ANYT_LENGTH_INDEF) //TODO
	{
		unsigned tag=0, len1=0;
		bool le = little_endian_is();
		switch (value->t) {
		case ANY_T_UINT16P:  tag=65+le*4; len1=2; break;
		case ANY_T_UINT32P:  tag=66+le*4; len1=4; break;
		case ANY_T_UINT64P:  tag=67+le*4; len1=8; break;

		case ANY_T_INT8P:    tag=72;      len1=1; break;
		case ANY_T_INT16P:   tag=73+le*4; len1=2; break;
		case ANY_T_INT32P:   tag=74+le*4; len1=4; break;
		case ANY_T_INT64P:   tag=75+le*4; len1=8; break;

		case ANY_T_FLOAT32P: tag=81+le*4; len1=4; break;
		case ANY_T_FLOAT64P: tag=82+le*4; len1=8; break;
		}
		if (tag) {
			TRYR( stio_cbor_write_tag(sio, tag) );
			itm_flags |= STIO_IF_DIRECT_ACCESS;
			mt = 2<<5; //byte string
			cbor_len *= len1;
		}
	}

	if (value->len == ANYT_LENGTH_INDEF)
	{
		if (!itm) return STIO_E_NESTING;
		assert( value == &itm->value );

		TRYR( stream_char_put(sio->s, mt | 0x1f) );
		sio->itm_sflags |= itm_flags;
		return stio_cbor_write_ctx_begin(sio, itm);
	}
	else if (value->len == 0)
	{
		TRYR( stream_char_put(sio->s, mt) );
		return STIO_R_OK;
	}
	else if (value->p.p) {
		stio_cbor_write_uint(sio, mt, cbor_len);

		switch (value->t) {
		case ANY_T_UINT8P:
		case ANY_T_STRING:
			return stio_cbor_write_chunk_string(sio, value->t, value);
		case ANY_T_ARRAY:
			return stio_cbor_write_chunk_array(sio, value->t, value);
		case ANY_T_MAP:
			return stio_cbor_write_chunk_map(sio, value->t, value);
		default:
			if (itm_flags & STIO_IF_DIRECT_ACCESS)
				return stio_cbor_write_chunk_vector(sio, value->t, value);
			else
				return stio_cbor_write_chunk_array(sio, value->t, value);
		}
	}
	else {
		if (!itm) return STIO_E_NESTING;
		assert( value == &itm->value );

		stio_cbor_write_uint(sio, mt, cbor_len);
		sio->itm_sflags |= itm_flags;
		return stio_cbor_write_ctx_begin(sio, itm);
	}
}

static inline
int stio_cbor_write_value(StioStream* sio, StioItem* itm, const Any* value)
{
	DebugLog("stio_cbor_write_value %d", value->t);

	TRYR( stream_write_prep(sio->s, 18) );

	switch (value->t) {
	case ANY_T_NULL:
		TRYR( stream_char_put(sio->s, 0xf6) );
		return STIO_R_OK;

	case ANY_T_BOOL:
		TRYR( stream_char_put(sio->s,  value->p.b ? 0xf5 : 0xf4 ) );
		return STIO_R_OK;

	case ANY_T_CHAR:
	case ANY_T_UINT8:	return stio_cbor_write_uint(sio, 0, value->p.u8);
	case ANY_T_UINT16:	return stio_cbor_write_uint(sio, 0, value->p.u16);
	case ANY_T_UINT32:	return stio_cbor_write_uint(sio, 0, value->p.u32);
	case ANY_T_UINT64:	return stio_cbor_write_uint(sio, 0, value->p.u64);

	case ANY_T_INT8:	return stio_cbor_write_sint(sio, value->p.i8);
	case ANY_T_INT16:	return stio_cbor_write_sint(sio, value->p.i16);
	case ANY_T_INT32:	return stio_cbor_write_sint(sio, value->p.i32);
	case ANY_T_INT64:	return stio_cbor_write_sint(sio, value->p.i64);

	case ANY_T_FLOAT32:
		TRYR( stream_char_put(sio->s, 0xfa) );
		TRYR( stio_stream_write_be32(sio, &value->p.f32) );
		return STIO_R_OK;

	case ANY_T_FLOAT64: {
		float f32 = value->p.f64;
		if (f32 == value->p.f64) {
			TRYR( stream_char_put(sio->s, 0xfa) );
			TRYR( stio_stream_write_be32(sio, &f32) );
		} else {
			TRYR( stream_char_put(sio->s, 0xfb) );
			TRYR( stio_stream_write_be64(sio, &value->p.f64) );
		}
		} return STIO_R_OK;

	case ANY_T_UINT8P:	return stio_cbor_write_open(sio, itm, value, 0x40);
	case ANY_T_STRING:	return stio_cbor_write_open(sio, itm, value, 0x60);
	case ANY_T_ARRAY:	return stio_cbor_write_open(sio, itm, value, 0x80);
	case ANY_T_MAP:		return stio_cbor_write_open(sio, itm, value, 0xa0);
	default:
		if (anyb_pointer_is(value->t))
			return stio_cbor_write_open(sio, itm, value, 0x80);
	}

	return STIO_E_VALUE;
}

int stio_cbor_write(StioStream* sio, StioCtx* ctx, StioItem* itm)
{
	int r=0;

	DebugLog("stio_cbor_write %d", itm->type);
	switch (itm->type) {
	case STIO_T_NULL:
		return STIO_R_OK;

	case STIO_T_TAG: {
		uint64_t n;
		if (!anyp_cast(ANY_T_UINT64, &n, itm->value.t, &itm->value.p))
			return STIO_E_VALUE;

		TRYR( stream_write_prep(sio->s, 9) );
		return stio_cbor_write_tag(sio, n);
		}

	case STIO_T_KEY:
		if (!ctx) return STIO_E_CONTEXT;
		if (ctx->vtype != ANY_T_MAP) return STIO_E_CONTEXT;
		if (ctx->sflags & STIO_IF_KEY_DONE) return STIO_E_CONTEXT;
		if (!ctx->npend) return STIO_E_OVERFLOW;

		TRYR( r = stio_cbor_write_value(sio, itm, &itm->value) );

		ctx->sflags |= STIO_IF_KEY_DONE;
		break;

	case STIO_T_VALUE:
		if (ctx) {
			if (!ctx->npend) return STIO_E_OVERFLOW;
			if (ctx->vtype == ANY_T_MAP && ~ctx->sflags & STIO_IF_KEY_DONE)
				return STIO_E_CONTEXT;
		}

		if (ctx && ctx->sflags & STIO_IF_DIRECT_ACCESS) {
			if (!anyb_scalar_is(itm->value.t)) return STIO_E_VALUE;

			Any v = any_pointer_get(&itm->value);
			TRYR( r = stio_cbor_write_chunk(sio, ctx, &v) );
		}
		else {
			TRYR( r = stio_cbor_write_value(sio, itm, &itm->value) );
			if (ctx) {
				if (ctx->vtype == ANY_T_MAP) ctx->sflags &= ~STIO_IF_KEY_DONE;
				if (ctx->npend != STIO_LENGTH_INDEF) ctx->npend--;
			}
		}
		break;

	case STIO_T_CHUNK:
		if (!ctx) return STIO_E_CONTEXT;
		return stio_cbor_write_chunk(sio, ctx, &itm->value);

	case STIO_T_END:
		if (!ctx) return STIO_E_CONTEXT;
		if (ctx->npend == STIO_LENGTH_INDEF) {
			TRYR( stream_write_prep(sio->s, 1) );
			TRYR( stream_char_put(sio->s, 0xff) );
		}
		else if (ctx->npend) return STIO_E_UNDERFLOW;
		return STIO_R_CTX_END;

	default:
		return STIO_E_TYPE;
	}

	return r;
}

/*
	CBOR decoder
*/
static inline
int stio_cbor_read_value(StioStream* sio, StioItem* itm, Any* value);

static inline
int stio_cbor_read_chunk_vector(StioStream* sio, StioCtx* ctx, Any* value)
{
	TRYR( stio_read_chunk_vector(sio, value, ctx->vtype, ctx->npend,
		ctx->sflags & STIO_IF_DIRECT_BYTE_SWAP) );

	assert(ctx->npend != STIO_LENGTH_INDEF);
	ctx->npend -= value->len;

	return ctx->npend ? STIO_R_OK : STIO_R_CTX_END;
}

static inline
int stio_cbor_read_chunk_string(StioStream* sio, Any* value,
	AnyBaseType ctx_type, uint32_t* plen)
{
	DebugLog("stio_cbor_read_chunk_string");
	assert(*plen != STIO_LENGTH_INDEF);

	if (value->t == ANY_T_VOIDP)
		value->t = ANY_T_STRING;

	if (value->t != ANY_T_STRING && value->t != ANY_T_UINT8P)
		return STIO_E_VALUE;

	if (value->len > *plen)
		value->len = *plen;

	if (value->p.p) {
		size_t rr = stream_read(sio->s, value->len, value->p.p);
		if (rr < value->len) value->len = rr;
	} else {
		TRYR( stream_seek(sio->s, value->len, SEEK_CUR) );
	}

	assert(*plen != STIO_LENGTH_INDEF);
	*plen -= value->len;
	return *plen ? STIO_R_OK : STIO_R_CTX_END;
}

static inline
int stio_cbor_read_chunk_array(StioStream* sio, Any* value,
	AnyBaseType ctx_type, uint32_t* plen)
{
	DebugLog("stio_cbor_read_chunk_array");
	if (value->t == ANY_T_VOIDP) {
		value->t = ANY_T_ARRAY;
		value->len /= sizeof(Any);
	}

	uint32_t i=0, e=value->len;
	if (e > *plen) e = *plen;

	if (value->t == ANY_T_ARRAY)
	{
		Any v={0};
		Any* ac;
		unsigned step;
		if (value->p.ap) { ac = value->p.ap; step = 1; }
		else { ac = &v; step = 0; }

		for (; i<e; ++i)
		{
			int r = stio_cbor_read_value(sio, NULL, ac);
			if (r < 0) return r;
			if (r == STIO_R_CTX_END) { *plen=0; break; }
			ac += step;
		}
	}
	else if (anyb_pointer_is(value->t))
	{
		AnyBaseType vtype = anyb_pointer_deref(value->t);
		const size_t vstep = anyb_size(vtype);

		Any v={0};
		unsigned char * ac = value->p.p;
		for (; i<e; ++i)
		{
			int r = stio_cbor_read_value(sio, NULL, &v);
			if (r < 0) return r;
			if (r == STIO_R_CTX_END) { *plen=0; break; }
			if (ac && vstep) {
				anyp_cast(vtype, ac, v.t, &v.p);
				ac += vstep;
			}
		}
	}
	else return STIO_E_VALUE;

	value->len = i;
	if (*plen) {
		if (*plen != STIO_LENGTH_INDEF) *plen -= i;
	}
	return *plen ? STIO_R_OK : STIO_R_CTX_END;
}

static inline
int stio_cbor_read_chunk_map(StioStream* sio, Any* value,
	AnyBaseType ctx_type, uint32_t* plen)
{
	DebugLog("stio_cbor_read_chunk_map");
	if (value->t == ANY_T_VOIDP) {
		value->t = ANY_T_MAP;
		value->len /= sizeof(AnyPair);
	}
	if (value->t != ANY_T_MAP) return STIO_E_VALUE;

	uint32_t i=0, e=value->len;
	if (e > *plen) e = *plen;

	Any v={0};
	Any* ac;
	unsigned step;
	if (value->p.ap) { ac = value->p.ap; step = 1; }
	else { ac = &v; step = 0; }

	for (; i<e; ++i)
	{
		int r = stio_cbor_read_value(sio, NULL, ac);
		if (r < 0) return r;
		if (r == STIO_R_CTX_END) return STIO_E_DATA;
		ac += step;

		r = stio_cbor_read_value(sio, NULL, ac);
		if (r < 0) return r;
		if (r == STIO_R_CTX_END) { *plen=0; break; }
		ac += step;
	}

	value->len = i;
	if (*plen) {
		if (*plen != STIO_LENGTH_INDEF) *plen -= i;
	}
	return *plen ? STIO_R_OK : STIO_R_CTX_END;
}

static inline
int stio_cbor_read_pint(StioStream* sio, int c, Any* value)
{
	switch (c) {
	default: //0...23
		*value = any_uint8(c);
		break;
	case 24:
		TRYR( stream_read_var(sio->s, value->p.u8) );
		*value = any_uint8(value->p.u8);
		break;
	case 25:
		TRYR( stio_stream_read_var_be(sio, value->p.u16) );
		*value = any_uint16(value->p.u16);
		break;
	case 26:
		TRYR( stio_stream_read_var_be(sio, value->p.u32) );
		*value = any_uint32(value->p.u32);
		break;
	case 27:
		TRYR( stio_stream_read_var_be(sio, value->p.u64) );
		*value = any_uint64(value->p.u64);
		break;
	case 28:  return STIO_E_DATA;
	case 29:  return STIO_E_DATA;
	case 30:  return STIO_E_DATA;
	case 31:  return STIO_E_DATA;  //indef
	}
	return STIO_R_OK;
}

static inline
int stio_cbor_read_nint(StioStream* sio, int c, Any* value)
{
	switch (c) {
	default: //0...23
		*value = any_int8(-1-c);
		break;
	case 24: {
		uint8_t u8;
		TRYR( stream_read_var(sio->s, u8) );
		if (u8 < 0x7f) *value = any_int8(-1-u8);
		else *value = any_int16(-1-u8);
		} break;
	case 25: {
		uint16_t u16;
		TRYR( stio_stream_read_var_be(sio, u16) );
		if (u16 < 0x7fff) *value = any_int16(-1-u16);
		else *value = any_int32(-1-u16);
		} break;
	case 26: {
		uint32_t u32;
		TRYR( stio_stream_read_var_be(sio, u32) );
		if (u32 < 0x7fffffff) *value = any_int32(-1-u32);
		else *value = any_int64((int64_t)-1-u32);
		} break;
	case 27: {
		uint64_t u64;
		TRYR( stio_stream_read_var_be(sio, u64) );
		if (u64 < 0x7fffffffffffffff) *value = any_int64(-1-u64);
		else return STIO_E_NOT_IMPL;
		} break;
	case 28:  return STIO_E_DATA;
	case 29:  return STIO_E_DATA;
	case 30:  return STIO_E_DATA;
	case 31:  return STIO_E_DATA;  //indef
	}
	return STIO_R_OK;
}

static inline
int stio_cbor_read_length(StioStream* sio, int c, uint32_t* plen)
{
	assert(0 <= c && c <= 255);
	switch (c) {
	default: *plen = c; break;  //0...23
	case 24: {
		uint8_t u8;
		TRYR( stream_read_var(sio->s, u8) );
		*plen = u8;
		} break;
	case 25: {
		uint16_t u16;
		TRYR( stio_stream_read_var_be(sio, u16) );
		*plen = u16;
		} break;
	case 26: {
		uint16_t u32;
		TRYR( stio_stream_read_var_be(sio, u32) );
		*plen = u32;
		} break;
	case 27: {
		uint64_t u64;
		TRYR( stio_stream_read_var_be(sio, u64) );
		if (u64 > 0xffffffff) return STIO_E_NOT_IMPL;
		*plen = u64; }
		break;
	case 28:  return STIO_E_DATA;
	case 29:  return STIO_E_DATA;
	case 30:  return STIO_E_DATA;
	case 31: *plen = STIO_LENGTH_INDEF; break;
	}
	return STIO_R_OK;
}

#define CBOR_READ_LENGTH(_PLEN_,_C_) do {\
	TRYR( stio_cbor_read_length(sio, (_C_) & 0x1f, (_PLEN_)) ); \
} while(0)

static inline
int stio_cbor_read_open(StioStream* sio, StioItem* itm, AnyBaseType vtype,
	uint32_t len)
{
	DebugLog("stio_cbor_read_open %d, %lu", vtype, (unsigned long)len);

	if (!itm) {
		stream_unget(sio->s, 1);
		return STIO_E_NESTING;
	}

	itm->npend = len;
	if (len == 0) {
		itm->value = (Any){ .t=vtype };
		return STIO_R_OK;
	}
	else if (itm->value.p.p && itm->value.len > 0)
	{
		if (itm->value.t == ANY_T_VOIDP) itm->value.t = vtype;
		if (itm->value.t != vtype) goto ctx_new;

		int r;
		switch (vtype) {
		case ANY_T_STRING:
		case ANY_T_UINT8P:
		case ANY_T_INT8P:
			if (len == STIO_LENGTH_INDEF) goto ctx_new;
			r = stio_cbor_read_chunk_string(sio, &itm->value, vtype, &len);
			break;
		case ANY_T_ARRAY:
			r = stio_cbor_read_chunk_array(sio, &itm->value, vtype, &len);
			break;
		case ANY_T_MAP:
			r = stio_cbor_read_chunk_map(sio, &itm->value, vtype, &len);
			break;
		default: {
			if (len == STIO_LENGTH_INDEF) return STIO_E_NOT_IMPL;
			StioCtx ctx={ .vtype=itm->value.t, .npend=itm->npend };
			r = stio_cbor_read_chunk_vector(sio, &ctx, &itm->value);
			itm->npend = ctx.npend;
			}
		}

		if (r == STIO_E_VALUE) goto ctx_new;
		if (r < 0) return r;

		itm->npend = len;
		if (len)
		{
			return STIO_R_CTX_BEGIN;
		}
		else
		{
			return STIO_R_OK;
		}
	}

ctx_new:
	itm->value = (Any){ .t=vtype };
	return STIO_R_CTX_BEGIN;
}

static inline
int stio_cbor_read_typedarray(StioStream* sio, StioItem* itm, Any* value,
	AnyBaseType type, bool swap)
{
	int c;

	// Should be a byte string
	TRYR( c = stream_char_get(sio->s) );
	if ((c >> 5) != 2) return STIO_E_DATA;

	uint32_t len;
	CBOR_READ_LENGTH(&len, c);

	unsigned len1 = anyb_size( anyb_pointer_deref(type) );
	if (len % len1) {
		// Byte string length not multiple of vector unit length
		return STIO_E_VALUE;
	}
	len /= len1;

	if (!itm) return STIO_E_NESTING;
	sio->itm_sflags |= (swap) ? STIO_IF_DIRECT_BYTE_SWAP : STIO_IF_DIRECT_ACCESS;

	return stio_cbor_read_open(sio, itm, type, len);
}

static inline
int stio_cbor_read_tag(StioStream* sio, StioItem* itm, Any* value, int c)
{
	TRYR( stio_cbor_read_pint(sio, c, value) );

#define STIO_CBOR_READ_TYPEDARRAY(T,E) \
		stio_cbor_read_typedarray(sio, itm, value, ANY_T_##T, !E##_endian_is())

	uint64_t tag = anys_uint64_get(value);
	DebugLog("stio_cbor_read_tag %lu", (unsigned long)tag);
	switch (tag) {
	//case 40: // array of two arrays* | Multi-dimensional Array, row-major order
	//case 41: // array | Homogeneous Array
	case 64: // byte string | uint8 Typed Array
		return STIO_CBOR_READ_TYPEDARRAY(UINT8P, big);
	case 65: // byte string | uint16, big endian, Typed Array
		return STIO_CBOR_READ_TYPEDARRAY(UINT16P, big);
	case 66: // byte string | uint32, big endian, Typed Array
		return STIO_CBOR_READ_TYPEDARRAY(UINT32P, big);
	case 67: // byte string | uint64, big endian, Typed Array
		return STIO_CBOR_READ_TYPEDARRAY(UINT64P, big);
	case 68: // byte string | uint8 Typed Array, clamped  arithmetic
		return STIO_CBOR_READ_TYPEDARRAY(UINT8P, little);
	case 69: // byte string | uint16, little endian, Typed Array
		return STIO_CBOR_READ_TYPEDARRAY(UINT16P, little);
	case 70: // byte string | uint32, little endian, Typed Array
		return STIO_CBOR_READ_TYPEDARRAY(UINT32P, little);
	case 71: // byte string | uint64, little endian, Typed Array
		return STIO_CBOR_READ_TYPEDARRAY(UINT64P, little);
	case 72: // byte string | sint8 Typed Array
		return STIO_CBOR_READ_TYPEDARRAY(INT8P, big);
	case 73: // byte string | sint16, big endian, Typed Array
		return STIO_CBOR_READ_TYPEDARRAY(INT16P, big);
	case 74: // byte string | sint32, big endian, Typed Array
		return STIO_CBOR_READ_TYPEDARRAY(INT32P, big);
	case 75: // byte string | sint64, big endian, Typed Array
		return STIO_CBOR_READ_TYPEDARRAY(INT64P, big);
	//case 76: // byte string | (reserved)
	case 77: // byte string | sint16, little endian, Typed Array
		return STIO_CBOR_READ_TYPEDARRAY(INT16P, little);
	case 78: // byte string | sint32, little endian, Typed Array
		return STIO_CBOR_READ_TYPEDARRAY(INT32P, little);
	case 79: // byte string | sint64, little endian, Typed Array
		return STIO_CBOR_READ_TYPEDARRAY(INT64P, little);
	//case 80: // byte string | IEEE 754 binary16, big endian, Typed Array
	case 81: // byte string | IEEE 754 binary32, big endian, Typed Array
		return STIO_CBOR_READ_TYPEDARRAY(FLOAT32P, big);
	case 82: // byte string | IEEE 754 binary64, big endian, Typed Array
		return STIO_CBOR_READ_TYPEDARRAY(FLOAT64P, big);
	//case 83: // byte string | IEEE 754 binary128, big endian, Typed Array
	//case 84: // byte string | IEEE 754 binary16, little endian, Typed Array
	case 85: // byte string | IEEE 754 binary32, little endian, Typed Array
		return STIO_CBOR_READ_TYPEDARRAY(FLOAT32P, little);
	case 86: // byte string | IEEE 754 binary64, little endian, Typed Array
		return STIO_CBOR_READ_TYPEDARRAY(FLOAT64P, little);
	//case 87: // byte string | IEEE 754 binary128, little endian, Typed Array
	//case 1040 | array of two arrays* | Multi-dimensional Array, column-major order
	}

	//TODO: process

	if (itm) {
		itm->type = STIO_T_TAG;
	}
	else {
		stream_unget(sio->s, 1);
		return STIO_E_NESTING;
	}
	return STIO_R_OK;
}

static inline
int stio_cbor_read_value(StioStream* sio, StioItem* itm, Any* value)
{
	int c;
	TRYR( c = stream_char_get(sio->s) );

	DebugLog("stio_cbor_read_value %02x", c);
	switch (c) {
	case 0xf4:  *value = any_bool(false);  return STIO_R_OK;
	case 0xf5:  *value = any_bool(true);  return STIO_R_OK;
	case 0xf6:  *value = any_null();  return STIO_R_OK;

	case 0xf9: //float16
		return STIO_E_NOT_IMPL;

	case 0xfa:
		TRYR( stio_stream_read_var_be(sio, value->p.f32) );
		*value = any_float32(value->p.f32);
		return STIO_R_OK;

	case 0xfb:
		TRYR( stio_stream_read_var_be(sio, value->p.f64) );
		*value = any_float64(value->p.f64);
		return STIO_R_OK;

	case 0xfc: case 0xfd: case 0xfe:
		return STIO_E_DATA;

	case 0xff:
		return STIO_R_CTX_END;

	default:
		switch (c >> 5) {
		case 0:  //postive integer
			return stio_cbor_read_pint(sio, c, value);

		case 1:  //negative integer
			return stio_cbor_read_nint(sio, c & 0x1f, value);

		case 2: { //byte string
			uint32_t len;
			CBOR_READ_LENGTH(&len, c);
			return stio_cbor_read_open(sio, itm, ANY_T_UINT8P, len);
			}

		case 3: { //UTF-8 string
			uint32_t len;
			CBOR_READ_LENGTH(&len, c);
			return stio_cbor_read_open(sio, itm, ANY_T_STRING, len);
			}

		case 4: { //array
			uint32_t len;
			CBOR_READ_LENGTH(&len, c);
			return stio_cbor_read_open(sio, itm, ANY_T_ARRAY, len);
			}

		case 5: { //map
			uint32_t len;
			CBOR_READ_LENGTH(&len, c);
			return stio_cbor_read_open(sio, itm, ANY_T_MAP, len);
			}

		case 6: //tag
			return stio_cbor_read_tag(sio, itm, value, c & 0x1f);

		default: //7: svalue
			return STIO_E_NOT_IMPL;
		}
	}

	return STIO_E_DATA;
}

int stio_cbor_read(StioStream* sio, StioCtx* ctx, StioItem* itm)
{
	if (ctx && !ctx->npend) {
		itm->type = STIO_T_END;
		return STIO_R_CTX_END;
	}
	
	int r = stream_read_prep(sio->s, 1);
	if (r == STREAM_E_EOF) return STIO_E_EOF;
	if (r < 0) return r;
	
	if (!ctx) {
		itm->type = STIO_T_VALUE;
		return stio_cbor_read_value(sio, itm, &itm->value);
	}

	if (ctx->vtype == ANY_T_STRING || ctx->vtype == ANY_T_UINT8P)
	{
		if (itm->type != STIO_T_CHUNK) return STIO_E_TYPE;

		if (ctx->npend == STIO_LENGTH_INDEF)
		{
			int r = stio_cbor_read_value(sio, itm, &itm->value);
			if (r == STIO_R_CTX_END) {
				itm->type = STIO_T_END;
				ctx->npend = 0;
			}
			return r;
		}
		else
		{
			return stio_cbor_read_chunk_string(sio, &itm->value,
					ctx->vtype, &ctx->npend);
		}
	}
	else if (ctx->vtype == ANY_T_ARRAY)
	{
		if (itm->type == STIO_T_CHUNK)
		{
			return stio_cbor_read_chunk_array(sio, &itm->value,
					ctx->vtype, &ctx->npend);
		}
		else
		{
			itm->type = STIO_T_VALUE;
			int r = stio_cbor_read_value(sio, itm, &itm->value);
			if (r < 0) return r;
			if (r == STIO_R_CTX_END) {
				itm->type = STIO_T_END;
				ctx->npend = 0;
			}
			else if (ctx->npend != STIO_LENGTH_INDEF) ctx->npend--;
			return r;
		}
	}
	else if (ctx->vtype == ANY_T_MAP) {
		if (ctx->sflags & STIO_IF_KEY_DONE)
		{
			if (itm->type == STIO_T_CHUNK) return STIO_E_TYPE;

			itm->type = STIO_T_VALUE;
			int r = stio_cbor_read_value(sio, itm, &itm->value);
			if (r < 0) return r;
			if (r == STIO_R_CTX_END) {
				itm->type = STIO_T_END;
				ctx->npend = 0;
			}
			else ctx->sflags &= ~STIO_IF_KEY_DONE;
			return r;
		}
		else
		{
			if (itm->type == STIO_T_CHUNK)
			{
				return stio_cbor_read_chunk_map(sio, &itm->value,
					ctx->vtype, &ctx->npend);
			}
			else
			{
				itm->type = STIO_T_KEY;
				int r = stio_cbor_read_value(sio, itm, &itm->value);
				if (r < 0) return r;
				if (r == STIO_R_CTX_END) {
					itm->type = STIO_T_END;
					ctx->npend = 0;
				}
				else {
					if (ctx->npend != STIO_LENGTH_INDEF) ctx->npend--;
					ctx->sflags |= STIO_IF_KEY_DONE;
				}
				return r;
			}
		}
	}
	else if (anyb_pointer_is(ctx->vtype))
	{
		if (itm->type == STIO_T_CHUNK)
		{
			return stio_cbor_read_chunk_vector(sio, ctx, &itm->value);
		}
		else
		{
			AnyPayload p;
			Any v = { .t=ctx->vtype, .len=1, .p={ .p=&p } };
			int r = stio_cbor_read_chunk_vector(sio, ctx, &v);
			if (r < 0) return r;

			if (v.len) {
				itm->type = STIO_T_VALUE;
				any_pointer_index(&itm->value, &v, 0);
			}
			return r;
		}
	}

	return STIO_E_CONTEXT;
}

const StioClass stio_class_cbor = {
	stio_cbor_read,
	stio_cbor_write,
	"cbor"
};
