/* Copyright 2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: Zlib
 *
 * Interface to store and manipulate sounds in memory.
 * Also, some synthesis features.
 *
 * Example:
Sound snd={0};
int freq=44100;
snd_resize(&snd, freq*5, 1, freq, SND_FMT_F32);
snd_zero(&snd);
snd_wave_add(&snd, SND_WT_SINE, 440, 0.8, 0);
snd_adsr(&snd, 0.5,0.5,1, 1,0.7,0.7, 0);
// Play the sound
snd_free(&snd);
 */
#pragma once
#include <stdint.h>
#include <stdbool.h>

typedef enum SndFormat {
	SND_FMT_NONE,
	SND_FMT_F32,  //float
	SND_FMT_S16,
	SND_FMT_U8,
} SndFormat;
#define SND_FMT__COUNT  (SND_FMT_U8+1)

typedef struct SndFormatAttr {
	unsigned size;
	const char * name;
	int sdl2;  //AUDIO_* value in SDL2
	int silence;  //Silence value (for U8)
} SndFormatAttr;

extern const SndFormatAttr g_snd_format_attrs[SND_FMT__COUNT];

static inline
const SndFormatAttr* snd_format_attr(SndFormat fmt) {
	if (!(0 <= fmt && fmt < SND_FMT__COUNT)) fmt = SND_FMT_NONE;  //size=0
	return &g_snd_format_attrs[fmt];
}

enum SndFlags {
	SND_F_OWN_MEM 	= 1,
};

typedef struct Sound {
	uint8_t *    data;
	unsigned     len;     //Number of samples in data
	unsigned     ch;      //Channels
	int          ss;      //Sample stride in bytes, samples may be smaller
	unsigned     freq;    //Sample rate (e.g. 44100)
	                      //Duration in seconds: len/freq
	SndFormat    format;
	int          flags;
} Sound;

void snd_free(Sound* S);

static inline
bool snd_empty(const Sound* S) {
	return !S || !S->len;
}

void snd_resize(Sound* S, unsigned len, unsigned ch, unsigned freq, SndFormat fmt);

//Set all samples to zero.
void snd_zero(Sound* S);

//Resizes dst and copies src data to it.
void snd_copy(Sound* dst, const Sound* src);

//Convert from src->fmt to fmt.
//Stores the results in dst.
//Can be done in place (dst = src).
void snd_format_convert(Sound* dst, const Sound* src, SndFormat fmt);

//Sets dst such that it presents some of the data of src.
//Examples:
//	snd_view_make(dss, src, 0,src->len,1, 0,1,1);
//	View the first channel.
//	snd_view_make(dss, src, 1000,2000,1, 0,src->ch,1);
//	View samples 1000 to 2999, all channels.
void snd_view_make(Sound* dst, const Sound* src,
	int si, int se, int ss,		//Samples start, stop, step
	int ci, int ce, int cs);	//Channels statr, stop, step

static inline
Sound snd_slice_get(const Sound* S, int start, int stop) {
	Sound dst={0};
	snd_view_make(&dst, S, start, stop-start, 1, 0, S->ch, 1);
	return dst;
}

static inline
Sound snd_channel_get(const Sound* S, int idx) {
	Sound dst={0};
	snd_view_make(&dst, S, 0, S->len, 1, idx, 1, 1);
	return dst;
}

static inline
Sound snd_view_get(const Sound* S) {
	Sound dst = *S;
	dst.flags &= ~SND_F_OWN_MEM;
	return dst;
}

//Returns true is two sounds have the same ch, freq and format.
static inline
bool snd_specs_equal(const Sound* A, const Sound* B) {
	return A->ch == B->ch && A->freq == B->freq && A->format == B->format;
}

static inline
bool snd_contiguous_is(const Sound* snd) {
	const SndFormatAttr * fa = snd_format_attr(snd->format);
	return (snd->ss == (int)fa->size);
}

static inline
unsigned snd_bytesize(const Sound* snd) {
	const SndFormatAttr * fa = snd_format_attr(snd->format);
	return snd->len * snd->ch * fa->size;
}

//Returns a pointer to sample I, channel C.
#define SND_INDEX(SND,I,C) \
	((SND).data + ((I) * (SND).ch + (C)) * (SND).ss)

/* Sound synthesis */

typedef enum SndWaveType {
	SND_WT_NONE,
	SND_WT_SINE,
	SND_WT_SQUARE,
	SND_WT_TRIANGLE,
	SND_WT_SAWTOOTH,
	SND_WT_SINE_PM_SINE,  //sin(wt + sin(wt))
} SndWaveType;

//Add a periodic wave to the sound.
void snd_wave_add(Sound* S, SndWaveType type, double freq, double amp,
	double phase);

typedef enum SndNoiseType {
	SND_NT_NONE,
	SND_NT_WHITE,  //uniform spectra
	SND_NT_PINK,   //1/f spectra
} SndNoiseType;

//Add noise.
void snd_noise_add(Sound* S, SndNoiseType type, double amp);

typedef enum SndCurveType {
	SND_CT_LINEAR,
	SND_CT_EXP,
} SndCurveType;

//Modulate with a continuous amplitude variation from m1 to m2.
//sigma: used by EXP, try 5.
void snd_fade(Sound* S, SndCurveType type, double m1, double m2, double sigma);

//Modulate the sound using an ADSR envelope.
//Times in seconds.
void snd_adsr(Sound* S, double t_attack, double t_decay, double t_sustain,
	double m_attack, double m_decay, double m_sustain, SndCurveType ctype);

//Add two sounds.
//[dst] = [dst] * dmul + [src] * smul
//Does nothing if the specs or the lengths differ.
void snd_add(Sound* dst, const Sound* src, double dmul, double smul);

/* Utility */

#define SND_WAV_HEADER_SIZE  44

// Writes a .wav file header to buffer.
// If the data is contiguous, you can write it using:
// 	 write(file, header, sizeof(header));
// 	 write(file, snd->data, snd_bytesize(snd));
void snd_wav_header_fill(const Sound* S, uint8_t header[SND_WAV_HEADER_SIZE]);
