/* Copyright 2025-2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: Zlib
 *
 * Parses text with formatting commands and renders it to an image.
 *
 * All commands start with the escape character RichTextEngine.c.ec .
 * Good choices are '^' and '\x7f' (default).
 *
 * Commands:
 *   {   Begin group.
 *   }   End group.
 *   p   Phantom text: advances the cursor, but does not renders anything.
 *   cI  Change text color to id I.
 *   bI  Change text background color to id I.
 *   fI  Change font to id I.
 *   iI  Draw image with id I on the text base line.
 *   II  Draw image with id I from the line top.
 *   BI  Use the current position to set the bounding box component I.
 *       I: t(top), b(bottom), l(left), r(right).
 *   CI  Move the cursor to I:
 *       bbox: t(top), b(bottom), l(left), r(right)
 *       sbox: T(top), B(bottom), L(left), R(right)
 *   OI  Sets an option. Lowercase I enables, uppercase, disables. Options:
 *         w  Text wrapping.
 * Pending implementation:
 *   uI  Underline. I='1': enable, I=' ': disable.
 *   sI  Striketrough / cross out. I='1': enable, I=' ': disable.
 *   hI  Horizontal alignment. I can be l(eft), r(right), c(enter) or ' ' (default).
 *   vI  Vertical alignment. I can be t(op), b(ottom), c(enter) or ' ' (default).
 *   rNN Repeat the current group NN times.
 *
 * All ids I are a single unicode codepoint.
 *
 * Also the following ANSI escape sequences are implemented:
 *   \n  New line: move the cursor to initial position of the next line.
 *   \r  Carriage return: move the cursor the initial position of this line.
 *   \t  Horizontal tab: move the cursor to the next tabulation position.
 *   \v  Vertical tab: similar as above, but a vertical motion.
 *   \b  Backspace: moves the cursor to the previous position.
 *
 * Example texts using '^' as escape:
 * "Default^{^crRed^}Default"
 * "Default^crRed^cBBlack"
 * "Col1 L1\nCol1 Line2\nCol1 L3^CR ^Bl^CtCol2 L1\nCol2 Line2"
 */
#pragma once
#include "textrender.h"
#include "strslice.h"
#include "image.h"

typedef struct RichTextEngine {
	// Resources
	struct RichTextFont {
		unsigned id;  // Unicode character
		TextRender * font;  //not-owned
	} * fonts;  //vector, sorted

	struct RichTextColor {
		unsigned id;  // Unicode character
		ImgColor color;
	} * colors;  //vector, sorted

	struct RichTextImage {
		unsigned id;  // Unicode character
		const Image * image;  //not-owned
	} * images;  //vector, sorted

	// Configuration
	struct RichTextConfig {
		// Escape character: unicode character that starts a command.
		// If zero, '\x7f' is used.
		// If found twice in sequence, one instance is passed through.
		unsigned ec;
		
		// Defaults.
		// Set to the first set resource of each class.
		TextRender * def_font;
		ImgColor def_color;

		int flags;
	} c;
} RichTextEngine;

enum RichTextEngineConfigFlags {
	// Check textrender.h regarding the following flags
	RTE_CF_FAST_BLIT	= 0x0001,
	RTE_CF_NO_UTF8		= 0x0002,
	RTE_CF_NO_KERNING	= 0x0004,
};

void richtext_free(RichTextEngine* S);

/* Set the default colors:
 * B(lack), G(ray), w(hite), r(ed), g(reen), b(lue), y(ellow)
 */
void richtext_default_colors_set(RichTextEngine* S);

void richtext_font_set(RichTextEngine* S, unsigned id, TextRender* font);
TextRender* richtext_font_get(RichTextEngine* S, unsigned id);

void richtext_color_set(RichTextEngine* S, unsigned id, const ImgColor color);
ImgColor richtext_color_get(RichTextEngine* S, unsigned id);

void richtext_image_set(RichTextEngine* S, unsigned id, const Image* image);
const Image* richtext_image_get(RichTextEngine* S, unsigned id);

int richtext_size_get(RichTextEngine* S, const StrSlice text, ImgPoint* psz);

int richtext_render(RichTextEngine* S, Image* img, ImgRect* rect,
	const StrSlice text);

int richtext_render_m(RichTextEngine* S, Image* img, ImgRect* rect,
	unsigned n_text, const StrSlice* texts);
