/* Copyright 2021-2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: Zlib
 */
#include "image_io_tiff.h"
#include "logging.h"
#include "ccommon.h"
#include <string.h>

//TODO: implement load (at least partially)

// Types (from libtiff)
typedef enum {
	TIFF_SHORT	= 3,       // 16-bit unsigned integer
	TIFF_LONG	= 4,       // 32-bit unsigned integer
	TIFF_LONG8	= 16,      // BigTIFF 64-bit unsigned integer
} TIFFDataType;

// Tags (from libtiff)
#define	TIFFTAG_SUBFILETYPE		254	/* subfile data descriptor */
#define	    FILETYPE_REDUCEDIMAGE	0x1	/* reduced resolution version */
#define	    FILETYPE_PAGE		0x2	/* one page of many */
#define	    FILETYPE_MASK		0x4	/* transparency mask */
#define	TIFFTAG_IMAGEWIDTH		256	/* image width in pixels */
#define	TIFFTAG_IMAGELENGTH		257	/* image height in pixels */
#define	TIFFTAG_BITSPERSAMPLE		258	/* bits per channel (sample) */
#define	TIFFTAG_COMPRESSION		259	/* data compression technique */
#define	    COMPRESSION_NONE		1	/* dump mode */
#define	    COMPRESSION_DEFLATE		32946	/* Deflate compression */
#define	TIFFTAG_PHOTOMETRIC		262	/* photometric interpretation */
#define	    PHOTOMETRIC_MINISWHITE	0	/* min value is white */
#define	    PHOTOMETRIC_MINISBLACK	1	/* min value is black */
#define	    PHOTOMETRIC_RGB		2	/* RGB color model */
#define	TIFFTAG_DOCUMENTNAME		269	/* name of doc. image is from */
#define	TIFFTAG_IMAGEDESCRIPTION	270	/* info about image */
#define	TIFFTAG_STRIPOFFSETS		273	/* offsets to data strips */
#define	TIFFTAG_ORIENTATION		274	/* +image orientation */
#define	    ORIENTATION_TOPLEFT		1	/* row 0 top, col 0 lhs */
#define	TIFFTAG_SAMPLESPERPIXEL		277	/* samples per pixel */
#define	TIFFTAG_ROWSPERSTRIP		278	/* rows per strip of data */
#define	TIFFTAG_STRIPBYTECOUNTS		279	/* bytes counts for strips */
#define	TIFFTAG_PLANARCONFIG		284	/* storage organization */
#define	    PLANARCONFIG_CONTIG		1	/* single image plane */
#define	    PLANARCONFIG_SEPARATE	2	/* separate planes of data */
#define	TIFFTAG_PAGENUMBER		297	/* page numbers of multi-page */
#define	TIFFTAG_SOFTWARE		305	/* name & release */
#define	TIFFTAG_DATETIME		306	/* creation date and time */
#define	TIFFTAG_ARTIST			315	/* creator of image */
#define	TIFFTAG_HOSTCOMPUTER		316	/* machine where created */
#define	TIFFTAG_EXTRASAMPLES		338	/* !info about extra samples */
#define	    EXTRASAMPLE_UNSPECIFIED	0	/* !unspecified data */
#define	    EXTRASAMPLE_ASSOCALPHA	1	/* !associated alpha data */
#define	    EXTRASAMPLE_UNASSALPHA	2	/* !unassociated alpha data */

/* Detect */
bool imgio_tiff_detect(Stream* s, const char* fileext)
{
	if (s) {
		const unsigned char *c = s->cursor;
		if (c[0] == 'I' && c[1] == 'I' &&
			(c[2] == 0x2a || c[2] == 0x2b) && c[3] == 0x00)
			return true;
		if (c[0] == 'M' && c[1] == 'M' &&
			c[2] == 0x00 && (c[3] == 0x2a || c[3] == 0x2b))
			return true;
	}
	else if (fileext) {
		if (!strcmp(fileext, "tiff") || !strcmp(fileext, "tif"))
			return true;
	}
	return false;
}

/* Save */

#define WRITE_BEGIN(BYTES) \
	if (stream_write_prep(imgio->s, (BYTES)) < (BYTES)) \
		RETURN(IMG_ERROR_SAVE); \
	{ \
		uint8_t *end, *cur = stream_buffer_get(imgio->s, &end);

#define WRITE_CUR(T,V) do { \
	*((T*)cur) = (V);  cur += sizeof(T); \
} while (0)

#define WRITE_END() \
		stream_commit(imgio->s, cur); \
	}

#define WRITE_DATA(C,D) do { \
	if (stream_write(imgio->s, (C), (D)) < 0) RETURN(IMG_ERROR_SAVE); \
} while (0)

#define WRITE_VALUE(T,V) \
	WRITE_DATA(sizeof(T), (T[1]){(V)})

int imgio_tiff_save_init(CodecTiffSave* codec, ImageIO* imgio)
{
	ccUNUSED(imgio);
	*codec = (CodecTiffSave){0};

	uint16_t u16 = 0x0001;
	codec->be = ((uint8_t*)&u16)[1];

#ifdef IMGIO_TIFF_DEFAULT_BIGTIFF
	codec->big = true;
#endif

	return 0;
}

int imgio_tiff_save_init_big(CodecTiffSave* codec, ImageIO* imgio)
{
	int r = imgio_tiff_save_init(codec, imgio);
	if (!r) codec->big = true;
	return r;
}

int imgio_tiff_save_header(CodecTiffSave* codec, ImageIO* imgio)
{
	int R=0;

	WRITE_BEGIN(16)
	WRITE_CUR(uint16_t, codec->be ? 0x4d4d : 0x4949);
	WRITE_CUR(uint16_t, codec->big ? 43 : 42);
	if (codec->big) {
		WRITE_CUR(uint16_t, 8);   // Offset's size in bytes
		WRITE_CUR(uint16_t, 0);
		WRITE_CUR(uint64_t, 16);  // Offset first IFD
		codec->o_oifd = stream_pos_get(imgio->s) + 8;
	} else {
		WRITE_CUR(uint32_t, 8);   // Offset first IFD
		codec->o_oifd = stream_pos_get(imgio->s) + 4;
	}
	WRITE_END()

	codec->header = true;
end:
	return R;
}

int imgio_tiff_save_tag(CodecTiffSave* codec, ImageIO* imgio,
						uint16_t tag, uint16_t type, uint32_t count,
						size_t size, void* data)
{
	int R=0;

	if (size > (codec->big ? 8 : 4))
		return IMG_ERROR_UNSUPPORTED_VALUE;

	WRITE_BEGIN(28)
	if (codec->big) {
		WRITE_CUR(uint16_t, tag);
		WRITE_CUR(uint16_t, type);
		WRITE_CUR(uint64_t, count);
		if (size > 0) memcpy(cur, data, size);
		if (size < 8) memset(cur+size, 0, 8-size);
		cur += 8;
	} else {
		WRITE_CUR(uint16_t, tag);
		WRITE_CUR(uint16_t, type);
		WRITE_CUR(uint32_t, count);
		if (size > 0) memcpy(cur, data, size);
		if (size < 4) memset(cur+size, 0, 4-size);
		cur += 4;
	}
	WRITE_END()

	codec->ntag++;
end:
	return R;
}

#define WRITE_TAG2(T,V) \
	TRY( imgio_tiff_save_tag(codec, imgio, (T), TIFF_SHORT, 1, 2, ((uint16_t[1]){(V)})) )

#define WRITE_TAG4(T,V) \
	TRY( imgio_tiff_save_tag(codec, imgio, (T), TIFF_LONG , 1, 4, ((uint32_t[1]){(V)})) )

#define WRITE_TAG8(T,V) \
	TRY( imgio_tiff_save_tag(codec, imgio, (T), TIFF_LONG8, 1, 8, ((uint64_t[1]){(V)})) )

int imgio_tiff_save_op(CodecTiffSave* codec, ImageIO* imgio, Image* img)
{
	int R=0;

	if (!codec->header)
		TRY( imgio_tiff_save_header(codec, imgio) );

	unsigned imgsize = img->w * img->h * img->bypp;

	if (!codec->big && codec->o_oifd + 4+2+12*14+4 + imgsize > 0xFFFFFFFF) {
		log_error("[tiff] File larger than 4 GiB, should use BigTIFF.");
		return IMG_ERROR_SAVE;
	}

	// Number of tags placeholder
	uint64_t o_ntag = stream_pos_get(imgio->s);
	if (codec->big)
		WRITE_VALUE(uint64_t, 0);
	else
		WRITE_VALUE(uint16_t, 0);

	// Must be sorted in ascending order by tag
	//WRITE_TAG4( TIFFTAG_SUBFILETYPE    , FILETYPE_PAGE );
	WRITE_TAG4( TIFFTAG_IMAGEWIDTH     , img->w );
	WRITE_TAG4( TIFFTAG_IMAGELENGTH    , img->h );
	WRITE_TAG2( TIFFTAG_BITSPERSAMPLE  , 8 );
	WRITE_TAG2( TIFFTAG_COMPRESSION    , COMPRESSION_NONE );
	WRITE_TAG2( TIFFTAG_PHOTOMETRIC    ,
		(img->bypp == 1) ? PHOTOMETRIC_MINISBLACK : PHOTOMETRIC_RGB );
	
	// Offset of data placeholder
	if (codec->big)
		WRITE_TAG8( TIFFTAG_STRIPOFFSETS, 0 );
	else
		WRITE_TAG4( TIFFTAG_STRIPOFFSETS, 0 );
	uint64_t o_odata = stream_pos_get(imgio->s) - (codec->big ? 8 : 4);
	
	WRITE_TAG2( TIFFTAG_ORIENTATION    , ORIENTATION_TOPLEFT );
	WRITE_TAG2( TIFFTAG_SAMPLESPERPIXEL, img->bypp );
	WRITE_TAG4( TIFFTAG_ROWSPERSTRIP   , img->h );
	WRITE_TAG4( TIFFTAG_STRIPBYTECOUNTS, imgsize );
	WRITE_TAG2( TIFFTAG_PLANARCONFIG   , PLANARCONFIG_CONTIG );
	//WRITE_TAG2( TIFFTAG_PAGENUMBER     , codec->nframe );
	// TIFFTAG_DATETIME
	if (img->bypp == 4)
		WRITE_TAG2( TIFFTAG_EXTRASAMPLES, EXTRASAMPLE_ASSOCALPHA );

	// Next IFD offset
	codec->o_oifd = stream_pos_get(imgio->s);
	uint64_t o_data = codec->o_oifd + (codec->big ? 8 : 4);
	if (codec->big)
		WRITE_VALUE(uint64_t, o_data+imgsize);
	else
		WRITE_VALUE(uint32_t, o_data+imgsize);

	// Number of tags
	if (stream_seek(imgio->s, o_ntag, 0) < 0) return IMG_ERROR_SAVE;
	if (codec->big)
		WRITE_VALUE(uint64_t, codec->ntag);
	else
		WRITE_VALUE(uint16_t, codec->ntag);

	// Data offset
	if (stream_seek(imgio->s, o_odata, 0) < 0) return IMG_ERROR_SAVE;
	if (codec->big)
		WRITE_VALUE(uint64_t, o_data);
	else
		WRITE_VALUE(uint32_t, o_data);

	// Data
	if (stream_seek(imgio->s, o_data, 0) < 0) return IMG_ERROR_SAVE;
	if (img->w == img->pitch)
		WRITE_DATA( imgsize, img->data );
	else {
		for (unsigned y=0; y<img->h; ++y)
			WRITE_DATA( img->w * img->bypp, img->data + img->pitch * y );
	}

	codec->ntag = 0;
	codec->nframe++;
	codec->nframe_last = codec->nframe;
end:
	return R;
}

int imgio_tiff_save_close(CodecTiffSave* codec, ImageIO* imgio)
{
	int R=0;

	// Correct last IFD offset
	uint64_t pos = stream_pos_get(imgio->s);
	if (stream_seek(imgio->s, codec->o_oifd, 0) < 0) return IMG_ERROR_SAVE;
	if (codec->big)
		WRITE_DATA( 8, (uint64_t[1]){0} );
	else
		WRITE_DATA( 4, (uint32_t[1]){0} );
	if (stream_seek(imgio->s, pos, 0) < 0) return IMG_ERROR_SAVE;

end:
	return R;
}

void imgio_tiff_save_free(CodecTiffSave* codec, ImageIO* imgio)
{
	imgio_tiff_save_close(codec, imgio);
}

/* Values get/set */

int imgio_tiff_value_get(CodecTiff* obj, ImageIO* imgio,
	int id, void* buf, unsigned bufsz)
{
	ccUNUSED(bufsz);

	switch (id) {
/*	case IMG_VALUE_COMPRESSION:
		*((unsigned*)buf) = obj->deflate ? 6 : 0;
		break; */
	case IMG_VALUE_FRAME_IDX:
		*((unsigned*)buf) = obj->nframe;
		break;
	case IMG_VALUE_FRAME_COUNT:
		if (imgio->flags & IMGIO_F_END_FOUND)
			*((unsigned*)buf) = obj->nframe_last;
		else
			return IMG_ERROR_UNKNOWN;
		break;
/*	case IMG_VALUE_FRAME_DURATION:
		*((double*)buf) = ;
		break; */
	default:
		return IMG_ERROR_UNSUPPORTED_VALUE;
	}

	return 0;
}

int imgio_tiff_value_set(CodecTiff* obj, ImageIO* imgio,
	int id, const void* buf, unsigned bufsz)
{
	ccUNUSED(obj);
	ccUNUSED(imgio);
	ccUNUSED(buf);
	ccUNUSED(bufsz);

	switch (id) { 
/*	case IMG_VALUE_COMPRESSION:
		obj->deflate = !!*((unsigned*)buf);
		break; */
/*	case IMG_VALUE_FRAME_DURATION:
		= *((double*)buf);
		break; */
	default:
		return IMG_ERROR_UNSUPPORTED_VALUE;
	}

	return 0;
}

/* Codec */

const ImageCodec img_codec_tiff = {
	imgio_tiff_detect,
	{
		0
		//(int (*)(void*, ImageIO*, Image*)) imgio_tiff_load_op,
		//IMG_CODEC_F_ACCEPT_STREAM | IMG_CODEC_F_ACCEPT_FILENAME,
		//sizeof(CodecTiffLoad),
		//(int (*)(void*, ImageIO*)) imgio_tiff_load_init,
		//(void (*)(void*, ImageIO*)) imgio_tiff_load_free,
		//(int (*)(void*, ImageIO*, long, int)) imgio_tiff_load_seek,
		//(int (*)(void*, ImageIO*, int, void*, unsigned))
		//	imgio_tiff_load_value_get,
	},
	{
		(int (*)(void*, ImageIO*, Image*)) imgio_tiff_save_op,
		IMG_CODEC_F_ACCEPT_STREAM,
		sizeof(CodecTiffSave),
		(int (*)(void*, ImageIO*)) imgio_tiff_save_init,
		(void (*)(void*, ImageIO*)) imgio_tiff_save_free,
		NULL,  //seek
		(int (*)(void*, ImageIO*, int, void*, unsigned))
			imgio_tiff_save_value_get,
		(int (*)(void*, ImageIO*, int, const void*, unsigned))
			imgio_tiff_save_value_set,
	},
	"tiff", "tiff"
};

const ImageCodec img_codec_bigtiff = {
	NULL,
	{ 0 },
	{
		(int (*)(void*, ImageIO*, Image*)) imgio_tiff_save_op,
		IMG_CODEC_F_ACCEPT_STREAM,
		sizeof(CodecTiffSave),
		(int (*)(void*, ImageIO*)) imgio_tiff_save_init_big,
		(void (*)(void*, ImageIO*)) imgio_tiff_save_free,
		NULL,  //seek
		(int (*)(void*, ImageIO*, int, void*, unsigned))
			imgio_tiff_save_value_get,
		(int (*)(void*, ImageIO*, int, const void*, unsigned))
			imgio_tiff_save_value_set,
	},
	"bigtiff", "tiff"
};
