/* Copyright 2021-2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: Zlib
 */
#define IMGIO_LIBTIFF_IMPL
#include "image_io_libtiff.h"

#include <stdlib.h>
#include "logging.h"

//#define DebugLog(...) log_info(__VA_ARGS__)
#define DebugLog(...) {}

bool lib_init_done = false;

/*
	Codec Detect
*/
bool imgio_libtiff_detect(Stream* s, const char* fileext)
{
	if (s) {
		const unsigned char *c = s->cursor;
		if (c[0] == 'I' && c[1] == 'I' &&
			(c[2] == 0x2a || c[2] == 0x2b) && c[3] == 0x00)
			return true;
		if (c[0] == 'M' && c[1] == 'M' &&
			c[2] == 0x00 && (c[3] == 0x2a || c[3] == 0x2b))
			return true;
	}
	else if (fileext) {
		if (!strcmp(fileext, "tiff") || !strcmp(fileext, "tif"))
			return true;
	}
	return false;
}

/*
	Init & free
*/

static void tiff_error_handler(const char* module, const char* fmt,
	va_list args)
{
	char buffer[128];
	int n = snprintf(buffer, sizeof(buffer), "TIFF: %s: ", module);
	vsnprintf(buffer+n, sizeof(buffer)-n, fmt, args);
	log_logf(LOG_LVL_ERROR, "%s", buffer);
}
static void tiff_warning_handler(const char* module, const char* fmt,
	va_list args)
{
	char buffer[128];
	int n = snprintf(buffer, sizeof(buffer), "TIFF: %s: ", module);
	vsnprintf(buffer+n, sizeof(buffer)-n, fmt, args);
	log_logf(LOG_LVL_WARNING, "%s", buffer);
}

static tsize_t tiff_read(thandle_t fd, tdata_t buf, tsize_t size)
{
	return (tsize_t)stream_read((Stream*)fd, size, buf);
}

static tsize_t tiff_write(thandle_t fd, tdata_t buf, tsize_t size)
{
	//DebugLog("tiff_write %d", (int)size);
	return (tsize_t)stream_write((Stream*)fd, size, buf);
}

static toff_t tiff_seek(thandle_t fd, toff_t offset, int origin)
{
	DebugLog("tiff_seek %d %d", (int)offset, (int)origin);
	//TODO: seek pass the end and zero-fill pending
	if (stream_seek((Stream*)fd, offset, origin) < 0)
		return -1;
	return stream_pos_get((Stream*)fd);
}

static int tiff_close(thandle_t fd)
{
	ccUNUSED(fd);
	return 0;
}

static int tiff_map(thandle_t fd, tdata_t* pbase, toff_t* psize)
{
	ccUNUSED(fd);
	ccUNUSED(pbase);
	ccUNUSED(psize);
    return 0;
}

static void tiff_unmap(thandle_t fd, tdata_t base, toff_t size)
{
	ccUNUSED(fd);
	ccUNUSED(base);
	ccUNUSED(size);
    return;
}

static toff_t tiff_size(thandle_t fd)
{
	long save_pos;
	toff_t size;

	save_pos = stream_pos_get((Stream*)fd);
	stream_seek((Stream*)fd, 0, SEEK_END);
	size = stream_pos_get((Stream*)fd);
	stream_seek((Stream*)fd, save_pos, SEEK_SET);

	DebugLog("tiff_size %d %d", (int)save_pos, (int)size);

	return size;
}

int imgio_libtiff_init_inner(CodecLibtiff* codec, ImageIO* imgio,
	bool save)
{
	*codec = (CodecLibtiffLoad){0};

	if (!lib_init_done) {
		DebugLog("TIFF lib init");
		TIFFSetErrorHandler(tiff_error_handler);
		TIFFSetWarningHandler(tiff_warning_handler);
		lib_init_done = true;
	}

	if (imgio->filename) {
		codec->tiff = TIFFOpen(imgio->filename, save ? "w" : "r");
	}
	else if (imgio->s) {
		codec->tiff = TIFFClientOpen("image_io", save ? "wm" : "rm",
			(thandle_t)imgio->s,
			tiff_read, tiff_write, tiff_seek, tiff_close, tiff_size,
			tiff_map, tiff_unmap);
	}
	else
		return IMG_ERROR_UNSUPPORTED_INPUT_TYPE;

	if (!codec->tiff)
		return IMG_ERROR_UNKNOWN;

	return 0;
}

int imgio_libtiff_load_init(CodecLibtiffLoad* codec, ImageIO* imgio)
{
	int r = imgio_libtiff_init_inner(codec, imgio, false);
	if (r) return r;

	if (log_level_check(LOG_LVL_DEBUG)) {
		uint32_t w=0, h=0, rps=0;
		uint16_t bps=0, spp=0, comp=0, pm=0, orien=0, pc=0;
		const char *desc="", *soft="", *date="";
		TIFFGetFieldDefaulted(codec->tiff, TIFFTAG_IMAGEWIDTH, &w);
		TIFFGetFieldDefaulted(codec->tiff, TIFFTAG_IMAGELENGTH, &h);
		TIFFGetFieldDefaulted(codec->tiff, TIFFTAG_ROWSPERSTRIP, &rps);
		TIFFGetFieldDefaulted(codec->tiff, TIFFTAG_BITSPERSAMPLE, &bps);
		TIFFGetFieldDefaulted(codec->tiff, TIFFTAG_SAMPLESPERPIXEL, &spp);
		TIFFGetFieldDefaulted(codec->tiff, TIFFTAG_COMPRESSION, &comp);
		TIFFGetFieldDefaulted(codec->tiff, TIFFTAG_PHOTOMETRIC, &pm);
		TIFFGetFieldDefaulted(codec->tiff, TIFFTAG_ORIENTATION, &orien);
		TIFFGetFieldDefaulted(codec->tiff, TIFFTAG_PLANARCONFIG, &pc);
		TIFFGetFieldDefaulted(codec->tiff, TIFFTAG_IMAGEDESCRIPTION, &desc);
		TIFFGetFieldDefaulted(codec->tiff, TIFFTAG_SOFTWARE, &soft);
		TIFFGetFieldDefaulted(codec->tiff, TIFFTAG_DATETIME, &date);
		log_debug("TIFF: width=%d height=%d bps=%d spp=%d rps=%d compr=%d pm=%d"
			" orien=%d pc=%d",
			w, h, bps, spp, rps, comp, pm, orien, pc);
		log_debug("TIFF: desc=%s soft=%s date=%s",
			desc, soft, date);
	}

	return 0;
}

int imgio_libtiff_save_init(CodecLibtiffSave* codec, ImageIO* imgio)
{
	int r = imgio_libtiff_init_inner(codec, imgio, true);
	if (r) return r;

	codec->deflate = true;
	return 0;
}

void imgio_libtiff_free(CodecLibtiff* codec, ImageIO* imgio)
{
	ccUNUSED(imgio);
	if (codec->tiff) {
		TIFFClose(codec->tiff);
		codec->tiff = 0;
	}
}

/*
	Seek
*/
int imgio_libtiff_load_seek(CodecLibtiffLoad* codec, ImageIO* imgio,
	long offset, int mode)
{
	long idx = 0;

	switch (mode) {
	case IMG_SEEK_SET:
		idx = offset;
		break;
	case IMG_SEEK_CUR:
		idx = offset + codec->nframe;
		break;
	case IMG_SEEK_END:
		idx = offset + codec->nframe_last;
		break;
	default:
		return IMG_ERROR_PARAMS;
	}

	if (idx < 0)
		return IMG_ERROR_SEEK;
	if (imgio->flags & IMGIO_F_END_FOUND && idx >= codec->nframe_last)
		return IMG_ERROR_SEEK;

	// Seek
	if (!TIFFSetDirectory(codec->tiff, idx))
		return IMG_ERROR_SEEK;

	codec->nframe = idx;

	return 0;
}

/*
	Load
*/
int imgio_libtiff_load_op(CodecLibtiffLoad* codec, ImageIO* imgio, Image* img)
{
	if (imgio->flags & IMGIO_F_END_FOUND &&
		codec->nframe_last == codec->nframe)
		return IMG_ERROR_EOF;

	uint32_t w=0, h=0;
	TIFFGetField(codec->tiff, TIFFTAG_IMAGEWIDTH, &w);
	TIFFGetField(codec->tiff, TIFFTAG_IMAGELENGTH, &h);

	bool read_gray_raw = false;
	{
		uint16_t bps=0, spp=0, orien=0, pc=0;
		TIFFGetFieldDefaulted(codec->tiff, TIFFTAG_BITSPERSAMPLE, &bps);
		TIFFGetFieldDefaulted(codec->tiff, TIFFTAG_SAMPLESPERPIXEL, &spp);
		//TIFFGetFieldDefaulted(codec->tiff, TIFFTAG_PHOTOMETRIC, &pm);
		TIFFGetFieldDefaulted(codec->tiff, TIFFTAG_ORIENTATION, &orien);
		TIFFGetFieldDefaulted(codec->tiff, TIFFTAG_PLANARCONFIG, &pc);
		//log_debug(" bps=%d spp=%d pm=%d orien=%d pc=%d",
		//	bps, spp, pm, orien, pc);
		read_gray_raw = (bps==8 && spp==1 && orien==1 && pc==1);
	}

	if (read_gray_raw) {
		int r = img_resize(img, w, h, IMG_FORMAT_GRAY, w);
		if (r) return r;

		uint32_t rps=0;
		TIFFGetFieldDefaulted(codec->tiff, TIFFTAG_ROWSPERSTRIP, &rps);
		if (rps < 1) return IMG_ERROR_LOAD;

		uint32_t nstrip = h/rps;
		tsize_t sz_strip = rps * w;
		unsigned char * dcur = img->data;
		for (uint32_t i=0; i<nstrip; ++i) {
			//tsize_t nbyte =
			TIFFReadEncodedStrip(codec->tiff, i, dcur, sz_strip);
			//dcur += nbyte;
			dcur += sz_strip;
		}
	}
	else {
		int r = img_resize(img, w, h, IMG_FORMAT_RGBA, w*4);
		if (r) return r;

		if (!TIFFReadRGBAImageOriented(codec->tiff, w, h,
			(uint32_t*)img->data, ORIENTATION_TOPLEFT, 0))
		{
			//obs: no need to free img->data here
			return IMG_ERROR_LOAD;
		}
	}

	codec->nframe++;
	if (codec->nframe_last < codec->nframe)
		codec->nframe_last = codec->nframe;

	if (!TIFFReadDirectory(codec->tiff))
		imgio->flags |= IMGIO_F_END_FOUND;

    return 0;
}

/*
	Save
*/
int imgio_libtiff_save_op(CodecLibtiffSave* codec, ImageIO* imgio, Image* img)
{
	uint32_t rowsperstrip = 1;
	ccUNUSED(imgio);

	if (codec->nframe == 0)
	{
	}
	else
	{
		if (!TIFFWriteDirectory(codec->tiff))
			return IMG_ERROR_SAVE;
	}

	{
		TIFFSetField(codec->tiff, TIFFTAG_IMAGEWIDTH, img->w);
		TIFFSetField(codec->tiff, TIFFTAG_IMAGELENGTH, img->h);
		TIFFSetField(codec->tiff, TIFFTAG_BITSPERSAMPLE, 8);
		TIFFSetField(codec->tiff, TIFFTAG_SAMPLESPERPIXEL, img->bypp);
		if (img->bypp == 4) {
			uint16_t v[1] = { EXTRASAMPLE_ASSOCALPHA };
			TIFFSetField(codec->tiff, TIFFTAG_EXTRASAMPLES, 1, v);
		}
		TIFFSetField(codec->tiff, TIFFTAG_PHOTOMETRIC,
			(img->bypp == 1) ? PHOTOMETRIC_MINISBLACK : PHOTOMETRIC_RGB);
		TIFFSetField(codec->tiff, TIFFTAG_ORIENTATION, ORIENTATION_TOPLEFT);
		TIFFSetField(codec->tiff, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG);

		if (codec->deflate) {
			int comptag = COMPRESSION_DEFLATE;
			//int comptag = COMPRESSION_ADOBE_DEFLATE;
			TIFFSetField(codec->tiff, TIFFTAG_COMPRESSION, comptag);
			rowsperstrip = TIFFDefaultStripSize(codec->tiff, -1);
		} else {
			TIFFSetField(codec->tiff, TIFFTAG_COMPRESSION, COMPRESSION_NONE);
			rowsperstrip = 1;
		}
		TIFFSetField(codec->tiff, TIFFTAG_ROWSPERSTRIP, rowsperstrip);
		DebugLog("rowsperstrip %d", rowsperstrip);

		//TIFFSetField(codec->tiff, TIFFTAG_SOFTWARE, TIFFGetVersion());
		//TIFFSetField(codec->tiff, TIFFTAG_DOCUMENTNAME, );
	}

	unsigned strip_size = img->w * img->bypp * rowsperstrip;
	unsigned strip_pitch = img->pitch * rowsperstrip;
	if (rowsperstrip > 1 && strip_pitch != strip_size)  //TODO
		return IMG_ERROR_UNSUPPORTED_FORMAT;

	for (unsigned y=0, i=0; y<img->h; y+=rowsperstrip, ++i) {
		unsigned char* strip = img->data + strip_pitch * i;
		if (y + rowsperstrip > img->h)
			strip_size = img->w * img->bypp * (img->h - y);
		if (TIFFWriteEncodedStrip(codec->tiff, i, strip, strip_size) < 0)
			return IMG_ERROR_SAVE;
	}

	codec->nframe++;
	codec->nframe_last = codec->nframe;
	return 0;
}

/*
	Common
*/
int imgio_libtiff_value_get(CodecLibtiff* obj, ImageIO* imgio,
	int id, void* buf, unsigned bufsz)
{
	ccUNUSED(bufsz);
	switch (id) {
	case IMG_VALUE_COMPRESSION:
		*((unsigned*)buf) = obj->deflate ? 6 : 0;
		break;
	case IMG_VALUE_FRAME_IDX:
		*((unsigned*)buf) = obj->nframe;
		break;
	case IMG_VALUE_FRAME_COUNT:
		if (imgio->flags & IMGIO_F_END_FOUND)
			*((unsigned*)buf) = obj->nframe_last;
		else
			return IMG_ERROR_UNKNOWN;
		break;
/*	case IMG_VALUE_FRAME_DURATION:
		*((double*)buf) = ;
		break; */
	default:
		return IMG_ERROR_UNSUPPORTED_VALUE;
	}
	return 0;
}

int imgio_libtiff_value_set(CodecLibtiff* obj, ImageIO* imgio,
	int id, const void* buf, unsigned bufsz)
{
	ccUNUSED(imgio);
	ccUNUSED(bufsz);
	switch (id) { 
	case IMG_VALUE_COMPRESSION:
		obj->deflate = !!*((unsigned*)buf);
		break;
/*	case IMG_VALUE_FRAME_DURATION:
		= *((double*)buf);
		break; */
	default:
		return IMG_ERROR_UNSUPPORTED_VALUE;
	}
	return 0;
}

/*
	Codec
*/
const ImageCodec img_codec_libtiff = {
	.detect = imgio_libtiff_detect,
	.load = {
		(int (*)(void*, ImageIO*, Image*)) imgio_libtiff_load_op,
		IMG_CODEC_F_ACCEPT_STREAM | IMG_CODEC_F_ACCEPT_FILENAME,
		sizeof(CodecLibtiffLoad),
		(int (*)(void*, ImageIO*)) imgio_libtiff_load_init,
		(void (*)(void*, ImageIO*)) imgio_libtiff_load_free,
		(int (*)(void*, ImageIO*, long, int)) imgio_libtiff_load_seek,
		(int (*)(void*, ImageIO*, int, void*, unsigned))
			imgio_libtiff_load_value_get,
	},
	.save = {
		(int (*)(void*, ImageIO*, Image*)) imgio_libtiff_save_op,
		IMG_CODEC_F_ACCEPT_STREAM | IMG_CODEC_F_ACCEPT_FILENAME,
		sizeof(CodecLibtiffSave),
		(int (*)(void*, ImageIO*)) imgio_libtiff_save_init,
		(void (*)(void*, ImageIO*)) imgio_libtiff_save_free,
		NULL,  //seek
		(int (*)(void*, ImageIO*, int, void*, unsigned))
			imgio_libtiff_save_value_get,
		(int (*)(void*, ImageIO*, int, const void*, unsigned))
			imgio_libtiff_save_value_set,
	},
	.name = "libtiff",
	.ext = "tiff"
};
