/* Copyright 2021-2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: Zlib
 */
// Link with -ltiff
#define IMGIO_LIBAV_IMPL
#include "image_io_libav.h"
#include "logging.h"
#include <time.h>

//#define DebugLog(...) log_info(__VA_ARGS__)
#define DebugLog(...) {}

#define MAX_(X,Y) (X>Y ? X : Y)
#define MIN_(X,Y) (X<Y ? X : Y)

static inline
void string_copy(char* dst, const char* src, unsigned dstsz) {
	if (!dst || !dstsz) return;
	while (dstsz>1 && *src) { *dst++ = *src++; dstsz--; }
	*dst = 0;
}

#define IMGIO_LIBAV_ERROR(...) do { \
	log_error(__VA_ARGS__); \
	goto error; \
} while(0)

/*
	Load Init & free
*/

int imgio_libav_load_init(CodecLibAvLoad* obj, ImageIO* imgio)
{
	*obj = (CodecLibAvLoad){0};

	obj->fmtctx = avformat_alloc_context();
	if (!obj->fmtctx)
		IMGIO_LIBAV_ERROR("avformat_alloc_context");

	//avio_alloc_context();

	if (avformat_open_input(&obj->fmtctx, imgio->filename, NULL, NULL) < 0)
		IMGIO_LIBAV_ERROR("avformat_open_input");

	log_debug("LibAv: Format: %s", obj->fmtctx->iformat->long_name);

	// Read a few frames to fill some information
	if (avformat_find_stream_info(obj->fmtctx, NULL) < 0)
		IMGIO_LIBAV_ERROR("avformat_find_stream_info");

#ifdef DEBUG
	for (int i=0; i<obj->fmtctx->nb_streams; ++i) {
		AVStream* st = obj->fmtctx->streams[i];
		AVCodecParameters* cp = st->codecpar;

		log_info("Stream %d: frames %d, framerate %.1f",
			i, (int)st->nb_frames, av_q2d(st->avg_frame_rate));

		if (st->codecpar->codec_type == AVMEDIA_TYPE_VIDEO) {
			log_info("  Video: shape %dx%d, bitrate %.1f KB/s",
				cp->width, cp->height, (double)cp->bit_rate/8/1024);
		}
	}
#endif

	obj->isv = av_find_best_stream(obj->fmtctx, AVMEDIA_TYPE_VIDEO, -1, -1,
		&obj->codec, 0);
	if (obj->isv < 0)
		IMGIO_LIBAV_ERROR("av_find_best_stream");

	log_debug("LibAv: Video Stream: %d, Decoder: %s",
		obj->isv, obj->codec->long_name);

	obj->avctx = avcodec_alloc_context3(obj->codec);
	if (!obj->avctx)
		IMGIO_LIBAV_ERROR("avcodec_alloc_context3");

	AVCodecParameters* cp = obj->fmtctx->streams[obj->isv]->codecpar;
	if (avcodec_parameters_to_context(obj->avctx, cp) < 0)
		IMGIO_LIBAV_ERROR("avcodec_parameters_to_context");

	if (avcodec_open2(obj->avctx, obj->codec, NULL) < 0)
		IMGIO_LIBAV_ERROR("avcodec_open2");

	obj->pkt = av_packet_alloc();
	if (!obj->pkt)
		IMGIO_LIBAV_ERROR("av_packet_alloc");

	obj->frm = av_frame_alloc();
	if (!obj->frm)
		IMGIO_LIBAV_ERROR("av_frame_alloc");

	return 0;

error:
	imgio_libav_load_free(obj, imgio);
	return IMG_ERROR_UNKNOWN;
}

void imgio_libav_load_free(CodecLibAvLoad* obj, ImageIO* imgio)
{
	ccUNUSED(imgio);
	if (obj->swsctx) sws_freeContext(obj->swsctx);
	if (obj->flags & IMG_LIBAV_PACKET_UNREF)
		av_packet_unref(obj->pkt);
	if (obj->frm) av_frame_free(&obj->frm);
	if (obj->pkt) av_packet_free(&obj->pkt);
	if (obj->avctx) avcodec_free_context(&obj->avctx);
	if (obj->fmtctx) avformat_close_input(&obj->fmtctx);
	//if (obj->fmtctx) avformat_free_context(&obj->fmtctx);
}

/*
	Seek
*/

#define last_frame_get(obj, imgio) \
	((imgio->flags & IMGIO_F_END_FOUND) ? obj->nframe_last \
	: MAX_(obj->fmtctx->streams[obj->isv]->nb_frames, obj->nframe_last))

inline static
int64_t idx_to_pts(AVStream* stream, int64_t idx)
{
	return idx
		* ((int64_t)stream->r_frame_rate.den * stream->time_base.den)
		/ ((int64_t)stream->r_frame_rate.num * stream->time_base.num);
}

inline static
int64_t pts_to_idx(AVStream* stream, int64_t pts)
{
	return pts
		* ((int64_t)stream->r_frame_rate.num * stream->time_base.num)
		/ ((int64_t)stream->r_frame_rate.den * stream->time_base.den);
}

int imgio_libav_load_seek(CodecLibAvLoad* obj, ImageIO* imgio,
	long offset, int mode)
{
	long idx = 0;

	switch (mode) {
	case IMG_SEEK_SET:
		idx = offset;
		break;
	case IMG_SEEK_CUR:
		idx = offset + obj->nframe;
		break;
	case IMG_SEEK_END:
		idx = offset + last_frame_get(obj, imgio); //obj->nframe_last;
		break;
	default:
		return IMG_ERROR_PARAMS;
	}

	if (idx < 0)
		return IMG_ERROR_SEEK;
	if (idx == obj->nframe)
		return 0;
	if (imgio->flags & IMGIO_F_END_FOUND && idx > obj->nframe_last)
		return IMG_ERROR_SEEK;

	// Seek
	int64_t seek_pts = idx_to_pts(obj->fmtctx->streams[obj->isv], idx);
	if (av_seek_frame(obj->fmtctx, obj->isv, seek_pts,
		AVSEEK_FLAG_BACKWARD) < 0)
	{
		IMGIO_LIBAV_ERROR("av_seek_frame");
	}

	avcodec_flush_buffers(obj->avctx);

	obj->target_pts = seek_pts;
	obj->nframe = idx;

	return 0;

error:
	return IMG_ERROR_SEEK;
}

/*
	Load
*/

int imgio_libav_load_op(CodecLibAvLoad* obj, ImageIO* imgio, Image* img)
{
	bool done=false;
	int r=0;
	while (!done) {
		if (obj->flags & IMG_LIBAV_PACKET_UNREF) {
			obj->flags &= ~IMG_LIBAV_PACKET_UNREF;
			av_packet_unref(obj->pkt);
		}

		r = av_read_frame(obj->fmtctx, obj->pkt);
		//log_info("av_read_frame r=%d done=%d", r, done);
		if (r == AVERROR_EOF)
			break;
		if (r)
			IMGIO_LIBAV_ERROR("av_read_frame: %d", r);

		obj->flags |= IMG_LIBAV_PACKET_UNREF;

		//log_debug("Packet: stream %d, size %d, pos %ld, pts %ld, duration %ld",
		//	obj->pkt->stream_index, obj->pkt->size,	(long)obj->pkt->pos,
		//	(long)obj->pkt->pts, (long)obj->pkt->duration);

		if (obj->pkt->stream_index != obj->isv)
			continue;

		if ((r = avcodec_send_packet(obj->avctx, obj->pkt)))
			IMGIO_LIBAV_ERROR("avcodec_send_packet: %d", r);

		r = avcodec_receive_frame(obj->avctx, obj->frm);
		if (r == AVERROR(EAGAIN))
			continue;
		if (r == AVERROR_EOF)
			break;
		if (r)
			IMGIO_LIBAV_ERROR("avcodec_receive_frame: %d", r);

		//log_debug("Frame: format %d, shape %dx%d, pts %ld",
		//	obj->frm->format, obj->frm->width, obj->frm->height,
		//	(long)obj->frm->pts);

		if (obj->target_pts) {
			if (obj->frm->pts < obj->target_pts)
				continue;
			log_debug("libav: target pts found: %ld >= %ld",
				(long)obj->frm->pts, (long)obj->target_pts);
			obj->target_pts = 0;
		}
		
		int	w = obj->frm->width, h = obj->frm->height;

		bool convert = false;
		ImgFormat format = IMG_FORMAT_NULL;
		unsigned pitch = 0;
		switch (obj->frm->format) {
		case AV_PIX_FMT_RGB24:
			if (~imgio->oflags & IMG_OF_GRAY) {
				format = IMG_FORMAT_RGB;
				pitch = obj->frm->linesize[0];
			}
			break;
		case AV_PIX_FMT_RGB32:
			if (!(imgio->oflags & (IMG_OF_GRAY | IMG_OF_NO_ALPHA))) {
				format = IMG_FORMAT_RGBA;
				pitch = obj->frm->linesize[0];
			}
			break;
		case AV_PIX_FMT_GRAY8:
			format = IMG_FORMAT_GRAY;
			pitch = obj->frm->linesize[0];
			break;
		}
		if (!format) {
			if (!(obj->swsctx && obj->swsctx_pixfmt == obj->frm->format &&
					(int)obj->swsctx_w == w && (int)obj->swsctx_h == h))
			{
				obj->pixdesc = av_pix_fmt_desc_get(obj->frm->format);
				if (!obj->pixdesc)
					IMGIO_LIBAV_ERROR("av_pix_fmt_desc_next");
				
				enum AVPixelFormat avpixfmt;
				if (obj->pixdesc->flags & AV_PIX_FMT_FLAG_ALPHA &&
						~imgio->oflags & IMG_OF_NO_ALPHA)
				{
					obj->swsctx_imgfmt = IMG_FORMAT_RGBA;
					avpixfmt = AV_PIX_FMT_RGB32;
					log_debug("swscale init: from %s to RGBA", obj->pixdesc->name);
				}
				else if (obj->pixdesc->nb_components > 1 &&
						~imgio->oflags & IMG_OF_GRAY)
				{
					obj->swsctx_imgfmt = IMG_FORMAT_RGB;
					avpixfmt = AV_PIX_FMT_RGB24;
					log_debug("swscale init: from %s to RGB", obj->pixdesc->name);
				}
				else
				{
					obj->swsctx_imgfmt = IMG_FORMAT_GRAY;
					avpixfmt = AV_PIX_FMT_GRAY8;
					log_debug("swscale init: from %s to gray", obj->pixdesc->name);
				}
				
				obj->swsctx = sws_getContext(
						w, h, obj->frm->format,  //src
						w, h, avpixfmt,			 //dst
						0, NULL, NULL, NULL);
				
				if (!obj->swsctx)
					IMGIO_LIBAV_ERROR("sws_getContext");
			
				obj->swsctx_pixfmt = obj->frm->format;
				obj->swsctx_w = w;
				obj->swsctx_h = h;
			}
			format = obj->swsctx_imgfmt;
			convert = true;
		}

		int r2 = img_resize(img, w, h, format, pitch);
		if (r2) return r2;

		if (convert)
		{
			uint8_t * pixels[4] = { img->data };
			int pitchs[4] = { img->pitch };
			int h2 = sws_scale(obj->swsctx,
						(const uint8_t * const *)obj->frm->data,
							obj->frm->linesize, 0, h,
						pixels, pitchs);
			if (h2 != (int)img->h)
				IMGIO_LIBAV_ERROR("sws_scale");
		}
		else
		{
			memcpy(img->data, obj->frm->data[0], img->h * img->pitch);
		}

		//if (obj->frm->data[1] && obj->frm->data[2])
		//	log_debug("frm->data[1][0]=%d [2][0]=%d",
		//		obj->frm->data[1][0], obj->frm->data[2][0]);

		obj->nframe++;
		if (obj->nframe_last < obj->nframe)
			obj->nframe_last = obj->nframe;

		done = true;
	}

	if (!done) {
		if (obj->target_pts) {
			if (~obj->flags & IMG_LIBAV_LOAD_RETRY) {
				// Target seek frame could not be reached,
				// estimate the current frame index and seek there
				unsigned nframe_last =
					pts_to_idx(obj->fmtctx->streams[obj->isv], obj->frm->pts);
				if (!imgio_libav_load_seek(obj, imgio, nframe_last, 0)) {
					obj->flags |= IMG_LIBAV_LOAD_RETRY;
					return imgio_libav_load_op(obj, imgio, img);
				}
			}
		}
		else {
			imgio->flags |= IMGIO_F_END_FOUND;
		}
		return IMG_ERROR_EOF;
	}

	obj->flags &= ~IMG_LIBAV_LOAD_RETRY;
    return 0;

error:
	return IMG_ERROR_LOAD;
}

/*
	Values get & set
*/

int imgio_libav_value_get(CodecLibAv* obj, ImageIO* imgio,
	int id, void* buf, unsigned bufsz)
{
	int r=0;

	AVStream* st = obj->fmtctx->streams[obj->isv];

	switch (id) {
	case IMG_VALUE_FRAME_IDX:
		*((unsigned*)buf) = obj->nframe;
		break;
	case IMG_VALUE_FRAME_COUNT:
		*((unsigned*)buf) = last_frame_get(obj, imgio);
		break;
	case IMG_VALUE_FRAME_DURATION:
		*((double*)buf) = 1.0 / av_q2d(st->avg_frame_rate);
		break;
	case IMG_VALUE_METADATA: {
		const AVDictionaryEntry* e =
			av_dict_get(obj->fmtctx->metadata, buf, NULL, 0);
		if (!e)
			r = IMG_ERROR_UNKNOWN;
		else {
			r = strlen(e->value);
			if (bufsz)
				string_copy(buf, e->value, bufsz);
		}
		} break;
	default:
		r=IMG_ERROR_UNSUPPORTED_VALUE;
	}

	return r;
}

int imgio_libav_value_set(CodecLibAv* obj, ImageIO* imgio,
	int id, void* buf, unsigned bufsz)
{
	AVStream* st = obj->fmtctx->streams[obj->isv];
	ccUNUSED(imgio);  ccUNUSED(bufsz);

	switch (id) {
		//TODO: quality, compresion, and key frames
	case IMG_VALUE_FRAME_DURATION:
		if (obj->flags & IMG_LIBAV_HEADER_PENDING) {
			double d = *((double*)buf);
			if (!(d > 0 && isfinite(d))) return IMG_ERROR_UNSUPPORTED_VALUE;
			st->avg_frame_rate = av_d2q(1.0 / d, 1000000);
		}
		else
			return IMG_ERROR_UNKNOWN;
		break;
	case IMG_VALUE_METADATA: {
		const char* key = buf;
		const char* value = buf;
		while (*value++);  //skips first zero-terminated string
		int rr = av_dict_set(&obj->fmtctx->metadata, key, value, 0);
		if (rr < 0)
			log_error("av_dict_set '%s'='%s': %d", key, value, rr);
		//else
		//	log_debug("av_dict_set '%s'='%s'", key, value);
		} break;
	default:
		return IMG_ERROR_UNSUPPORTED_VALUE;
	}

	return 0;
}


/*
	Save Init & free
*/

int imgio_libav_save_init(CodecLibAvSave* obj, ImageIO* imgio)
{
	*obj = (CodecLibAvSave){0};

	if (log_level_check(LOG_LVL_INFO+1))
		av_log_set_level(AV_LOG_INFO);
	else  // AV_LOG_INFO is too verbose
		av_log_set_level(AV_LOG_WARNING);

	if (avformat_alloc_output_context2(
		&obj->fmtctx, NULL, NULL, imgio->filename) < 0)
		IMGIO_LIBAV_ERROR("avformat_alloc_output_context2");

	// Open output file
	if (~obj->fmtctx->oformat->flags & AVFMT_NOFILE)
		if (avio_open(&obj->fmtctx->pb, imgio->filename, AVIO_FLAG_WRITE) < 0)
			IMGIO_LIBAV_ERROR("avio_open");

	AVStream * st = NULL;
	st = avformat_new_stream(obj->fmtctx, NULL);
	if (!st)
		IMGIO_LIBAV_ERROR("avformat_new_stream");

	//obj->isv = 0;

	obj->codec = avcodec_find_encoder(obj->fmtctx->oformat->video_codec);
	if (!obj->codec)
		IMGIO_LIBAV_ERROR("avcodec_find_encoder");

	obj->avctx = avcodec_alloc_context3(obj->codec);
	if (!obj->avctx)
		IMGIO_LIBAV_ERROR("avcodec_alloc_context3");

	//obj->avctx->width = 0;	//to be filled
	//obj->avctx->height = 0;	//to be filled

	if (obj->codec->pix_fmts)
		obj->avctx->pix_fmt = obj->codec->pix_fmts[0];
	else
		obj->avctx->pix_fmt = AV_PIX_FMT_YUV420P;

    // frame rate, can be set until the first frame saved
    //obj->avctx->time_base = (AVRational){1, 25};
    //obj->avctx->framerate = (AVRational){25, 1};
	//st->time_base = obj->avctx->time_base;
	st->avg_frame_rate = (AVRational){25, 1};

	// emit one intra frame every ten frames
	//obj->avctx->gop_size = 10;
	//obj->avctx->max_b_frames = 1;
	//
	//obj->avctx->bit_rate = 400000;
	//obj->avctx->qmin = 0;
	//obj->avctx->qmax = 30; //25; //20;

	if (obj->codec->id == AV_CODEC_ID_H264) {
		//TODO config or ini
		//av_dict_set(&obj->codec_options, "preset", "slow", 0);
	}

	if (obj->fmtctx->oformat->flags & AVFMT_GLOBALHEADER)
		obj->avctx->flags |= AV_CODEC_FLAG_GLOBAL_HEADER;
	//
	obj->pkt = av_packet_alloc();
	if (!obj->pkt)
		IMGIO_LIBAV_ERROR("av_packet_alloc");

	obj->frm = av_frame_alloc();
	if (!obj->frm)
		IMGIO_LIBAV_ERROR("av_frame_alloc");

	obj->flags |= IMG_LIBAV_HEADER_PENDING;

	log_debug("LibAv: Format: %s, Codec: %s"
		, obj->fmtctx->oformat->long_name
		, obj->codec->long_name);

	{
		time_t t = time(NULL);
		struct tm* ts = gmtime(&t);  //warning: this is not thread safe
		char buffer[128];

		snprintf(buffer, sizeof(buffer), "%04d-%02d-%02dT%02d:%02d:%02dZ",
			1900+ts->tm_year, 1+ts->tm_mon, ts->tm_mday,
			ts->tm_hour, ts->tm_min, ts->tm_sec);
		av_dict_set(&obj->fmtctx->metadata, "creation_time", buffer, 0);
	}

	return 0;

error:
	imgio_libav_save_free(obj, imgio);
	return IMG_ERROR_UNKNOWN;
}

// Finish the codec initilization using information from the first frame
int imgio_libav_save_init_finish(CodecLibAvSave* obj, ImageIO* imgio,
	Image* img)
{
	ccUNUSED(imgio);
	if (~obj->flags & IMG_LIBAV_HEADER_PENDING)
		return IMG_ERROR_UNKNOWN;

	AVStream* st = obj->fmtctx->streams[obj->isv];

	// stream->avg_frame_rate can be set up to this point
	st->time_base = av_inv_q(st->avg_frame_rate);
	obj->avctx->time_base = st->time_base;
    obj->avctx->framerate = st->avg_frame_rate;

	obj->avctx->width = img->w;
	obj->avctx->height = img->h;

	if (avcodec_open2(obj->avctx, obj->codec, &obj->codec_options) < 0)
		IMGIO_LIBAV_ERROR("avcodec_open2");

	//log_debug("#codec_options=%d", av_dict_count(obj->codec_options));

	if (avcodec_parameters_from_context(st->codecpar, obj->avctx) < 0)
		IMGIO_LIBAV_ERROR("avcodec_parameters_from_context");

	// Print information
#ifdef DEBUG
    av_dump_format(obj->fmtctx, 0, imgio->filename, 1);
#endif

	if (avformat_write_header(obj->fmtctx, NULL) < 0)
		IMGIO_LIBAV_ERROR("avformat_write_header");

	obj->frm->width  = img->w;
	obj->frm->height = img->h;

	obj->frm->format = obj->avctx->pix_fmt;
	
	obj->pixdesc = av_pix_fmt_desc_get(obj->frm->format);
	if (!obj->pixdesc)
		IMGIO_LIBAV_ERROR("av_pix_fmt_desc_next");

    if (av_frame_get_buffer(obj->frm, 0) < 0)
		IMGIO_LIBAV_ERROR("av_frame_get_buffer");

	if (obj->frm->format == AV_PIX_FMT_YUV420P) {
	//if (obj->pixdesc->flags & ???) {
		// Fill Cr & Cb for gray
		if (obj->frm->data[1])
			memset(obj->frm->data[1], 128,
				obj->frm->linesize[1] * obj->frm->height/2);
		if (obj->frm->data[2])
			memset(obj->frm->data[2], 128,
				obj->frm->linesize[2] * obj->frm->height/2);
	}

	log_debug("LibAV: Codec PixFmt: %d %s, Frame PixFmt: %d, FPS: %.1f"
		, obj->avctx->pix_fmt, obj->pixdesc->name, obj->frm->format
		, av_q2d(st->avg_frame_rate));

	//log_debug("LibAV: gop_size=%d qmax=%d", obj->avctx->gop_size, obj->avctx->qmax);

	obj->duration_tb = av_rescale_q(1, obj->avctx->time_base,
		av_inv_q(st->avg_frame_rate));

	obj->flags &= ~IMG_LIBAV_HEADER_PENDING;

	return 0;

error:
	return IMG_ERROR_UNKNOWN;
}

int imgio_libav_save_frame_write(CodecLibAvSave* obj, ImageIO* imgio,
	bool flush)
{
	ccUNUSED(imgio);
	if (avcodec_send_frame(obj->avctx, flush ? 0 : obj->frm) < 0)
		IMGIO_LIBAV_ERROR("avcodec_send_frame");

	int r=0;
	while (r >= 0) {
		r = avcodec_receive_packet(obj->avctx, obj->pkt);
		if (r == AVERROR(EAGAIN) || r == AVERROR_EOF)
			break;
		if (r < 0)
			IMGIO_LIBAV_ERROR("avcodec_receive_packet");

		obj->pkt->stream_index = obj->isv;
		obj->pkt->duration = obj->duration_tb;

		av_packet_rescale_ts(obj->pkt, obj->avctx->time_base,
			obj->fmtctx->streams[obj->isv]->time_base);

		//log_debug("Packet, pts=%ld, dts=%ld, duration=%ld"
		//	, (long)obj->pkt->pts, (long)obj->pkt->dts
		//	, (long)obj->pkt->duration);

		if (av_interleaved_write_frame(obj->fmtctx, obj->pkt) < 0)
			IMGIO_LIBAV_ERROR("av_interleaved_write_frame");

		av_packet_unref(obj->pkt);
	}

	return 0;

error:
	return IMG_ERROR_SAVE;
}


int imgio_libav_save_finish(CodecLibAvSave* obj, ImageIO* imgio)
{
	int r=0;
	if ((r = imgio_libav_save_frame_write(obj, imgio, true)))
		return r;

    if (av_write_trailer(obj->fmtctx) < 0)
		IMGIO_LIBAV_ERROR("av_write_trailer");

error:
	return IMG_ERROR_UNKNOWN;
}

void imgio_libav_save_free(CodecLibAvSave* obj, ImageIO* imgio)
{
	if (~obj->flags & IMG_LIBAV_HEADER_PENDING)
		imgio_libav_save_finish(obj, imgio);

	if (obj->swsctx) sws_freeContext(obj->swsctx);
	if (obj->flags & IMG_LIBAV_PACKET_UNREF)
		av_packet_unref(obj->pkt);
	if (obj->frm) av_frame_free(&obj->frm);
	if (obj->pkt) av_packet_free(&obj->pkt);
	if (obj->avctx) avcodec_free_context(&obj->avctx);
	if (obj->codec_options) av_dict_free(&obj->codec_options);
	if (obj->fmtctx && ~obj->fmtctx->oformat->flags & AVFMT_NOFILE)
        avio_closep(&obj->fmtctx->pb);
	if (obj->fmtctx) avformat_free_context(obj->fmtctx);
}

/*
	Save
*/

int imgio_libav_save_op(CodecLibAvSave* obj, ImageIO* imgio, Image* img)
{
	int r=0;

	if (obj->flags & IMG_LIBAV_HEADER_PENDING)
		if ((r = imgio_libav_save_init_finish(obj, imgio, img)))
			return r;

	if (obj->frm->format == AV_PIX_FMT_YUV420P &&
			img->format == IMG_FORMAT_GRAY)
	{
		unsigned y;
		const unsigned h = MIN_((unsigned)obj->frm->height, img->h);
		//const unsigned w = MIN_((unsigned)obj->frm->width, img->w);
		const unsigned lsz = MIN_((unsigned)obj->frm->linesize[0], img->pitch);
		for (y=0; y<h; ++y) {
			memcpy(	obj->frm->data[0]+(y * obj->frm->linesize[0]),
					img->data+(y * img->pitch),
					lsz);
		}
	}
	//TODO: more direct copy formats
	else 
	{
		if (!(obj->swsctx && obj->swsctx_imgfmt == img->format &&
				obj->swsctx_w == img->w && obj->swsctx_h == img->h))
		{
			// Pixel format conversion set-up
			enum AVPixelFormat avpixfmt;
			switch (img->format) {
			case IMG_FORMAT_GRAY:
				avpixfmt = AV_PIX_FMT_GRAY8;
				log_debug("swscale init: from gray to %s", obj->pixdesc->name);
				break;
			case IMG_FORMAT_RGB:
				avpixfmt = AV_PIX_FMT_RGB24;
				log_debug("swscale init: from RGB to %s", obj->pixdesc->name);
				break;
			case IMG_FORMAT_RGBA:
				avpixfmt = AV_PIX_FMT_RGB32;
				log_debug("swscale init: from RGBA to %s", obj->pixdesc->name);
				break;
			default:
				return IMG_ERROR_UNSUPPORTED_FORMAT;
			}

			if (obj->swsctx)
				sws_freeContext(obj->swsctx);				
			
			obj->swsctx = sws_getContext(
					img->w, img->h, avpixfmt,
					obj->frm->width, obj->frm->height, obj->frm->format,
					0, NULL, NULL, NULL);
			if (!obj->swsctx)
				IMGIO_LIBAV_ERROR("sws_getContext");
			
			obj->swsctx_imgfmt = img->format;
			obj->swsctx_w = img->w;
			obj->swsctx_h = img->h;
		}

		const uint8_t * pixels[4] = { img->data };
		int pitchs[4] = { img->pitch };
		int h2 = sws_scale(obj->swsctx,
					pixels, pitchs, 0, img->h,
					obj->frm->data, obj->frm->linesize);
		if (h2 != (int)img->h)
			IMGIO_LIBAV_ERROR("sws_scale");
	}

	obj->frm->pts = obj->duration_tb * obj->nframe;
	//obj->frm->pts = av_rescale_q(obj->nframe, obj->avctx->time_base,
	//	obj->fmtctx->streams[obj->isv]->time_base);
	//log_debug("Frame %d, pts=%ld", obj->nframe, (long)frm->pts);

	if ((r = imgio_libav_save_frame_write(obj, imgio, false)))
		return r;

	obj->nframe++;
	obj->nframe_last = obj->nframe;

    return 0;

error:
	return IMG_ERROR_SAVE;
}

/*
	Codec
*/
const ImageCodec img_codec_libav = {
	0,
	{
		(int (*)(void*, ImageIO*, Image*)) imgio_libav_load_op,
		IMG_CODEC_F_ACCEPT_FILENAME | IMG_CODEC_F_TRY_DETECT,
		sizeof(CodecLibAvLoad),
		(int (*)(void*, ImageIO*)) imgio_libav_load_init,
		(void (*)(void*, ImageIO*)) imgio_libav_load_free,
		(int (*)(void*, ImageIO*, long, int)) imgio_libav_load_seek,
		(int (*)(void*, ImageIO*, int, void*, unsigned))
			imgio_libav_load_value_get,
		(int (*)(void*, ImageIO*, int, const void*, unsigned))
			imgio_libav_load_value_set,
	},
	{
		(int (*)(void*, ImageIO*, Image*)) imgio_libav_save_op,
		IMG_CODEC_F_ACCEPT_FILENAME | IMG_CODEC_F_TRY_DETECT,
		sizeof(CodecLibAvLoad),
		(int (*)(void*, ImageIO*)) imgio_libav_save_init,
		(void (*)(void*, ImageIO*)) imgio_libav_save_free,
		0,
		(int (*)(void*, ImageIO*, int, void*, unsigned))
			imgio_libav_save_value_get,
		(int (*)(void*, ImageIO*, int, const void*, unsigned))
			imgio_libav_save_value_set,
	},
	"libav", 0
};

