/* Copyright 2024-2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: Zlib
 */
#include "image_io_imgseq.h"
#include "logging.h"
#include "str_util.h"

int imgio_imgseq_detect_filename(const char* filename,
	unsigned* fn_num_begin, unsigned* fn_num_len)
{
	if (!filename || !filename[0])
		return IMG_ERROR_UNSUPPORTED_INPUT_TYPE;

	// Search the filename for a digits sequence starting from the back
	// i.e.: path/preffix####suffix.ext
	const char* cur = filename + strlen(filename) - 1;
	const char* fn_num_e=0;
	for (; filename <= cur; --cur) {
		if ('0' <= *cur && *cur <= '9') {
			if (!fn_num_e) fn_num_e = cur;
			if (fn_num_e - cur >= 8)	//8 digits maximum
				break;
		}
		else if (fn_num_e)
			break;
	}

	if (fn_num_begin)
		*fn_num_begin = cur+1 - filename;
	if (fn_num_len)
		*fn_num_len = fn_num_e - cur;

	if ((fn_num_e - cur) < 4)
		return IMG_ERROR_UNKNOWN;

	return 0;
}

int imgio_imgseq_init(CodecImgSeq* obj, ImageIO* imgio)
{
	int r=0;

	*obj = (CodecImgSeq){0};

	r = imgio_imgseq_detect_filename(imgio->filename,
		&obj->fn_num_begin, &obj->fn_num_len);
	if (r) return r;

	//TODO: reuse the same ImageIO using a custom ImageCodec?
	r = imgio_open_filename(&obj->imgio, imgio->filename, imgio->oflags, 0);
	if (r) return r;

	obj->flags |= IMGIO_IMGSEQ_OPEN_DONE;
	obj->codec = obj->imgio.codec;

	if (!strcpy_alloc(&obj->filename, imgio->filename))
		return IMG_ERROR_OUT_OF_MEMORY;

	obj->nframe_base = atoi(obj->filename + obj->fn_num_begin);

	return 0;
}

void imgio_imgseq_free(CodecImgSeq* obj, ImageIO* imgio)
{
	ccUNUSED(imgio);
	if (obj->filename) {
		free(obj->filename);
		obj->filename = 0;
	}
	imgio_free(&obj->imgio);
}

int imgio_imgseq_open(CodecImgSeq* obj, ImageIO* imgio, unsigned nframe)
{
	char buf[12];
	sprintf(buf, "%08d", obj->nframe_base + nframe);
	memcpy(obj->filename + obj->fn_num_begin,
		buf + 8 - obj->fn_num_len, obj->fn_num_len);

	log_debug("ImgSeq seek '%s'", obj->filename);

	int r = imgio_open_filename(&obj->imgio, obj->filename,
		imgio->oflags, obj->codec);

	if (r == IMG_ERROR_FILE_OPEN && obj->nframe_last+1 == nframe)
	{
		imgio->flags |= IMGIO_F_END_FOUND;
		r = IMG_ERROR_EOF;
	}

	if (r) {
		obj->flags |= IMGIO_IMGSEQ_OPEN_ERROR;
		return r;
	}

	obj->nframe = nframe;
	if (obj->nframe_last < nframe)
		obj->nframe_last = nframe;

	obj->flags |= IMGIO_IMGSEQ_OPEN_DONE;
	obj->flags &= ~IMGIO_IMGSEQ_OPEN_ERROR;

	return 0;
}

int imgio_imgseq_seek(CodecImgSeq* obj, ImageIO* imgio,
	long offset, int mode)
{
	switch (mode) {
	case IMG_SEEK_SET:
		break;
	case IMG_SEEK_CUR:
		offset += obj->nframe;
		break;
	case IMG_SEEK_END:
		offset += obj->nframe_last+1;
		break;
	default:
		return IMG_ERROR_PARAMS;
	}

	if (offset < 0)
		return IMG_ERROR_SEEK;
	//if (imgio->flags & IMGIO_F_END_FOUND && offset > obj->nframe_last)
	//	return IMG_ERROR_SEEK;
	if (offset > 99999999)	//TODO: calculate real maximum from fn_num_len
		return IMG_ERROR_SEEK;
	if (offset == obj->nframe && obj->flags & IMGIO_IMGSEQ_OPEN_DONE)
		return 0;

	return imgio_imgseq_open(obj, imgio, offset);
}

int imgio_imgseq_op(CodecImgSeq* obj, ImageIO* imgio, Image* img)
{
	int r=0;

	if (~obj->flags & IMGIO_IMGSEQ_OPEN_DONE) {
		r = imgio_imgseq_open(obj, imgio, obj->nframe);
		if (r) return r;
	}

	if (imgio->oflags & IMG_OF_SAVE)
		r = imgio_save(&obj->imgio, img);
	else
		r = imgio_load(&obj->imgio, img);
	if (r) return r;

	obj->nframe++;
	obj->flags &= ~IMGIO_IMGSEQ_OPEN_DONE;

	return 0;
}

int imgio_imgseq_value_get(CodecImgSeq* obj, ImageIO* imgio,
	int id, void* buf, unsigned bufsz)
{
	switch (id) {
	case IMG_VALUE_FRAME_IDX:
		*((unsigned*)buf) = obj->nframe;
		break;
	case IMG_VALUE_FRAME_COUNT:
		if (imgio->flags & IMGIO_F_END_FOUND)
			*((unsigned*)buf) = obj->nframe_last + 1;
		else
			return IMG_ERROR_UNKNOWN;
		break;
	default: {
		if (obj->flags & IMGIO_IMGSEQ_OPEN_ERROR)
			return IMG_ERROR_UNKNOWN;
		int r = imgio_value_get(&obj->imgio, id, buf, bufsz);
		if (r == IMG_ERROR_UNSUPPORTED_FUNCTION)
			r = IMG_ERROR_UNSUPPORTED_PARAM;
		return r;
	}
	}
	return 0;
}

const ImageCodec img_codec_imgseq = {
	.load={
		(int (*)(void*, ImageIO*, Image*)) imgio_imgseq_op,
		IMG_CODEC_F_ACCEPT_FILENAME,
		sizeof(CodecImgSeq),
		(int (*)(void*, ImageIO*)) imgio_imgseq_init,
		(void (*)(void*, ImageIO*)) imgio_imgseq_free,
		(int (*)(void*, ImageIO*, long, int)) imgio_imgseq_seek,
		(int (*)(void*, ImageIO*, int, void*, unsigned))
			imgio_imgseq_value_get,
	},
	.save={
		(int (*)(void*, ImageIO*, Image*)) imgio_imgseq_op,
		IMG_CODEC_F_ACCEPT_FILENAME,
		sizeof(CodecImgSeq),
		(int (*)(void*, ImageIO*)) imgio_imgseq_init,
		(void (*)(void*, ImageIO*)) imgio_imgseq_free,
		(int (*)(void*, ImageIO*, long, int)) imgio_imgseq_seek,
		(int (*)(void*, ImageIO*, int, void*, unsigned))
			imgio_imgseq_value_get,
	},
	.name="ImgSeq"
};

