/* Copyright 2024-2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: Zlib
 */
#include "image_draw.h"
#include "image_draw_inner.h"
#include <math.h>
#include <assert.h>
#include <stdlib.h>

#define SWAPT_(T,A,B)	{ T tmp=A; A=B; B=tmp; }
#define MIN_(L,R)		((L) > (R) ? (R) : (L))
#define MAX_(L,R)		((L) < (R) ? (R) : (L))

//#include <stdio.h>
//#define dprintf  printf
//#define dprintf(...)  do {} while(0) 

void imgdraw_pixel_inner(Image* dst, int x, int y, ImgColor col, bool blend)
{
	IMGDRAW_INNER(IMGDRAW_FORM_PIXEL, IMGDRAW_FILL_COLOR)
}

void imgdraw_rect_color_inner(Image* dst, int x1, int y1, int x2, int y2,
	ImgColor col, bool blend)
{
	IMGDRAW_INNER(IMGDRAW_FORM_RECT, IMGDRAW_FILL_COLOR)
}

static inline
void imgdraw_line_inner(Image* dst,
	int w, int h, int y1, int y2, float m, float b,
	ImgColor col, bool transp, bool blend)
{
	// y-clip
	if (y1 < 0)  y1 = 0;
	if (y2 >= h) y2 = h-1;
	
	// x-clip
	int x1 = y1*m +b +0.5;
	int x2 = y2*m +b +0.5;

	if (m > 0) {
		if (x1 < 0 ) y1 = ceil(-b/m);
		if (x2 >= w) y2 = floor((w-1-b)/m);
	}
	else if	(m < 0)	{
		if (x1 >= w) y1 = ceil((w-1-b)/m);
		if (x2 < 0 ) y2 = floor(-b/m);
	}
	else {
		if (x1 < 0 ) return;
		if (x2 >= w) return;
	}
	
	assert(0 <= y1 && y2 < h);
	
	if (transp)
		IMGDRAW_INNER(IMGDRAW_FORM_LINE_T, IMGDRAW_FILL_COLOR)
	else
		IMGDRAW_INNER(IMGDRAW_FORM_LINE, IMGDRAW_FILL_COLOR)
}

void imgdraw_line(Image* dst, ImgPoint p1, ImgPoint p2, ImgColor col, int lw)
{
	int x1,y1, x2,y2, w,h;
	bool transp;
	const bool blend = col.a != 255;
	
	if (abs(p2.x - p1.x) > abs(p2.y - p1.y)) {
		x1=p1.y;  y1=p1.x;  x2=p2.y;  y2=p2.x;  w=dst->h;  h=dst->w;
		transp = true;  //exchange x-y
	} else {
		if (p2.y == p1.y) {
			imgdraw_pixel(dst, p1, col);  //TODO: lw
			return;
		}
		x1=p1.x;  y1=p1.y;  x2=p2.x;  y2=p2.y;  w=dst->w;  h=dst->h;
		transp = false;
	}

	if (y1 > y2) { SWAPT_(int, y1, y2);  SWAPT_(int, x1, x2); }
	float m = (float)(x2 - x1) / (y2 - y1);
	float b = x1 -m*y1;
	
	// Line width
	MAXSET(lw, 1);
	int w0=-lw/2, wE=lw-lw/2;
	
	for (int i=w0; i<wE; ++i)
		imgdraw_line_inner(dst, w, h, y1, y2, m, b+i, col, transp, blend);
}

void imgdraw_polygon_fill(Image* dst, unsigned npts, const ImgPoint* pts,
	ImgColor col)
{
	if (npts < 3) return;
	bool blend = col.a != 255;
	int i1, i2, i1p, i2p, y, ystop;
	float m1=0, m2=0, b1=0, b2=0;

	// Find lower point
	i1 = 0;
	y = pts[0].y;
	for (unsigned ip=1; ip<npts; ++ip) {
		if (pts[ip].y < y) {
			y  = pts[ip].y;
			i1 = ip;
		}
	}
	i2 = i2p = i1p = i1;
	ystop = y;

	//dprintf("i %d y %d p %d,%d\n", i1, y, IMG_POINT_UNPACK(pts[i1]));

	// Draw horizontal lines bounded by the polygon lines
	do {
		if (ystop == pts[i1].y) {
			i1p = i1;
			i1++;
			if (i1 >= (int)npts) i1 = 0;

			int dy = pts[i1].y - pts[i1p].y;
			if (dy == 0) {
				m1 = 0;
				b1 = pts[i1].x;
			} else {
				m1 = (float)(pts[i1].x - pts[i1p].x) / dy;
				b1 = pts[i1p].x - m1 * y;
			}
			//dprintf("i1 %d m1 %f b1 %d\n", i1, m1, b1);
		}
		if (ystop == pts[i2].y && i1 != i2) {
			i2p = i2;
			i2--;
			if (i2 < 0) i2 = (int)npts-1;

			int dy = pts[i2].y - pts[i2p].y;
			if (dy == 0) {
				m2 = 0;
				b2 = pts[i2].x;
			} else {
				m2 = (float)(pts[i2].x - pts[i2p].x) / dy;
				b2 = pts[i2p].x - m2 * y;
			}
			//dprintf("i2 %d m2 %f b2 %d\n", i2, m2, b2);
		}
		//TODO: dy=0 ?
		
		ystop = MIN_(pts[i1].y, pts[i2].y);

		//dprintf("i %d,%d y %d p %d,%d %d,%d\n", i1, i2, ystop,
		//	IMG_POINT_UNPACK(pts[i1]), IMG_POINT_UNPACK(pts[i2]));

		// Clip
		if (ystop >= (int)dst->h) {
			ystop = (int)dst->h - 1;
			i1 = i2 = -1;  //end
		}
		if (y < 0) {
			if (ystop < 0) y = ystop+1;
			else y = 0;
		}

		for (; y<=ystop; ++y) {
			int x1 = b1 + m1 * y +0.5;
			int x2 = b2 + m2 * y +0.5;
			if (x2 < x1) SWAPT_(int, x1, x2);
			// Clip
			if (x1 < 0) x1 = 0;
			if (x2 < 0) continue;
			if (x2 >= (int)dst->w) x2 = (int)dst->w - 1;
			// Draw
			IMGDRAW_INNER(IMGDRAW_FORM_HLINE, IMGDRAW_FILL_COLOR)
		}
	} while (i1 != i2);
}

void imgdraw_circle_border_inner(Image* dst, int x, int y, int r,
	ImgColor col, bool blend)
{
	IMGDRAW_INNER(IMGDRAW_FORM_CIRCLE_1PX, IMGDRAW_FILL_COLOR)
}

void imgdraw_dot_smooth(Image* dst, float cx, float cy, float rx, float ry,
	ImgColor col)
{
	if (rx <= 0 || ry <= 0) return;
	const int x1 = MAX_(0, floor(cx-rx)),
			  y1 = MAX_(0, floor(cy-ry)),
			  x2 = MIN_(dst->w, ceil(cx+rx)),
			  y2 = MIN_(dst->h, ceil(cy+ry));
	const float fx=1.0/(rx*rx), fy=1.0/(ry*ry);
	const bool blend = true;
	IMGDRAW_INNER(IMGDRAW_FORM_DOT_SMOOTH, IMGDRAW_FILL_COLOR_A)
}

void imgdraw_ring_smooth(Image* dst, float cx, float cy, float rad,
	float thi, ImgColor col)
{
	if (rad <= 0 || thi <= 0) return;
	const int x1 = MAX_(0, floor(cx-rad)),
			  y1 = MAX_(0, floor(cy-rad)),
			  x2 = MIN_(dst->w, ceil(cx+rad)+1),
			  y2 = MIN_(dst->h, ceil(cy+rad)+1);
	const bool blend = true;
	IMGDRAW_INNER(IMGDRAW_FORM_RING_SMOOTH, IMGDRAW_FILL_COLOR_A)
}

void imgdraw_blit_inner(Image* dst,
	const uint8_t* sdata, unsigned ssy, ImgFormat sfmt,
	bool blend)
{
	unsigned ssx=0;

	switch (sfmt) {
	case IMG_FORMAT_NULL:
		break;
	case IMG_FORMAT_GRAY:
		ssx=1;
		//TODO
		break;
	case IMG_FORMAT_RGB:
		ssx=3;
		switch (dst->format) {
		case IMG_FORMAT_NULL:
			break;
		case IMG_FORMAT_GRAY:
			IMGDRAW_FORM_RECT_DS(1)
				imgdraw_pixel_set_gray_a(drow, 0,0,0, *srow);
			IMGDRAW_FORM_END
			break;
		case IMG_FORMAT_RGB:
			IMGDRAW_FORM_RECT_DS(3)
				imgdraw_pixel_set_rgb(drow, srow[0], srow[1], srow[2], 255);
			IMGDRAW_FORM_END
			break;
		case IMG_FORMAT_RGBA:
			IMGDRAW_FORM_RECT_DS(4)
				imgdraw_pixel_set_rgba(drow, srow[0], srow[1], srow[2], 255);
			IMGDRAW_FORM_END
			break;
		}
		break;
	case IMG_FORMAT_RGBA:
		ssx=4;
		if (blend) {
			switch (dst->format) {
			case IMG_FORMAT_NULL:
				break;
			case IMG_FORMAT_GRAY:
				IMGDRAW_FORM_RECT_DS(1)
					imgdraw_pixel_blend_gray(drow, 0,0,0, srow[3]);
				IMGDRAW_FORM_END
				break;
			case IMG_FORMAT_RGB:
				IMGDRAW_FORM_RECT_DS(3)
					imgdraw_pixel_blend_rgb(drow,
						srow[0], srow[1], srow[2], srow[3]);
				IMGDRAW_FORM_END
				break;
			case IMG_FORMAT_RGBA:
				IMGDRAW_FORM_RECT_DS(4)
					imgdraw_pixel_blend_rgba(drow,
						srow[0], srow[1], srow[2], srow[3]);
				IMGDRAW_FORM_END
				break;
			}
		} else {
			switch (dst->format) {
			case IMG_FORMAT_NULL:
				break;
			case IMG_FORMAT_GRAY:
				IMGDRAW_FORM_RECT_DS(1)
					imgdraw_pixel_set_gray_a(drow, 0,0,0, srow[3]);
				IMGDRAW_FORM_END
				break;
			case IMG_FORMAT_RGB:
				IMGDRAW_FORM_RECT_DS(3)
					imgdraw_pixel_set_rgb(drow,
						srow[0], srow[1], srow[2], srow[3]);
				IMGDRAW_FORM_END
				break;
			case IMG_FORMAT_RGBA:
				IMGDRAW_FORM_RECT_DS(4)
					imgdraw_pixel_set_rgba(drow,
						srow[0], srow[1], srow[2], srow[3]);
				IMGDRAW_FORM_END
				break;
			}
		}
		break;
	}
}

void imgdraw_alpha_mask_inner(Image* dst,
		const uint8_t* sdata, unsigned ssx, unsigned ssy, //alpha mask
		ImgColor col, bool fast)
{
	if (fast) {
		switch (dst->format) {
		case IMG_FORMAT_NULL:
			break;
		case IMG_FORMAT_GRAY:
			IMGDRAW_FORM_RECT_DS(1)
				imgdraw_pixel_set_gray_a(drow, 0,0,0, *srow);
			IMGDRAW_FORM_END
			break;
		case IMG_FORMAT_RGB:
			IMGDRAW_FORM_RECT_DS(3)
				if (*srow > 127)
					imgdraw_pixel_set_rgb(drow, col.r, col.g, col.b, 255);
			IMGDRAW_FORM_END
			break;
		case IMG_FORMAT_RGBA:
			IMGDRAW_FORM_RECT_DS(4)
				imgdraw_pixel_set_rgba(drow, col.r, col.g, col.b, *srow);
			IMGDRAW_FORM_END
			break;
		}
	} else {
		switch (dst->format) {
		case IMG_FORMAT_NULL:
			break;
		case IMG_FORMAT_GRAY:
			IMGDRAW_FORM_RECT_DS(1)
				int a = (col.a * *srow + 255)>>8;
				imgdraw_pixel_blend_gray(drow, 0,0,0,a);
			IMGDRAW_FORM_END
			break;
		case IMG_FORMAT_RGB:
			IMGDRAW_FORM_RECT_DS(3)
				int a = (col.a * *srow + 255)>>8;
				imgdraw_pixel_blend_rgb(drow, col.r, col.g, col.b, a);
			IMGDRAW_FORM_END
			break;
		case IMG_FORMAT_RGBA:
			IMGDRAW_FORM_RECT_DS(4)
				int a = (col.a * *srow + 255)>>8;
				imgdraw_pixel_blend_rgba(drow, col.r, col.g, col.b, a);
			IMGDRAW_FORM_END
			break;
		}
	}
}
