/* Copyright 2025-2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: MIT
 *
 * Graphical user interface using SDL.
 */
#pragma once
#include "game.h"
#include "ccommon/vector.h"
#include "ccommon/image.h"
#include "ccommon/textrender.h"
#include "ccommon/richtext.h"
#include "ccommon/im_gui.h"
#include "ccommon/sound.h"
#include "ccommon/audio_sdl.h"
#include <SDL2/SDL.h>

typedef struct GameSprite {
	Image img;
} GameSprite;

void game_sprite_free(GameSprite* S);

typedef struct GameSound {
	Sound snd;
	//SDL_AudioSpec spec;
	//Uint8 * audio_buf;
	//Uint32 audio_len;
} GameSound;

void game_sound_free(GameSound* S);

//int game_sound_load_file(GameSound* S, const char* path);

typedef struct {
	int x, y;  // Position on screen
	int n;  // Current value
	int d;  // Delta value (recent variation)
	double t;  // Time of last variation
} GameUiNumber;

typedef struct {
	int x, y;  // Position on screen
	int n;  // Current value
	int d;  // Delta value (recent variation)
	double t;  // Time of last variation
} GameUiAlert;

enum GameUiConfigFlag {
	GAME_UI_CF_FULLSCREEN = 1,
	GAME_UI_CF_MUTE = 2,
};

enum {
	GAME_UI_DBG_LVL_COUNT = 4,
};

enum {
	GAME_UI_RESULT_DONE = 0x0001,
	GAME_UI_RESULT_QUIT = 0x1001,
};

typedef enum GameUiState {
	GAME_UI_STATE_NULL		= 0,
	GAME_UI_STATE_QUIT		= 1,
	GAME_UI_STATE_SETUP		= 2,  // Setup menu
	GAME_UI_STATE_GAME		= 3,  // Game
} GameUiState;

typedef struct {
	ImGuiSdl2 win;
	ImGui gui;
	AudioSdl aud;

	struct {
		double turn, last, events, draw, present;
	} times;

	// State
	GameUiState state;
	int trade_res;
	DynStr tooltip;
	
	struct {
		GameUiNumber round, turn, dice_total;
		struct {
			GameUiNumber vp, lroad, army, res_total, devcard, res[GAME_RES__COUNT];
		} pls[GAME_PLAYER_COUNT];
	} num;

	struct {
		GameUiAlert * tiles;  //vector
		GameUiAlert * nodes;  //vector
		GameUiAlert * edges;  //vector
		GameUiAlert thief;
	} alert;
	
	// Resources
	RichTextEngine rte;
	TextRender *font_normal, *font_small, *font_bold;
	
	GameSprite * sprites;  //vector
	GameSound * sounds;  //vector

	struct GameUiImageIndex {
		// Positions at .sprites
		int res[GAME_RES__COUNT], res_icon[GAME_RES__COUNT];
		int road[GAME_PLAYER_COUNT];
		int settle[GAME_SETTLE_LEVEL_COUNT][GAME_PLAYER_COUNT];
		int thief, thief_hl;
		int ship, harbor;
		int circle, circle_hl;
		int icon_res, icon_card, icon_dice, icon_vp, icon_lroad, icon_army;
		int icon_pl[GAME_PLAYER_COUNT];
		int card_knight;
	} ii;

	struct GameUiSoundIndex {
		// Positions at .sounds
		int click, error, game_start, turn_start, build_edge,
			build_settle[GAME_SETTLE_LEVEL_COUNT], devcard_use, thief_move;
	} si;

	struct {
		int flags;
		int dbg_lvl;
		double ai_end_turn_delay;
	} c;
} GameUi;

void game_ui_free(GameUi* S);

// Create a window and initialize video and audio.
int game_ui_create(GameUi* S);

// Finish resources load.
// You should call this after you have loaded fonts, images, and sounds,
// and after game_ui_create.
void game_ui_resources_finish(GameUi* S);

int game_ui_menu_loop_step(GameUi* S, Game* G);

int game_ui_game_loop_step(GameUi* S, Game* G);

static inline
int game_ui_menu_loop(GameUi* S, Game* G)
{
	int r;
	while ((r = game_ui_menu_loop_step(S, G)) == 0) ;
	return r;
}

static inline
int game_ui_game_loop(GameUi* S, Game* G)
{
	int r;
	while ((r = game_ui_game_loop_step(S, G)) == 0) ;
	return r;
}

// Data

extern const ImgColor col_player[GAME_PLAYER_COUNT+1];
