/* Copyright 2025-2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: MIT
 */
#include "ui_sdl.h"
#include "ai.h"
#include <stdbool.h>
#include <stdint.h>
#include "ccommon/ccommon.h"
#include "ccommon/timing.h"
#include "ccommon/vector.h"
#include "ccommon/logging.h"
#include "ccommon/image.h"
#include "ccommon/image_draw.h"

static const ImgColor col_dbg_txt = {0,0,0,255};
//static const ImgColor col_none = {0,0,0,0};
static const ImgColor col_text = {0,0,0,255};
static const ImgColor col_text_grey = {63,63,63,255};
static const ImgColor col_text_red = {255,31,31,255};
static const ImgColor col_text_blue = {31,31,255,255};
static const ImgColor col_ui_sep = {31,31,31,255};
static const ImgColor col_ui_fill = {191,191,127,255};
static const ImgColor col_ui_title = {127,127,63,255};
static const ImgColor col_sel = {0,200,0,127};
static const ImgColor col_shine = {255,255,93,255};

static const ImgColor col_bkgr = { 63,127,255,255};

const ImgColor col_player[GAME_PLAYER_COUNT+1] = {
	{ 63, 63,255,255},
	{255, 63, 63,255},
	{ 63,223, 63,255},
	{223,223, 63,255},
	{255,255,255,255},
};

static inline
ImgColor col_player_get(int pl) {
	pl = pl < 0 ? GAME_PLAYER_COUNT : pl % GAME_PLAYER_COUNT;
	return col_player[pl];
}

static const ImgColor col_res[GAME_RES__COUNT+1] = {
	{127,223,127,255},
	{191,191, 0,255},
	{ 63,127, 63,255},
	{127, 95, 95,255},
	{200,200,223,255},
	{255,255,200,255},
};

static inline
ImgColor col_res_get(int res) {
	res = res < 0 ? GAME_RES__COUNT : res;
	return col_res[res];
}

#define HH  0.86602540378443864676
static const float xnode[6] = {-1, -0.5, +0.5, +1, +0.5, -0.5};
static const float ynode[6] = {0, HH, HH, 0, -HH, -HH};

enum {
	IMGUI_STYLE_BUTTON_PLAYER     = IMGUI_STYLE__COUNT,
	IMGUI_STYLE_BUTTON_PLAYER_HL  = IMGUI_STYLE_BUTTON_PLAYER + GAME_PLAYER_COUNT,
	IMGUI_STYLE__LAST = IMGUI_STYLE_BUTTON_PLAYER_HL + GAME_PLAYER_COUNT,
};

void game_sprite_free(GameSprite* S)
{
	img_free(&S->img);
}

void game_sound_free(GameSound* S)
{
	snd_free(&S->snd);
	//if (S->audio_buf) {
	//	SDL_FreeWAV(S->audio_buf);
	//	S->audio_buf = NULL;
	//}
	//S->audio_len = 0;
}

//int game_sound_load_file(GameSound* S, const char* path)
//{
//	if (S->audio_buf) game_sound_free(S);
//	if (SDL_LoadWAV(path, &S->spec, &S->audio_buf, &S->audio_len) == NULL) {
//		log_error("Could not load sound '%s': %s", path, SDL_GetError());
//		return -1;
//	}
//	return 1;
//}

void game_ui_free(GameUi* S)
{
	vec_free(S->alert.nodes);
	vec_free(S->alert.edges);
	vec_free(S->alert.tiles);

	vec_for(S->sprites, i, 0) game_sprite_free(&S->sprites[i]);
	vec_free(S->sprites);

	textrender_destroy(S->font_bold);
	textrender_destroy(S->font_small);
	textrender_destroy(S->font_normal);
	
	richtext_free(&S->rte);

	audio_sdl_free(&S->aud);
	
	imgui_sdl2_free(&S->win);
	imgui_free(&S->gui);

    SDL_Quit();
}

int game_ui_create(GameUi* S)
{
	int R=1;

	if (SDL_Init(0))
		ERROR_LOG(-1, "Unable to initialize SDL: %s", SDL_GetError());
	
	// Video init
	S->win.c.win_sdl_flags |= SDL_WINDOW_RESIZABLE;
	if (S->c.flags & GAME_UI_CF_FULLSCREEN)
		S->win.c.win_sdl_flags |= SDL_WINDOW_FULLSCREEN_DESKTOP;

	IFFALSESET(S->win.c.win_title, "Catan Game SDL");
	IFFALSESET(S->win.c.win_w, 920);
	IFFALSESET(S->win.c.win_h, 680);
	
	S->win.c.wflags |= IMGUI_WF_NAV_KEYBOARD;
	S->win.c.wflags |= IMGUI_WF_NAV_GAMEPAD;

	TRY( imgui_sdl2_create(&S->win, &S->gui) );

	// Audio Init
	TRY( audio_sdl_init(&S->aud) );

	// Init state
	MEM_ZERO(S->times);
	S->trade_res = -1;
	dstr_resize(S->tooltip, 0);

end:
	return R;
}

static const ImgColor col_ph = {255,0,255,255};

static
int sprite_copy(GameUi* S, unsigned s)
{
	GameSprite sprite={0};
	img_copy(&sprite.img, &S->sprites[s].img);

	int idx = vec_count(S->sprites);
	vec_push(S->sprites, sprite);
	return idx;
}

static
void sprite_player_color(GameUi* S, unsigned s, unsigned p)
{
	const ImgColorHSV hsv_ph = img_color_rgb2hsv(col_ph);
	const ImgColorHSV hsv_pl = img_color_rgb2hsv(col_player[p]);
	img_hue_replace(&S->sprites[s].img, hsv_ph.h, hsv_pl.h);
}

void game_ui_resources_finish(GameUi* S)
{
	// Replace placeholder color by each player's color
	for (unsigned p=1; p<GAME_PLAYER_COUNT; ++p) {
		S->ii.road[p] = sprite_copy(S, S->ii.road[0]);
		S->ii.settle[0][p] = sprite_copy(S, S->ii.settle[0][0]);
		S->ii.settle[1][p] = sprite_copy(S, S->ii.settle[1][0]);
		S->ii.icon_pl[p] = sprite_copy(S, S->ii.icon_pl[0]);
	}
	for (unsigned p=0; p<GAME_PLAYER_COUNT; ++p) {
		sprite_player_color(S, S->ii.road[p], p);
		sprite_player_color(S, S->ii.settle[0][p], p);
		sprite_player_color(S, S->ii.settle[1][p], p);
		sprite_player_color(S, S->ii.icon_pl[p], p);

		Image * img = &S->sprites[ S->ii.icon_pl[p] ].img;
		char buffer[32];
		sprintf(buffer, "%d", p+1);
		textrender_render(S->font_small, img, NULL, buffer, NULL,
			(ImgColor){255,255,255,255}, TR_RF_ALIGN_CENTER);
	}

	// Convert sounds to hardware format
	assert( S->aud.dev );
	vec_for(S->sounds, i, 0) {
		audio_sdl_sound_convert(&S->aud, &S->sounds[i].snd, &S->sounds[i].snd);
	}

	// Rich text engine setup
	assert( S->font_normal );
	S->rte.c.ec = '^';
	richtext_default_colors_set(&S->rte);
	richtext_color_set(&S->rte, 'B', col_text);
	richtext_color_set(&S->rte, 'r', col_text_red);
	richtext_color_set(&S->rte, 'b', col_text_blue);
	richtext_font_set(&S->rte, 'n', S->font_normal);
	richtext_font_set(&S->rte, 's', S->font_small);
	richtext_font_set(&S->rte, 'b', S->font_bold);
	vec_for(S->sprites, i, 0) {
		richtext_image_set(&S->rte, ' '+i, &S->sprites[i].img);
	}

	S->gui.rte = &S->rte;

	// GUI styles
	imgui_default_styles_set(&S->gui);

	//Allocate all new styles so that returned pointers remain valid
	imgui_style_get(&S->gui, IMGUI_STYLE__LAST);

	ImGuiStyle *sty, *sty_b, *sty_bh;
	sty_b  = sty = imgui_style_get(&S->gui, IMGUI_STYLE_BUTTON);
	imgui_style_button_colors_set(sty, col_ui_fill);
	
	sty_bh = sty = imgui_style_get(&S->gui, IMGUI_STYLE_BUTTON_HL);
	ImgColor col_hl;
	IMG_COLOR_TRANSF_RGB_HSV_RGB(col_hl, col_ui_fill, v=v*2);
	imgui_style_button_colors_set(sty, col_hl);

	for (int p=0; p<GAME_PLAYER_COUNT; ++p) {
		ImgColor col_hl, col = col_player_get(p);
		IMG_COLOR_TRANSF_RGB_HSV_RGB(col_hl, col, v=v*3/4);

		sty = imgui_style_get(&S->gui, IMGUI_STYLE_BUTTON_PLAYER + p);
		*sty = *sty_b;
		sty->sty_hl = IMGUI_STYLE_BUTTON_PLAYER_HL + p;
		imgui_style_button_colors_set(sty, col);

		sty = imgui_style_get(&S->gui, IMGUI_STYLE_BUTTON_PLAYER_HL + p);
		*sty = *sty_bh;
		imgui_style_button_colors_set(sty, col_hl);
	}
}

static
void game_ui_sound_play(GameUi* S, int idx)
{
	if (S->c.flags & GAME_UI_CF_MUTE) return;
	assert( vec_idx_check(S->sounds, idx) );
	audio_sdl_sound_queue(&S->aud, &S->sounds[idx].snd, 0);
}

static
void game_ui_game_state_clear(GameUi* S, Game* G)
{
	vec_resize_zero(S->alert.tiles, vec_count(G->tiles));
	vec_resize_zero(S->alert.nodes, vec_count(G->nodes));
	vec_resize_zero(S->alert.edges, vec_count(G->edges));
	MEM_ZERO(S->num);
}

static
void game_ui_turn_state_clear(GameUi* S, Game* G)
{
	vec_for(S->alert.tiles, t, 0)
		S->alert.tiles[t].n = 0;
}

/* Numbers */

enum {
	GAME_UI_NDF_INTERACTION = 1,
	GAME_UI_NDF_WARNING = 2,
	GAME_UI_NDF_BOLD = 4,
};

static
void game_ui_num_draw(GameUi* S, GameUiNumber* N, int x, int y, int n, int flags)
{
	if (n != N->n) {
		if (N->t) N->d += (n - N->n);
		N->t = S->times.last;
		N->n = n;
	}
	else if (!N->t) N->t = S->times.last;

	TextRender * font = flags & GAME_UI_NDF_BOLD ? S->font_bold : S->font_normal;
	int font_h  = font->metrics.lineskip;
	int font_em = font->metrics.m_width;

	DynStr tmp_s = dstr_stack(32);
	
	ImgColor col_n = col_text;
	if (flags & GAME_UI_NDF_INTERACTION) col_n = col_text_blue;
	else if (flags & GAME_UI_NDF_WARNING) col_n = col_text_red;
	dstr_printf(tmp_s, "%d", n);
	textrender_render(font, &S->win.fb,
		&(ImgRect){x-16, y-font_h/2, 32, font_h},
		tmp_s, NULL, col_n, TR_RF_ALIGN_CENTER);

	if (N->d) {
		double dt_max = 2;
		double f = (S->times.last - N->t) / dt_max;
		if (f > 1) {
			N->d = 0;
		} else {
			ImgColor col_d = (N->d > 0) ? col_text_blue : col_text_red;
			col_d.a = 55+200*(1-f);
			dstr_printf(tmp_s, "%+d", N->d);
			textrender_render(font, &S->win.fb,
				&(ImgRect){x-16+font_em*fabs(f-0.5), y-font_h*(3/2+f), 32, font_h},
				tmp_s, NULL, col_d, TR_RF_ALIGN_CENTER);
		}
	}

}

enum {
	GAME_UI_ADF_UPDATE_ONLY = 1,
};

static
void game_ui_alert_draw(GameUi* S, GameUiAlert* N, int x, int y, int n, int pl,
	int flags)
{
	if (n != N->n) {
		if (N->t) N->d += (n - N->n);
		N->t = S->times.last;
		N->n = n;
	}
	else if (!N->t) N->t = S->times.last;

	if (flags & GAME_UI_ADF_UPDATE_ONLY) return;

	if (N->d) {
		double dt_max = 2;
		double f = (S->times.last - N->t) / dt_max;
		if (f > 1) {
			N->d = 0;
		} else {
			ImgColor col_d = (pl >= 0) ? col_player_get(pl)
				: (ImgColor){255,255,200,255};
			col_d.a = 55+200*(1-f);
			imgdraw_ring_smooth(&S->win.fb, x, y, 16+f*32, 16, col_d);
		}
	}
}

enum {
	GAME_UI_EVENT_QUIT = 0x1000,
};

#define KMOD_CHECK(VAR, FLAGS) \
	((VAR) && ((VAR) & (FLAGS)) == (VAR))

static
int game_ui_event_process_common(GameUi* S, const SDL_Event* ev)
{
	int R=0;

	R = imgui_sdl2_event_process(&S->win, ev);
	if (R) return R;

	switch (ev->type) {
	case SDL_KEYDOWN:
		switch (ev->key.keysym.sym) {
#ifndef GAME_UI_NO_QUIT
		case 'q':
			if (KMOD_CHECK(ev->key.keysym.mod, KMOD_CTRL))
				R = GAME_UI_EVENT_QUIT;
			break;
		case SDLK_ESCAPE:
			R = GAME_UI_EVENT_QUIT;
			break;
#endif
		case 'm':
			ccFLAG_TOGGLE(S->c.flags, GAME_UI_CF_MUTE);
			break;
		case SDLK_F12:  // Debug level
			S->c.dbg_lvl = (S->c.dbg_lvl + 1) % GAME_UI_DBG_LVL_COUNT;
			R=1;
			break;
		case SDLK_F11:  // Fullscreen toggle
			if (S->c.flags & GAME_UI_CF_FULLSCREEN) {
				S->c.flags &= ~GAME_UI_CF_FULLSCREEN;
				SDL_SetWindowFullscreen(S->win.window, 0);
			}
			else {
				S->c.flags |= GAME_UI_CF_FULLSCREEN;
				SDL_SetWindowFullscreen(S->win.window,
					SDL_WINDOW_FULLSCREEN_DESKTOP);
			}
			R=1;
			break;
		}
		break;

	case SDL_QUIT:
		R = GAME_UI_EVENT_QUIT;
		break;
	}

	return R;
}

static
void game_ui_draw_common(GameUi* S)
{
	DynStr tmp_s = dstr_stack(128);
	Image * frame = &S->win.fb;

	// ** Debug info

	// Interaction areas
	if (S->c.dbg_lvl == 3) {
		imgui_draw_debug_regions(&S->gui, &S->win.fb, col_dbg_txt);
	}

	// Times
	if (S->c.dbg_lvl > 0) {
		dstr_printf(tmp_s, "%05.0f %05.0f %05.0f",
			S->times.events*1e6, S->times.draw*1e6, S->times.present*1e6);
		textrender_render(S->font_normal, frame, NULL, tmp_s, NULL,
			col_dbg_txt, TR_RF_FAST_BLIT | TR_RF_ALIGN_BOTTOM |
				TR_RF_ALIGN_HCENTER);
	}
}

static inline
void game_ui_time_measure(GameUi* S, double* dst)
{
	double t = timing_time();
	*dst += (t - S->times.last - *dst) * 0.1;
	S->times.last = t;
}

/* Game set-up menu */

static
void game_ui_draw_menu_background(GameUi* S)
{
	Image * fb = &S->win.fb;
	int fw=fb->w, fh=fb->h;
	
	img_fill(fb, col_bkgr);

	int step = 64;

	// Moving view of an infinite field with sprites
	double t = S->times.last;
	int x0 = fabs(fmod(t,20)/10 - 1) * 200;
	int y0 = fabs(fmod(t+5,20)/10 - 1) * 200;
	int ix0 = x0 / step - 1;
	int iy0 = y0 / step - 1;
	int ixE = ix0 + (fw + step - 1) / step + 2;
	int iyE = iy0 + (fh + step - 1) / step + 2;

	for (int iy=iy0; iy<iyE; ++iy) {
		for (int ix=ix0; ix<ixE; ++ix) {
			int res = (ix*3+iy) % GAME_RES__COUNT; 
			if (res < 0) res += GAME_RES__COUNT;
			const Image * img = &S->sprites[ S->ii.res[res] ].img;
			int x = -x0 + ix * step;
			int y = -y0 + iy * step + (ix%2) * step/2;
			imgdraw_blit(fb, (ImgPoint){x, y}, img, true);
		}
	}
}

int game_ui_menu_loop_step(GameUi* S, Game* G)
{
	if (!S->win.window) return -1;
	int R=0, r;

	if (S->state != GAME_UI_STATE_SETUP) {
		S->state = GAME_UI_STATE_SETUP;
		imgui_reset(&S->gui);
	}

	// Process events
	SDL_Event event;
	while (SDL_PollEvent(&event)) {
		r = game_ui_event_process_common(S, &event);
		if (r == GAME_UI_EVENT_QUIT)
			RETURN( GAME_UI_RESULT_QUIT );
	}
	game_ui_time_measure(S, &S->times.events);

	// Draw
	imgui_frame_begin(&S->gui);
	game_ui_draw_menu_background(S);

	// Some help at the top
	richtext_render(&S->rte, &S->win.fb, NULL,
		strsl_fromz(" ^{^fbF11^} Fullscreen   ^{^fbm^} Mute"));

	// "Dialog" box
	//TODO: move to a function in im_gui
	ImgRect rect, rect_dlg = { 0, 0, 540, 170};
	rect_dlg.x = (S->win.fb.w - 110*5) / 2;
	rect_dlg.y = (S->win.fb.h - 20*5) / 2;
	imgdraw_rect_fill(&S->win.fb, rect_dlg, (ImgColor){0,0,0,127});
	imgdraw_rect_border(&S->win.fb, rect_dlg, (ImgColor){127,127,127,255}, 1);
	S->gui.lm.cur.x = rect_dlg.x + S->gui.lm.m_sep;
	S->gui.lm.cur.y = rect_dlg.y + S->gui.lm.m_sep;

	// Players config
	imgui_label(&S->gui, "^fb^cwPlayers:", NULL, 0);
	rect = (ImgRect){.w=100};
	for (int p=0; p<GAME_PLAYER_COUNT; ++p) {
		const char * type_str="";
		if (G->pls[p].type == GAME_PLAYER_TYPE_NONE) type_str = "None";
		else if (G->pls[p].type == GAME_PLAYER_TYPE_EXTERN) type_str = "Human";
		else if (G->pls[p].type == GAME_PLAYER_TYPE_AI) type_str = "AI";
		
		int style=0;
		if (G->pls[p].type) style = IMGUI_STYLE_BUTTON_PLAYER + p;
		
		r = imgui_button(&S->gui, type_str, &rect, style);
		if (r & IMGUI_STATE_PRESS) {
			G->pls[p].type = (G->pls[p].type + 1) % GAME_PLAYER_TYPE__COUNT;
		}

		rect.x += rect.w + 10;
	}

	// Options
	int sel = G->cfg.flags & GAME_CF_AUTO_SETUP;
	imgui_label(&S->gui, "^fb^cwAuto setup:", NULL, 0);
	imgui_select_buttons_va(&S->gui, &sel, 2, " off ", " on ");
	ccFLAG_SET(G->cfg.flags, GAME_CF_AUTO_SETUP, sel);
	
	sel = G->cfg.board_radius - 1;
	imgui_label(&S->gui, "^fb^cwBoard size:", NULL, 0);
	imgui_select_buttons_va(&S->gui, &sel, 3, " 1 ", " 2 ", " 3 ");
	G->cfg.board_radius = sel + 1;

	//Victory points

	//TODO: seed with a textedit

	// End
	S->gui.lm.cur.y += S->gui.lm.m_sep;  //TODO: to function
	
	rect = (ImgRect){ 0, S->gui.lm.cur.y, 100, 0 };
#ifndef GAME_UI_NO_QUIT
	r = imgui_button(&S->gui, "Quit", &rect, 0);
	if (r & IMGUI_STATE_PRESS)
		RETURN( GAME_UI_RESULT_QUIT );
#endif
	
	rect.x = rect_dlg.x + rect_dlg.w - rect.w - S->gui.lm.m_sep;
	rect.h = 0;
	r = imgui_button(&S->gui, "^fbStart", &rect, 0);
	if (r & IMGUI_STATE_PRESS) {
		game_ui_sound_play(S, S->si.game_start);
		RETURN( GAME_UI_RESULT_DONE );
	}

	game_ui_draw_common(S);
	game_ui_time_measure(S, &S->times.draw);

	if (S->gui.region_press >= 0) {  // On Click
		game_ui_sound_play(S, S->si.click);
	}

	// Present
	imgui_sdl2_present(&S->win);
	game_ui_time_measure(S, &S->times.present);

end:
	return R;
}

/* Game GUI */

static inline
void game_res_to_dstr_a(GameUi*S, DynStr* out, const int res[GAME_RES__COUNT])
{
	char rtcmd[4] = "^i";
	for (int r=0; r<GAME_RES__COUNT; ++r) {
		rtcmd[2] = ' ' + S->ii.res_icon[r];
		for (int i=0; i<res[r]; ++i) {
			dstr_append(*out, 3, rtcmd);
		}
	}
}

static inline
void game_res_to_dstr(GameUi*S, DynStr* out, const int res[GAME_RES__COUNT])
{
	dstr_resize(*out, 0);
	game_res_to_dstr_a(S, out, res);
}

static
int end_turn(GameUi* S, Game* G)
{
	int r = game_player_action(G, G->pl_curr, GAME_PA_END_TURN, 0, 0);
	if (r<0) return r;
	
	game_ui_turn_state_clear(S, G);
	
	if (G->pls[G->pl_curr].type == GAME_PLAYER_TYPE_AI) {  //TODO: move to game
		game_player_action_ai(G, G->pl_curr);
	}
	else {
		game_ui_sound_play(S, S->si.turn_start);
	}
	
	S->times.turn = S->times.last;
	
	return 1;
}

enum {
	GAME_UI_ACTION_END_TURN,
	GAME_UI_ACTION_TILE,
	GAME_UI_ACTION_EDGE,
	GAME_UI_ACTION_NODE,
	GAME_UI_ACTION_RES,
	GAME_UI_ACTION_DEVCARD_BUY,
	GAME_UI_ACTION_DEVCARD_USE,

	GAME_UI_ACTION_NEW_GAME,
	GAME_UI_ACTION_QUIT,
};

static
int game_ui_action(GameUi* S, Game* G, int action, int param)
{
	int R=0;
	int sound=S->si.click;

	switch (action) {
	case GAME_UI_ACTION_END_TURN:
		R = end_turn(S, G);
		break;
	
	case GAME_UI_ACTION_TILE:
		if (G->state == GAME_STATE_THIEF_MOVE) {
			R = game_player_action(G, G->pl_curr, GAME_PA_THIEF_MOVE,
				param, 0);
			sound = S->si.thief_move;
		}
		break;
	
	case GAME_UI_ACTION_EDGE:
		R = game_player_action(G, G->pl_curr, GAME_PA_BUILD_EDGE,
			param, 0);
		sound = S->si.build_edge;
		break;
	
	case GAME_UI_ACTION_NODE:
		R = game_player_action(G, G->pl_curr, GAME_PA_BUILD_NODE,
			param, 0);
		sound = S->si.build_settle[0];
		break;
	
	case GAME_UI_ACTION_DEVCARD_BUY:
		R = game_player_action(G, G->pl_curr, GAME_PA_DEVCARD_BUY,
			0, 0);
		break;
	
	case GAME_UI_ACTION_DEVCARD_USE:
		R = game_player_action(G, G->pl_curr, GAME_PA_DEVCARD_USE,
			param, 0);
		sound = S->si.devcard_use;
		break;
	
	case GAME_UI_ACTION_RES:
		if (S->trade_res == -1) {
			S->trade_res = param;
		} else {
			R = game_player_action(G, G->pl_curr, GAME_PA_TRADE_RES,
				S->trade_res, param);
			S->trade_res = -1;
		}
		break;
	}

	if (R > 0) {
		game_ui_sound_play(S, sound);
	} else if (R < 0) {
		game_ui_sound_play(S, S->si.error);
	}

	return R;
}

typedef struct {
	GameUi* S;
	Game* G;
	Image* frame;
	int scale, x0, y0;
} DrawCtx;
#define DRAW_CTX_UNPACK \
	GameUi* S = ((DrawCtx*)user)->S; ccUNUSED(S); \
	Game* G = ((DrawCtx*)user)->G; ccUNUSED(G); \
	Image* frame = ((DrawCtx*)user)->frame; ccUNUSED(frame); \
	int scale = ((DrawCtx*)user)->scale; ccUNUSED(scale); \
	int x0 = ((DrawCtx*)user)->x0; ccUNUSED(x0); \
	int y0 = ((DrawCtx*)user)->y0; ccUNUSED(y0);

static
int tile_draw(void* user, int id, float xf, float yf)
{
	DRAW_CTX_UNPACK
	char buffer[32];
	xf = xf * scale + x0 +0.5;
	yf = yf * scale + y0 +0.5;

	int res = G->tiles[id].res;
	int num = G->tiles[id].num;
	ImgColor col =  col_res_get(res);

	// Draw hexagon
	ImgPoint pts[6];
	for (int i=0; i<6; ++i) {
		pts[i].x = xf + xnode[i] * scale;
		pts[i].y = yf + ynode[i] * scale;
	}
	imgdraw_polygon_fill(frame, 6, pts, col);

	// Resource
	if (res >= 0) {
		const Image * img = &S->sprites[ S->ii.res[res] ].img;
		imgdraw_blit(frame, (ImgPoint){xf,yf}, img, true);
		//imgdraw_blit(frame, (ImgPoint){xf-img->w,yf}, img, true);
		//imgdraw_blit(frame, (ImgPoint){xf-img->w/2,yf-img->h}, img, true);
	}

	// Draw lot number
	if (num > 0) {
		int ani = (num == G->lot_curr) ? S->ii.circle_hl : S->ii.circle;
		const Image * img = &S->sprites[ani].img;
		imgdraw_blit(frame, (ImgPoint){xf-img->w, yf-img->h}, img, true);

		ImgColor col = col_text;
		if (G->rule.num_prob[num] > 0.125) col = (ImgColor){127,0,0,255};
		sprintf(buffer, "%d", num);
		textrender_render(S->font_bold, &S->win.fb,
			&(ImgRect){xf-img->w, yf-img->h, img->w, img->h},
			buffer, NULL,
			col, TR_RF_ALIGN_VCENTER | TR_RF_ALIGN_HCENTER);
		
		// Probability
		sprintf(buffer, "%d%%", (int)(G->rule.num_prob[num]*100+0.5));
		textrender_render(S->font_small, &S->win.fb,
			&(ImgRect){xf-img->w, yf-img->h, img->w, img->h},
			buffer, NULL,
			col_text_grey, TR_RF_ALIGN_BOTTOM | TR_RF_ALIGN_HCENTER);
		
		game_ui_alert_draw(S, &S->alert.tiles[id], xf-img->w/2, yf-img->h/2,
			(num == G->lot_curr), -1, 0);
			//(num != G->lot_curr) ? GAME_UI_ADF_UPDATE_ONLY : 0);
	}
	
	// Thief
	if (id == G->thief_tile) {
		int ani = (G->state == GAME_STATE_THIEF_MOVE)
			? S->ii.thief_hl : S->ii.thief;
		const Image * img = &S->sprites[ani].img;
		imgdraw_blit(frame, (ImgPoint){xf-img->w/2, yf-img->h/2}, img, true);

		game_ui_alert_draw(S, &S->alert.thief, xf, yf,
			G->thief_tile, G->pl_curr, 0);
	}

	// Interaction
	if (G->state == GAME_STATE_THIEF_MOVE && id != G->thief_tile) {
		int r = imgui_circle_interact(&S->gui, (ImGuiPoint){xf, yf}, HH*scale);
		if (r & IMGUI_STATE_PRESS) {
			game_ui_action(S, G, GAME_UI_ACTION_TILE, id);
		}
		if (r & IMGUI_STATE_HOVER) {
			// Selection highlight
			ImgPoint pts[6];
			for (int i=0; i<6; ++i) {
				pts[i].x = xf + xnode[i]*scale*0.9;
				pts[i].y = yf + ynode[i]*scale*0.9;
			}
			imgdraw_polygon_border(frame, 6, pts, col_sel, 5);
			//imgdraw_polygon_fill(frame, 6, pts, col_sel);
		}
	}
	
	// ** Debug info
	if (S->c.dbg_lvl != 2) return 0;

	sprintf(buffer, "t%d", id);
	textrender_render(S->font_small, &S->win.fb,
		&(ImgRect){xf-scale, yf-scale, scale*2, scale*2},
		buffer, NULL,
		col_dbg_txt, TR_RF_FAST_BLIT | TR_RF_ALIGN_VCENTER | TR_RF_ALIGN_HCENTER);
	
	return 0;
}

static
int edge_draw(void* user, int id, float xf1, float yf1, float xf2, float yf2)
{
	DRAW_CTX_UNPACK
	int x1 = xf1 * scale + x0;
	int y1 = yf1 * scale + y0;
	int x2 = xf2 * scale + x0;
	int y2 = yf2 * scale + y0;
	int x=(x1+x2)/2, y=(y1+y2)/2;

	int pl = G->edges[id].pl;
	ImgColor col = col_player_get(pl);
	int lw = pl >= 0 ? 7 : 3; 

	// Highlight longest road
	if (G->lroad_pl == pl) {
		bool hl = false;
		vec_for(G->lroad_edges, i, 0)
			if (G->lroad_edges[i] == id) { hl=true; break; }
		if (hl) {
			imgdraw_line(frame, (ImgPoint){x1,y1}, (ImgPoint){x2,y2},
				col_shine, lw+4);
		}
	}
	
	// Draw edge
	imgdraw_line(frame, (ImgPoint){x1,y1}, (ImgPoint){x2,y2}, col, lw);
	
	// Alert on change
	game_ui_alert_draw(S, &S->alert.edges[id], x, y, pl, pl, 0);

	// Interaction
	if (pl == -1 && G->state != GAME_STATE_THIEF_MOVE) {
		int r = imgui_circle_interact(&S->gui, (ImGuiPoint){x, y}, scale*3/8);
		if (r & IMGUI_STATE_PRESS) {
			game_ui_action(S, G, GAME_UI_ACTION_EDGE, id);
		}
		if (r & IMGUI_STATE_HOVER) {
			// Selection highlight
			imgdraw_line(frame, (ImgPoint){x1,y1}, (ImgPoint){x2,y2}, col_sel, 9);

			// Tooltip (cost)
			dstr_printf(S->tooltip, "^I%c ^Bl^CtBuild a road\n",
				' '+S->ii.road[G->pl_curr]);
			game_res_to_dstr_a(S, &S->tooltip, G->rule.cost_edge);
		}
	}
	
	// ** Debug info
	if (S->c.dbg_lvl != 2) return 0;

	char buffer[32];
	sprintf(buffer, "e%d", id);
	textrender_render(S->font_small, &S->win.fb,
		&(ImgRect){x-scale, y-scale, scale*2, scale*2},
		buffer, NULL,
		col_dbg_txt, TR_RF_FAST_BLIT | TR_RF_ALIGN_VCENTER | TR_RF_ALIGN_HCENTER);
	
	return 0;
}

static
int node_draw(void* user, int id, float xf, float yf)
{
	DRAW_CTX_UNPACK
	int x = xf * scale + x0;
	int y = yf * scale + y0;

	int pl = G->nodes[id].pl;
	int lvl = G->nodes[id].lvl;

	// Draw trade route
	int tr = G->nodes[id].tr;
	if (tr >= 0) {
		// Draw connection from harbor to ship
		int x2 = G->troutes[tr].x * scale + x0;
		int y2 = G->troutes[tr].y * scale + y0;
		ImgColor col = {127,127,127,255};
		if (pl != -1) col = col_player_get(pl);
		imgdraw_line(frame, (ImgPoint){x,y}, (ImgPoint){x2,y2}, col, 2);
		
		// Draw harbor
		int ani = S->ii.harbor;
		const Image * img = &S->sprites[ani].img;
		imgdraw_blit(frame, (ImgPoint){x-img->w/2, y-img->h/2}, img, true);
	}
	
	// Draw settlement
	if (pl >= 0) {
		int ani = S->ii.settle[lvl][pl];
		const Image * img = &S->sprites[ani].img;
		imgdraw_blit(frame, (ImgPoint){x-img->w/2, y-img->h/2}, img, true);
	}
	
	// Alert on change
	game_ui_alert_draw(S, &S->alert.nodes[id], x, y, pl+lvl*1000, pl, 0);

	// Interaction
	if ((pl == -1 || (pl == G->pl_curr && lvl+1 < GAME_SETTLE_LEVEL_COUNT)) &&
		G->state != GAME_STATE_THIEF_MOVE)
	{
		assert( lvl >= -1 );
		int r = imgui_circle_interact(&S->gui, (ImGuiPoint){x, y}, scale/4);
		if (r & IMGUI_STATE_PRESS) {
			game_ui_action(S, G, GAME_UI_ACTION_NODE, id);
		}
		if (r & IMGUI_STATE_HOVER) {
			// Selection highlight
			//imgdraw_circle_border(frame, (ImgPoint){x,y}, scale/4, col_sel);
			//TODO: fast filled circle
			imgdraw_ring_smooth(frame, x, y, scale/4, scale/8, col_sel);

			// Tooltip (cost)
			dstr_printf(S->tooltip, "^I%c ^Bl^CtBuild a %s\n",
				' '+S->ii.settle[lvl+1][G->pl_curr],
				lvl==-1 ? "settlement" : "city");
			game_res_to_dstr_a(S, &S->tooltip, G->rule.cost_node[lvl+1]);
		}
	}
	
	// ** Debug info
	if (S->c.dbg_lvl != 2) return 0;

	char buffer[32];
	sprintf(buffer, "n%d", id);
	textrender_render(S->font_small, &S->win.fb,
		&(ImgRect){x-scale, y-scale, scale*2, scale*2},
		buffer, NULL,
		col_dbg_txt, TR_RF_FAST_BLIT | TR_RF_ALIGN_VCENTER | TR_RF_ALIGN_HCENTER);
	
	return 0;
}

static
void game_ui_draw_game_log(GameUi* S, Game* G)
{
	DynStr tmp_s = dstr_stack(128);
	Image * fb = &S->win.fb;
	int h_font = S->font_normal->metrics.height;
	int n_game_log = vec_count(G->log);
	
	for (int i=0, line=0; line<8 && i<n_game_log; ++i) {
		const GameLogEntry * E = &G->log[n_game_log-1-i];

		// Generate text
		dstr_copyz(tmp_s, "^fs");
		if (E->common.pl >= 0)
			//dstr_printfa(tmp_s, "%s ", G->pls[E->common.pl].name);
			dstr_printfa(tmp_s, "^i%c ", ' '+S->ii.icon_pl[E->common.pl]);

		switch (E->type) {
		//case GAME_LET_TURN_START:
		//	dstr_printfa(tmp_s, "starts turn %d", E->turn_start.turn+1);
		//	break;
		case GAME_LET_DICE_THROW:
			dstr_printfa(tmp_s, "dice throw: %d", E->dice_throw.value);
			break;
		case GAME_LET_THIEF_MOVE:
			dstr_printfa(tmp_s, "moves thief");
			break;
		case GAME_LET_RES_PAY:
			dstr_printfa(tmp_s, "pays ");
			game_res_to_dstr_a(S, &tmp_s, E->res_pay.cost);
			break;
		case GAME_LET_RES_PROD:
			dstr_printfa(tmp_s, "produces %d^i%c",
				E->res_prod.qty, ' '+S->ii.res_icon[E->res_prod.res]);
			break;
		case GAME_LET_RES_TRADE:
			dstr_printfa(tmp_s, "trades %d^i%c for %d^i%c",
				E->res_trade.qty_src, ' '+S->ii.res_icon[E->res_trade.res_src],
				E->res_trade.qty_dst, ' '+S->ii.res_icon[E->res_trade.res_dst]);
			break;
		case GAME_LET_RES_STEAL:
			dstr_printfa(tmp_s, "steals %d^i%c from ^i%c",
				E->res_steal.qty, ' '+S->ii.res_icon[E->res_steal.res],
				' '+S->ii.icon_pl[E->res_steal.pl2]);
			break;
		case GAME_LET_BUILD_EDGE:
			dstr_printfa(tmp_s, "builds a road");
			break;
		case GAME_LET_BUILD_NODE:
			dstr_printfa(tmp_s, "builds a %s",
				E->build_node.lvl == 0 ? "settlement" : "city");
			break;
		case GAME_LET_DEVCARD_BUY:
			dstr_printfa(tmp_s, "buys a dev.card");
			break;
		case GAME_LET_DEVCARD_USE:
			dstr_printfa(tmp_s, "uses a dev.card");
			break;
		case GAME_LET_VP_CHG:
			if (E->vp_chg.delta > 0)
				dstr_printfa(tmp_s, "^cwgains %d VP", E->vp_chg.delta);
			else if (E->vp_chg.delta < 0)
				dstr_printfa(tmp_s, "loses %d VP", -E->vp_chg.delta);
			break;
		default:
			dstr_resize(tmp_s, 0);
		}
		if (dstr_empty(tmp_s)) continue;

		// Draw text
		richtext_render(&S->rte, fb, &(ImgRect){0, fb->h - h_font*(line+1), 0, 0},
			strsl_fromd(tmp_s));
		line++;
	}
}

static
void game_ui_draw_game(GameUi* S, Game* G)
{
	DynStr tmp_s = dstr_stack(128);
	Image * frame = &S->win.fb;

	int fw=frame->w, fh=frame->h;
	int scale=64, sep=5, mar=10;
	int x_bar = fw - 256;  //Info bar x position

	// Compact view for small resolutions
	if (fw < 920 || fh < 680) {
		scale = 48;
		sep = 3;
		mar = 7;
		x_bar = fw - 200;
	}
	
	int font_h  = S->font_normal->metrics.lineskip;
	int font_em = S->font_normal->metrics.m_width;
	//int font_nw = S->font_normal->metrics.num_width;
	
	ImgColor col_pl = col_player_get(G->pl_curr);
	const GamePlayer * player = &G->pls[G->pl_curr];

	dstr_resize(S->tooltip, 0);

	// Background
	img_fill(frame, col_bkgr);

	// Latest log entries
	//TODO: optional
	game_ui_draw_game_log(S, G);

	// Board
	int x0=x_bar/2, y0=fh/2;
	DrawCtx dctx = {S, G, frame, scale, x0, y0};
	board_draw(&G->brd, &dctx, &(BoardHexDrawClass){
		tile_draw, edge_draw, node_draw});

	// Draw ships (trade routes)
	{
		const Image * img = &S->sprites[ S->ii.ship ].img;
		vec_for(G->troutes, i, 1) {
			int x = G->troutes[i].x * scale + x0 - img->w/2;
			int y = G->troutes[i].y * scale + y0 - img->h/2;
			imgdraw_blit(frame, (ImgPoint){x, y}, img, true);

			int res = G->troutes[i].res;
			if (res == -1) {
				dstr_printf(tmp_s, "^fb^hc%d:1", G->troutes[i].qty);
			} else {
				dstr_printf(tmp_s, "^fb^hc^i%c%d",
					' '+S->ii.res_icon[res], G->troutes[i].qty);
			}
			richtext_render(&S->rte, frame, &(ImgRect){x, y, img->w, img->h},
				strsl_fromd(tmp_s));
		}
	}
	
	// ** UI

	// Dice result
	if (G->lot_curr > 0) {
		const Image * img = &S->sprites[ S->ii.icon_dice ].img;
		imgdraw_blit(frame, (ImgPoint){mar, mar}, img, true);
		
		int x = mar+img->w+mar+sep;
		int y = mar+img->h/2;
		game_ui_num_draw(S, &S->num.dice_total, x, y, G->lot_curr, 0);
	}
	
	// UI info bar
	int x=x_bar, y=0;
	imgdraw_rect_fill(frame, (ImgRect){x, y, sep, fh-y}, col_ui_sep);
	x += sep;
	imgdraw_rect_fill(frame, (ImgRect){x, y, fw-x, fh-y}, col_ui_fill);

	{
		int w = (fw - x)/2;
		int h = font_h + sep*2;

		imgdraw_rect_fill(frame, (ImgRect){x, y, w*2, h}, col_ui_title);
		
		dstr_printf(tmp_s, "Round %d", G->n_round+1);
		textrender_render(S->font_normal, frame,
			&(ImgRect){x, y, w, h}, tmp_s, NULL, col_text, TR_RF_ALIGN_CENTER);
		
		dstr_printf(tmp_s, "Turn %d", G->n_turn+1);
		textrender_render(S->font_normal, frame,
			&(ImgRect){x+w, y, w, h}, tmp_s, NULL, col_text, TR_RF_ALIGN_CENTER);
		
		y += h;
	}
	
	// All players public info
	{
		x += sep;
		y += sep;

		// Columns positions
		int x_vp    = fw - font_em*2;
		int x_res   = fw - font_em*6;
		int x_card  = fw - font_em*10;
		int x_lroad = fw - font_em*13;
		int x_army  = fw - font_em*16;
		
		// Icons header
		{
			const Image * img = &S->sprites[ S->ii.icon_vp ].img;
			imgdraw_blit(frame, (ImgPoint){x_vp-img->w/2, y-img->h/2}, img, true);
		}
		{
			const Image * img = &S->sprites[ S->ii.icon_res ].img;
			imgdraw_blit(frame, (ImgPoint){x_res-img->w/2, y-img->h/2}, img, true);
		}
		{
			const Image * img = &S->sprites[ S->ii.icon_card ].img;
			imgdraw_blit(frame, (ImgPoint){x_card-img->w/2, y-img->h/2}, img, true);
		}
		{
			const Image * img = &S->sprites[ S->ii.icon_lroad ].img;
			imgdraw_blit(frame, (ImgPoint){x_lroad-img->w/2, y-img->h/2}, img, true);
		}
		{
			const Image * img = &S->sprites[ S->ii.icon_army ].img;
			imgdraw_blit(frame, (ImgPoint){x_army-img->w/2, y-img->h/2}, img, true);
		}
		
		y += font_h/2 + sep;
		
		for (int p=0; p<GAME_PLAYER_COUNT; ++p) {
			const GamePlayer * player = &G->pls[p];
			if (!player->type) continue;
		
			//ImgColor col_pl = col_player_get(p);
			//imgdraw_dot_smooth(frame, x-mar, y+font_h/2, mar*3/4, mar*3/4, col_pl);
			//textrender_render(S->font_normal, frame,
			//	&(ImgRect){x, y, fw-x, fh-y}, player->name, NULL, col_text, 0);
			
			dstr_printf(tmp_s, "^i%c %s%s",
				' '+S->ii.icon_pl[p], (scale < 64) ? "^fs": "", player->name);
			richtext_render(&S->rte, frame, &(ImgRect){x, y, fw-x, fh-y},
				strsl_fromd(tmp_s));

			// Victory points
			game_ui_num_draw(S, &S->num.pls[p].vp, x_vp, y+font_h/2,
				player->victory_points, 0);
			
			// Total resource count
			game_ui_num_draw(S, &S->num.pls[p].res_total, x_res, y+font_h/2,
				player->n_res,
				(player->n_res >= G->rule.thief_res_min) ? GAME_UI_NDF_WARNING : 0);
			
			// Dev.card count
			game_ui_num_draw(S, &S->num.pls[p].devcard, x_card, y+font_h/2,
				player->n_devcard, 0);

			// Longest road
			game_ui_num_draw(S, &S->num.pls[p].lroad, x_lroad, y+font_h/2,
				player->longest_road,
				G->lroad_pl == p ? GAME_UI_NDF_INTERACTION : 0 );

			// Armies
			game_ui_num_draw(S, &S->num.pls[p].army, x_army, y+font_h/2,
				player->n_army,
				G->larmy_pl == p ? GAME_UI_NDF_INTERACTION : 0 );

			y += font_h;
		}
		x -= sep;
		y += mar;
	}
	
	// Separator
	imgdraw_rect_fill(frame, (ImgRect){x, y, fw-x, sep}, col_ui_sep);
	y += sep;

	// Current player information
	int h = font_h + mar*2;
	imgdraw_rect_fill(frame, (ImgRect){x, y, fw-x, h}, col_pl);
	textrender_render(S->font_normal, frame,
		&(ImgRect){x, y, fw-x, h}, player->name, NULL,
		col_text, TR_RF_ALIGN_CENTER);
	y += h;
	
	// Resources
	{
		x += mar;
		y += mar;

		for (unsigned r=0; r<GAME_RES__COUNT; ++r) {
			int ws = (fw-x)/2;
			int hs = font_h*2;
			int xs = x + (r%2) * ws;
			int ys = y + (r/2) * hs;

			// Interaction (trade)
			ImgRect irect = {xs, ys, ws, hs};
			int state = imgui_rect_interact(&S->gui, irect);
			if (state & IMGUI_STATE_PRESS) {
				game_ui_action(S, G, GAME_UI_ACTION_RES, r);
			}
			if (state & IMGUI_STATE_HOVER) {
				imgdraw_rect_fill(frame, irect, col_sel);

				// Tooltip
				if (S->trade_res == -1)
					dstr_printf(S->tooltip, "Trade %d", player->res_tq[r]);
				else if (S->trade_res == (int)r)
					dstr_printf(S->tooltip, "Cancel trade");
				else
					dstr_printf(S->tooltip, "Receive 1");
			}

			// Large icon
			const Image * img = &S->sprites[ S->ii.res[r] ].img;
			imgdraw_blit(frame, (ImgPoint){xs-img->w/2+ws/4, ys-img->h/2+hs/2},
				img, true);
			
			// Quantity
			int ndflags = GAME_UI_NDF_BOLD;
			if (player->res[r] >= player->res_tq[r])
				ndflags |= GAME_UI_NDF_INTERACTION;
			game_ui_num_draw(S, &S->num.pls[G->pl_curr].res[r],
				xs+ws*5/8, ys+hs/2, player->res[r], ndflags);
		}
		
		x -= mar;
		y += font_h * ((GAME_RES__COUNT+1)/2)*2 + mar*2;
	}

	// Hint
	const char * hint_text=NULL;
	if (player->type == GAME_PLAYER_TYPE_AI) {
		hint_text = "AI done: end turn please";
	}
	else if (G->state == GAME_STATE_SETUP) {
		if (player->n_node <= G->n_round)
			hint_text = "Setup: place a settlement";
		else if (player->n_edge <= G->n_round)
			hint_text = "Setup: place a road";
		else
			hint_text = "End turn please";  //TODO: should be automatic?
	}
	else if (G->state == GAME_STATE_THIEF_MOVE) {
		hint_text = "Move the thief";
	}
	if (hint_text) {
		textrender_render(S->font_normal, frame,
			&(ImgRect){x, y, fw-x, font_h*2}, hint_text, NULL,
			col_text, TR_RF_ALIGN_CENTER);
	}
	
	y += font_h*3 + mar;

	// Dev.cards
	for (unsigned j=0; j<COUNTOF(player->devcards); ++j) {
		int xs = x + (fw - x) * (1 + j % 4) / 5;
		int ys = y + font_h*2 * (j/4);
		if (player->devcards[j] == 0) {  // Knights  //TODO: other cards
			// Card image
			const Image * img = &S->sprites[ S->ii.card_knight ].img;
			imgdraw_blit(frame, (ImgPoint){xs-img->w/2, ys-img->h/2}, img, true);

			// Interaction
			int m=3;
			ImgRect irect = {xs-img->w/2-m, ys-img->h/2-m, img->w+m*2, img->h+m*2};
			int r = imgui_rect_interact(&S->gui, irect);
			if (r & IMGUI_STATE_PRESS) {
				game_ui_action(S, G, GAME_UI_ACTION_DEVCARD_USE, j);
			}
			if (r & IMGUI_STATE_HOVER) {
				// Selection highlight
				imgdraw_rect_border(frame, irect, col_sel, m);
			
				// Tooltip
				if (player->turn_flags & GAME_PTF_DEVCARD_USED) {
					dstr_printf(S->tooltip, "One per turn");
				} else if (player->devcards_lock[j]) {
					dstr_printf(S->tooltip, "Just bought");
				} else {
					dstr_printf(S->tooltip, "Use card");
				}
			}
		}
	}

	y += font_h*2 * 2 + mar;

	int h_button = font_h + mar*2;
	int button_style = IMGUI_STYLE_BUTTON_PLAYER + G->pl_curr;

	// Dev card buy button
	int r = imgui_button(&S->gui, "Buy dev. card",
		&(ImGuiRect){x + mar, y, fw-x-mar*2, h_button}, button_style);
		//TODO: style player color
	if (r & IMGUI_STATE_PRESS) {
		game_ui_action(S, G, GAME_UI_ACTION_DEVCARD_BUY, 0);
	}
	if (r & IMGUI_STATE_HOVER) {
		// Tooltip (cost)
		game_res_to_dstr(S, &S->tooltip, G->rule.cost_devcard);
	}
	
	y += h_button + mar;

	// End turn button
	r = imgui_button(&S->gui, "End turn (E)",
		&(ImGuiRect){x, fh - h_button, fw-x, h_button}, button_style);
		//TODO: style player color
	if (r & IMGUI_STATE_PRESS) {
		game_ui_action(S, G, GAME_UI_ACTION_END_TURN, 0);
	}

	// Tooltip
	if (!dstr_empty(S->tooltip)) {
		imgui_tooltip(&S->gui, S->tooltip, NULL, 0);
	}
}

static
int game_ui_event_process_game(GameUi* S, Game* G, const SDL_Event* ev)
{
	int sym, mod;

	switch (ev->type) {
	case SDL_KEYDOWN:
		sym = ev->key.keysym.sym;
		mod = ev->key.keysym.mod;
		if (sym == SDLK_F12 && KMOD_CHECK(mod, KMOD_CTRL)) {
			// Cheat
			int res[] = {99,99,99,99,99};
			game_res_set(G->pls[G->pl_curr].res, res);
			return 1;
		}
		else if (sym == SDLK_s && mod == 0) {
			double d = S->c.ai_end_turn_delay;
			if (d > 1) d = 1;
			else if (d > 0.1) d = 0.1;
			else if (d > 0) d = 0;
			else d = 2;
			S->c.ai_end_turn_delay = d;
			return 1;
		}
		else if (sym == SDLK_e && mod == 0) {
			end_turn(S, G);
			return 1;
		}
	}

	return 0;
}

int game_ui_game_loop_step(GameUi* S, Game* G)
{
	if (!S->win.window) return -1;
	int R=0, r;

	if (S->state != GAME_UI_STATE_GAME) {
		S->state = GAME_UI_STATE_GAME;
		game_ui_game_state_clear(S, G);
		S->times.turn = timing_time();
		imgui_reset(&S->gui);
	}

	// Process events
	SDL_Event event;
	while (SDL_PollEvent(&event)) {
		r = game_ui_event_process_common(S, &event);
		//TODO: quit confirmation
		if (r == GAME_UI_EVENT_QUIT)
			RETURN( GAME_UI_RESULT_QUIT );
		game_ui_event_process_game(S, G, &event);
	}
	game_ui_time_measure(S, &S->times.events);

	// Auto-end AI turn after a delay
	if (G->pls[G->pl_curr].type == GAME_PLAYER_TYPE_AI &&
		S->c.ai_end_turn_delay > 0 &&
		S->times.last - S->times.turn > S->c.ai_end_turn_delay)
	{
		end_turn(S, G);
	}

	// Draw
	imgui_frame_begin(&S->gui);
	game_ui_draw_game(S, G);
	game_ui_draw_common(S);
	game_ui_time_measure(S, &S->times.draw);

	// Present
	imgui_sdl2_present(&S->win);
	game_ui_time_measure(S, &S->times.present);

end:
	return R;
}
