/* Copyright 2025-2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: MIT
 *
 * Program entry point.
 */
#include "ccommon/ccommon.h"
#include "ccommon/vector.h"
#include "ccommon/logging.h"
#include "ccommon/image_io.h"
//#include "ccommon/textrender_ft2.h"
#include "ccommon/textrender.h"
#include "ccommon/font_dejavu_sans_16px.h"
#include "ccommon/font_dejavu_sans_bold_16px.h"
#include "ccommon/font_dejavu_sans_10px.h"
#include "ccommon/sound.h"
#include "ccommon/sound_synth.h"
#include "game.h"
#include "ui_sdl.h"
#include "ai.h"

#ifdef __EMSCRIPTEN__
#include <emscripten.h>
#endif

static 
int sprite_load(GameUi* S, const char* path)
{
	GameSprite sprite={0};
	int r = img_load_file(&sprite.img, path);
	if (r < 0) {
		log_error("Could not load '%s' (error %xh)", path, -r);
		return r;
	}

	vec_push(S->sprites, sprite);
	return vec_count(S->sprites) - 1;
}

//static
//int sound_load(GameUi* S, const char* path)
//{
//	GameSound snd={0};
//	TRYR( game_sound_load_file(&snd, path) );
//	vec_push(S->sounds, snd);
//	return vec_count(S->sounds) - 1;
//}

static
int sound_synth(GameUi* S, SoundSynth* syn, const char* cmds)
{
	GameSound gs={0};
	audio_sdl_sound_spec_get(&S->aud, &gs.snd.ch, &gs.snd.freq, &gs.snd.format);
	TRYR( sndsynth_synth(syn, &gs.snd, cmds) );
	vec_push(S->sounds, gs);
	return vec_count(S->sounds) - 1;
}

static
int res_load(GameUi* S)
{
	int R=1;

	//const char * path_font = "/usr/share/fonts/TTF/DejaVuSans.ttf";
	//const char * path_font = "res/acalc.ttf";
	//S->font_normal = textrender_ft2_create(path_font, 16);
	//S->font_small  = textrender_ft2_create(path_font, 10);
	//S->font_bold   = textrender_ft2_create(path_font, 18);
	// Fast with no quality impact for our use
	//S->font_normal->c.rflags = TR_RF_NO_UTF8 | TR_RF_NO_KERNING;
	//S->font_small ->c.rflags = TR_RF_NO_UTF8 | TR_RF_NO_KERNING;
	//S->font_bold  ->c.rflags = TR_RF_NO_UTF8 | TR_RF_NO_KERNING;

	// Fonts
	S->font_normal = FONT_C_DejaVuSans_16;
	S->font_small  = FONT_C_DejaVuSans_10;
	S->font_bold   = FONT_C_DejaVuSansBold_16;

	// Graphics
	TRY( S->ii.res[0] = sprite_load(S, "res/res_sheep.png") );
	TRY( S->ii.res[1] = sprite_load(S, "res/res_wheat.png") );
	TRY( S->ii.res[2] = sprite_load(S, "res/res_wood.png") );
	TRY( S->ii.res[3] = sprite_load(S, "res/res_brick.png") );
	TRY( S->ii.res[4] = sprite_load(S, "res/res_ore.png") );
	
	TRY( S->ii.res_icon[0] = sprite_load(S, "res/icon_sheep.png") );
	TRY( S->ii.res_icon[1] = sprite_load(S, "res/icon_wheat.png") );
	TRY( S->ii.res_icon[2] = sprite_load(S, "res/icon_wood.png") );
	TRY( S->ii.res_icon[3] = sprite_load(S, "res/icon_brick.png") );
	TRY( S->ii.res_icon[4] = sprite_load(S, "res/icon_ore.png") );

	TRY( S->ii.ship = sprite_load(S, "res/ship.png") );
	TRY( S->ii.harbor = sprite_load(S, "res/harbor.png") );
	TRY( S->ii.thief = sprite_load(S, "res/thief.png") );
	TRY( S->ii.thief_hl = sprite_load(S, "res/thief_hl.png") );
	TRY( S->ii.circle = sprite_load(S, "res/circle.png") );
	TRY( S->ii.circle_hl = sprite_load(S, "res/circle_hl.png") );

	TRY( S->ii.icon_res = sprite_load(S, "res/icon_res.png") );
	TRY( S->ii.icon_card = sprite_load(S, "res/icon_card.png") );
	TRY( S->ii.icon_dice = sprite_load(S, "res/icon_dice.png") );
	TRY( S->ii.icon_vp = sprite_load(S, "res/icon_vp.png") );
	TRY( S->ii.icon_lroad = sprite_load(S, "res/icon_lroad.png") );
	TRY( S->ii.icon_army = sprite_load(S, "res/icon_army.png") );
	
	TRY( S->ii.card_knight = sprite_load(S, "res/card_knight.png") );

	TRY( S->ii.road[0] = sprite_load(S, "res/road.png") );
	TRY( S->ii.settle[0][0] = sprite_load(S, "res/settle0.png") );
	TRY( S->ii.settle[1][0] = sprite_load(S, "res/settle1.png") );
	TRY( S->ii.icon_pl[0] = sprite_load(S, "res/icon_player.png") );
	
	//TRY( S->si.button_click = sound_load(S, "res/pluck60.wav") );
	
	// Synthetize sounds
	SoundSynth syn={0};

	TRY( S->si.click = sound_synth(S, &syn, "b240 ece 72") );
	TRY( S->si.error = sound_synth(S, &syn,
		"b240 ece eo50 ea50 wsaw a20 '40 40") );
	TRY( S->si.game_start = sound_synth(S, &syn,
		"b240 eo50 wsisi '60 '66 '72 '78 _84") );
	TRY( S->si.turn_start = sound_synth(S, &syn,
		"b240 ece eo50 '66 '66") );
	TRY( S->si.build_edge = sound_synth(S, &syn,
		"b240 eo50 ea50 eA150 ed10 np 'np") );
	TRY( S->si.build_settle[0] = sound_synth(S, &syn,
		"b240 {ea50 _np}|'66 78") );
	TRY( S->si.build_settle[1] = sound_synth(S, &syn,
		"b240 {ea50 _np}|'66 '72 _78") );
	TRY( S->si.devcard_use = sound_synth(S, &syn,
		"b240 _np'|np 'np 'nw") );
	TRY( S->si.thief_move = sound_synth(S, &syn,
		"b240 ece ea50 np 'nw 'nw") );
	
end:
	sndsynth_free(&syn);
	return R;
}

#define APP_NAME_VERSION  "catan v0.5"

const char version_string[] = APP_NAME_VERSION "\n";

const char help_string[] =
APP_NAME_VERSION "\n"
"Settlers of Catan board game.\n"
"\n"
"Usage: catan [COMMAND[=VALUE]]\n"
"\n"
"Commands:\n"
"  v verbose         Verbose: increases information output. Can be repeated.\n"
"  q quiet           Output only errors.\n"
"    debug           Enables debug output.\n"
"  h help            Print this message and exit.\n"
"  V version         Print the version and exit.\n"
//TODO: document other commands
;

int cmd_parse_check(const char* line, char c, const char* key, const char** pvalue)
{
	int i=0;
	if (c && line[0] == c && (!line[1] || line[1] == '=')) i=1;
	else while (key[i] && (line[i] == key[i] || (line[i] == '-' && key[i] == '_')))
		++i;
	if (!line[i]) return 1;
	else if (line[i] == '=') {
		*pvalue = line+i+1;
		return 1;
	}
	else return 0;
}

#define ERROR_ARG_HELP -0xffff

typedef struct {
	bool skip_menu;
} Options;

int cmd_exec(const char* line, Game* G, GameUi* S, Options* opts)
{
	log_debug("cmd '%s'", line);

	const char *value="";
#define IF_CMD(SHORT, NAME) \
	else if (cmd_parse_check(line, SHORT, NAME, &value))

	if (!line) ;
	IF_CMD('h', "help") {
		fwrite(help_string, 1, sizeof(help_string), stdout);
		return ERROR_ARG_HELP;
	}
	IF_CMD('V', "version") {
		puts(APP_NAME_VERSION);
		return ERROR_ARG_HELP;
	}
	IF_CMD(0, "debug") {
		int n = value[0] ? (atoi(value) - 1) : 0;
		log_level_set(LOG_LVL_DEBUG + LOG_LVL_STEP*n);
	}
	IF_CMD('v', "verbose") {
		log_level_inc(+LOG_LVL_STEP);
	}
	IF_CMD('q', "quiet") {
		log_level_set(LOG_LVL_ERROR);
	}
	IF_CMD('F', "fullscreen") {
		int n = value[0] ? atoi(value) : true;
		ccFLAG_SET(S->c.flags, GAME_UI_CF_FULLSCREEN, n);
	}
	IF_CMD('p', "players") {
		int l = strlen(value);
		for (int p=0; p<GAME_PLAYER_COUNT; ++p) {
			char t = p<l ? value[p]	: 0;
			G->pls[p].type = (t == 'h') ? GAME_PLAYER_TYPE_EXTERN : 
			                 (t == 'a') ? GAME_PLAYER_TYPE_AI :
							 GAME_PLAYER_TYPE_NONE;
		}
		opts->skip_menu = true;
	}
	IF_CMD('R', "board_radius") {
		G->cfg.board_radius = atoi(value);
	}
	IF_CMD('s', "seed") {
		G->cfg.seed = atoi(value);
	}
	IF_CMD(0, "auto_setup") {
		int n = value[0] ? atoi(value) : true;
		ccFLAG_SET(G->cfg.flags, GAME_CF_AUTO_SETUP, n);
	}
	IF_CMD(0, "ai_end_turn_delay") {
		double delay = value[0] ? atof(value) : 2;
		S->c.ai_end_turn_delay = delay;

	}
	else {
		log_error("invalid command '%s'", line);
		return -1;
	}

	return 1;
}

// Web browsers / Emscripten uses cooperative multitasking and requieres the
// applicatio to manually give back control to the operation environment.

Game game={0};
GameUi ui={0};
Options opts={0};
int state=0;

static void mainloop(void)
{
	int r;
	if (state == 0) {
		if (opts.skip_menu)
			r = GAME_UI_RESULT_DONE;
		else
			r = game_ui_menu_loop_step(&ui, &game);
		if (r == GAME_UI_RESULT_DONE) {
			game_generate(&game);
			state = 1;
		}
	}
	else {
		r = game_ui_game_loop_step(&ui, &game);
	}
	if (r == GAME_UI_RESULT_QUIT || r < 0) {
#ifdef __EMSCRIPTEN__
		emscripten_cancel_main_loop();
#else
		exit(0);
#endif
	}
}

int main(int argc, char* argv[])
{
	int R=0;

	extern const ImageCodec img_codec_png;
	img_codec_register(&img_codec_png);

	game_default_config_set(&game);

	ui.c.ai_end_turn_delay = 0.5;

	for (int i=1; i<argc; ++i)
		TRY( cmd_exec(argv[i], &game, &ui, &opts) );

	TRY( game_ui_create(&ui) );

	TRY( res_load(&ui) );

	game_ui_resources_finish(&ui);

#ifdef __EMSCRIPTEN__
    emscripten_set_main_loop(mainloop, 0, 1);
#else
	while (1) { mainloop(); };
#endif

end:
	if (R == ERROR_ARG_HELP) R=0;
	if (R<0) {
		log_error("Error exit: %x", -R);
		R = -R;
	}
	game_ui_free(&ui);
	game_free(&game);
	return R;
}
