/* Copyright 2025-2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: MIT
 *
 * Game logic.
 */
#pragma once
#include <stddef.h>
#include <stdbool.h>
#include <stdint.h>
#include "ccommon/rng_mt19937.h"
#include "board_hex.h"

enum GameResource {
	GAME_RES_SHEEP,
	GAME_RES_WHEAT,
	GAME_RES_WOOD,
	GAME_RES_BRICK,
	GAME_RES_ORE,
	GAME_RES__COUNT,
};

enum {
	GAME_SETTLE_LEVEL_COUNT = 2,
	GAME_PLAYER_COUNT = 4,
	GAME_PLAYER_TR_MAX = 8,
	GAME_PLAYER_DC_MAX = 8,
};

enum GamePlayerType {
	GAME_PLAYER_TYPE_NONE   = 0,  //Not playing
	GAME_PLAYER_TYPE_EXTERN = 1,  //Human
	GAME_PLAYER_TYPE_AI     = 2,  //AI
	GAME_PLAYER_TYPE__COUNT
};

enum GamePlayerTurnFlag {
	GAME_PTF_DEVCARD_USED = 1,
};

typedef struct {
	// Config	
	int type;
	char name[32];
	// State
	int res[GAME_RES__COUNT];  // Amount of each resource
	int res_tq[GAME_RES__COUNT];  // Trade quantities for each resource
	int troutes[GAME_PLAYER_TR_MAX];  // Trade routes availables to this player
	int devcards[GAME_PLAYER_DC_MAX];  // Development cards
	int devcards_lock[GAME_PLAYER_DC_MAX];  // Just bought, cannot use
	int turn_flags;
	// Stats
	int n_edge, n_node, n_node_lvl[GAME_SETTLE_LEVEL_COUNT];
	int n_res, n_devcard, n_army;
	int last_node;
	int longest_road;
	int victory_points;
} GamePlayer;

typedef struct {
	int res;  // Resource type
	int num;  // Dice roll number
} GameTile;

typedef struct {
	int pl;  // Controlling player
} GameEdge;

typedef struct {
	int pl;  // Controlling player
	int lvl;  // Player struture level
	int tr;  // Connect to this trade route
} GameNode;

typedef struct {
	int res;  // Which resource is received (-1 for any)
	int qty;  // Quantity of the resouce that is traded for one of another
	float x,y;  // Position in the board (decorative)
} GameTradeRoute;

enum GameLogEntryType {
	GAME_LET_TURN_START,
	GAME_LET_DICE_THROW,
	GAME_LET_THIEF_MOVE,
	GAME_LET_RES_PAY,
	GAME_LET_RES_PROD,
	GAME_LET_RES_TRADE,
	GAME_LET_RES_STEAL,
	GAME_LET_BUILD_EDGE,
	GAME_LET_BUILD_NODE,
	GAME_LET_DEVCARD_BUY,
	GAME_LET_DEVCARD_USE,
	GAME_LET_VP_CHG,
	GAME_LET__COUNT
};

typedef union {
	int type;
	struct { int type, pl; } common;
	struct { int type, pl, turn; } turn_start;
	struct { int type, pl, value; } dice_throw;
	struct { int type, pl, tile_src, tile_dst; } thief_move;
	struct { int type, pl, cost[GAME_RES__COUNT]; } res_pay;
	struct { int type, pl, res, qty, tile, node; } res_prod;
	struct { int type, pl, res_src, res_dst, qty_src, qty_dst; } res_trade;
	struct { int type, pl, pl2, res, qty; } res_steal;
	struct { int type, pl, edge; } build_edge;
	struct { int type, pl, node, lvl; } build_node;
	struct { int type, pl, delta; } vp_chg;
} GameLogEntry;

enum GameState {
	GAME_STATE_NONE,
	GAME_STATE_SETUP,
	GAME_STATE_NORMAL,
	GAME_STATE_THIEF_MOVE,
};

enum GameCfgFlag {
	// Automatically set the initial roads and settlements for all players
	GAME_CF_AUTO_SETUP = 1,
};

typedef struct {
	GamePlayer pls[GAME_PLAYER_COUNT];  // Players, configure before starting

	// Board
	BoardHex brd;
	// Board payloads (game-specific data)
	GameTile * tiles;  //vector
	GameEdge * edges;  //vector
	GameNode * nodes;  //vector
	GameTradeRoute * troutes;  //vector
	// State
	MT19937State rng;
	int state;
	int thief_tile;  // Tile with the theft
	int pl_curr;  // Current player
	int lot_curr;  // Current lot (tile or thief)
	int n_turn, n_round;
	// Larges army
	int larmy_n, larmy_pl;
	// Longest road
	int lroad_pl;
	int * lroad_edges;  //vector

	GameLogEntry * log;  //vector
	
	// Rules
	struct {
		int cost_edge[GAME_RES__COUNT];
		int cost_node[2][GAME_RES__COUNT];
		int cost_devcard[GAME_RES__COUNT];
		int dice_count, dice_faces;
		float * num_prob;  //Number probability distribution, vector
		int thief_num;  // Random number for theft activation
		int thief_res_min;
		int larmy_min, larmy_vp;
		int lroad_min;  // Minimum length to award the longest road
		int lroad_vp;
		//int flags;
	} rule;

	// Configuration
	struct {
		uint32_t seed;
		unsigned board_radius;
		int flags;
	} cfg;
} Game;

void game_free(Game* S);

// Sets the default configuration, rules and players.
// You should call this first and then change the values of .cfg, .rule and .pls.
void game_default_config_set(Game* S);

// Generate and start a new game with the current configuration.
void game_generate(Game* S);

enum GamePlayerAction {
	GAME_PA_END_TURN,
	GAME_PA_BUILD_EDGE,  // edge_idx
	GAME_PA_BUILD_NODE,  // node_idx
	GAME_PA_TRADE_RES,   // res_give, res_receive
	GAME_PA_THIEF_MOVE,  // tile_idx
	GAME_PA_DEVCARD_BUY,
	GAME_PA_DEVCARD_USE,
};

int game_player_action(Game* S, int pl, int action, int p1, int p2);
//TODO: check only flag

// Resource vector operations

static inline
int game_res_count(const int res[GAME_RES__COUNT])
{
	int n=0;
	for (unsigned i=0; i<GAME_RES__COUNT; ++i) n += res[i];
	return n;
}

static inline
bool game_res_check(const int res[GAME_RES__COUNT], const int cost[GAME_RES__COUNT])
{
	for (unsigned i=0; i<GAME_RES__COUNT; ++i)
		if (res[i] < cost[i]) return false;
	return true;
}

static inline
void game_res_set(int res[GAME_RES__COUNT], const int other[GAME_RES__COUNT])
{
	for (unsigned i=0; i<GAME_RES__COUNT; ++i)
		res[i] = other[i];
}

static inline
void game_res_sub(int res[GAME_RES__COUNT], const int other[GAME_RES__COUNT])
{
	for (unsigned i=0; i<GAME_RES__COUNT; ++i)
		res[i] -= other[i];
}

static inline
void game_res_add(int res[GAME_RES__COUNT], const int other[GAME_RES__COUNT])
{
	for (unsigned i=0; i<GAME_RES__COUNT; ++i)
		res[i] += other[i];
}
