/* Copyright 2025-2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: MIT
 */
#include "game.h"
#include "ai.h"
#include "ccommon/ccommon.h"
#include "ccommon/timing.h"
#include "ccommon/vector.h"
#include "ccommon/logging.h"
#include <math.h>

#define RES_CNT  GAME_RES__COUNT

#define RES_SET(VAR, ...) \
	memcpy(&(VAR), (int[RES_CNT]){__VA_ARGS__}, sizeof(int[RES_CNT]))

static
void game_log_clean(Game* S)
{
	vec_resize(S->log, 0);
}

void game_free(Game* S)
{
	//game_log_clean(S);
	vec_free(S->log);
	board_hex_free(&S->brd);
	vec_free(S->troutes);
	vec_free(S->nodes);
	vec_free(S->edges);
	vec_free(S->tiles);
	vec_free(S->rule.num_prob);
	vec_free(S->lroad_edges);
}

int game_randint(Game* S, int min, int max)
{
	return mt19937_randint(&S->rng, min, max);
}

int game_dice_throw(Game* S)
{
	int n = 0;
	for (int i=0; i<S->rule.dice_count; ++i)
		n += game_randint(S, 1, S->rule.dice_faces);
	return n;
}

int game_random_pick(Game* S, int n_total, int n_opt, int q_opt[n_opt])
{
	if (n_total <= 0) {
		n_total = 0;
		for (int i=0; i<n_opt; ++i) n_total += q_opt[i];
	}
	assert( n_total > 0 );

	int n = game_randint(S, 0, n_total-1);
	int i=-1;
	while (n >= 0 && i < n_opt) { i++; n -= q_opt[i]; }
	assert( i < n_opt && q_opt[i] > 0 );
	return i;
}

void game_log(Game* S, enum GameLogEntryType type, int pl, ...)
{
	GameLogEntry entry={.common={.type=type, .pl=pl}};
	va_list ap;
	va_start(ap, pl);
	
	switch (type) {
	case GAME_LET_TURN_START: {
		int n = entry.turn_start.turn = va_arg(ap, int);
		log_debug("P%d turn start %d", pl, n);
		break;
	}
	case GAME_LET_DICE_THROW: {
		int n = entry.dice_throw.value = va_arg(ap, int);
		log_debug("P%d dice throw %d", pl, n);
		break;
	}
	case GAME_LET_THIEF_MOVE: {
		int t1 = entry.thief_move.tile_src = va_arg(ap, int);
		int t2 = entry.thief_move.tile_dst = va_arg(ap, int);
		log_debug("P%d thief move from %d to %d", pl, t1, t2);
		break;
	}
	case GAME_LET_RES_PAY: {
		int * cost = va_arg(ap, int*);
		game_res_set(entry.res_pay.cost, cost);
		if (log_line_begin(LOG_LVL_DEBUG)) {
			log_line_strf("P%d pays:", pl);
			for (int r=0; r<GAME_RES__COUNT; ++r)
				log_line_strf(" %d", cost[r]);
			log_line_end();
		}
		break;
	}
	case GAME_LET_RES_PROD: {
		int r = entry.res_prod.res = va_arg(ap, int);
		int q = entry.res_prod.qty = va_arg(ap, int);
		int t = entry.res_prod.tile = va_arg(ap, int);
		int n = entry.res_prod.node = va_arg(ap, int);
		log_debug("P%d harvests %d of resource %d from node %d in tile %d",
			pl, q, r, n, t);
		break;
	}
	case GAME_LET_RES_TRADE: {
		int r1 = entry.res_trade.res_src = va_arg(ap, int);
		int r2 = entry.res_trade.res_dst = va_arg(ap, int);
		int q1 = entry.res_trade.qty_src = va_arg(ap, int);
		int q2 = entry.res_trade.qty_dst = va_arg(ap, int);
		log_debug("P%d trades %d of resource %d for %d of resource %d",
			pl, q1, r1, q2, r2);
		break;
	}
	case GAME_LET_RES_STEAL: {
		int p = entry.res_steal.pl2 = va_arg(ap, int);
		int r = entry.res_steal.res = va_arg(ap, int);
		int q = entry.res_steal.qty = va_arg(ap, int);
		log_debug("P%d steals %d of resource %d from P%d", pl, q, r, p);
		break;
	}
	case GAME_LET_BUILD_EDGE: {
		int e = entry.build_edge.edge = va_arg(ap, int);
		log_debug("P%d builds at edge %d", pl, e);
		break;
	}
	case GAME_LET_BUILD_NODE: {
		int n = entry.build_node.node = va_arg(ap, int);
		int l = entry.build_node.lvl = va_arg(ap, int);
		log_debug("P%d builds at node %d (level %d)", pl, n, l);
		break;
	}
	case GAME_LET_DEVCARD_BUY: {
		log_debug("P%d buys a development card", pl);
		break;
	}
	case GAME_LET_DEVCARD_USE: {
		int type = va_arg(ap, int);
		log_debug("P%d uses a development card of type %d", pl, type);
		break;
	}
	case GAME_LET_VP_CHG: {
		int n = entry.vp_chg.delta = va_arg(ap, int);
		log_debug("P%d victory points change: %d", pl, n);
		break;
	}
	default:
		log_debug("unhandled log entry type %d", type);
	}

	va_end(ap);
	vec_push(S->log, entry);
}

void game_default_config_set(Game* S)
{
	// Config
	S->cfg.board_radius = 2;

	// Rules
	RES_SET(S->rule.cost_edge   , 0,0,1,1,0);
	RES_SET(S->rule.cost_node[0], 1,1,1,1,0);
	RES_SET(S->rule.cost_node[1], 0,2,0,0,3);
	RES_SET(S->rule.cost_devcard, 1,1,0,0,1);
	S->rule.dice_count = 2;
	S->rule.dice_faces = 6;
	S->rule.thief_res_min = 7;
	S->rule.larmy_min = 3;
	S->rule.larmy_vp = 2;
	S->rule.lroad_min = 5;
	S->rule.lroad_vp = 2;
	
	int dc = S->rule.dice_count;
	int df = S->rule.dice_faces;
	
	S->rule.thief_num = (df+1) * dc / 2;  // Most common value
	
	// Players
	for (int p=0; p<GAME_PLAYER_COUNT; ++p) {
		S->pls[p].type = 0;
		S->pls[p].name[0] = 0;
	}
	S->pls[0].type = GAME_PLAYER_TYPE_EXTERN;
	S->pls[1].type = GAME_PLAYER_TYPE_AI;
}

static
void game_player_troute_add(Game* S, int pl, int tr)
{
	GamePlayer * player = &S->pls[pl];
	for (unsigned j=0; j<COUNTOF(player->troutes); ++j) {
		if (player->troutes[j] == tr) return;
		if (player->troutes[j] == -1) {
			player->troutes[j] = tr;
			break;
		}
	}

	// Update trade quantities
	int r = S->troutes[tr].res;
	int q = S->troutes[tr].qty;
	if (r >= 0) {
		MINSET(player->res_tq[r], q);
	} else {
		for (int j=0; j<GAME_RES__COUNT; ++j)
			MINSET(player->res_tq[j], q);
	}
}

static
void game_generate_board(Game* S)
{
	IFFALSESET(S->cfg.board_radius, 2);

	//Count per layer:
	//tiles: 1, 6, 6*2, 6*3
	//edges: 6, 6*4, 6*7, 6*10
	//nodes: 6, 6*3, 6*5, 6*7
	unsigned n_tiles=1, n_edges=6, n_nodes=6;
	for (unsigned r=1; r<=S->cfg.board_radius; ++r) {
		n_tiles += 6*r;
		n_edges += 6*(r*3+1);
		n_nodes += 6*(r*2+1);
	}
	board_hex_reserve(&S->brd, n_tiles, n_edges, n_nodes);
	
	// Add tiles one at a time
	board_tile_root_add(&S->brd);
	for (unsigned i=0; board_tile_count(&S->brd) < n_tiles; ++i) {
		for (unsigned j=0; j<6; ++j) {
			board_tile_add(&S->brd, i, j);  // May fail if the edge is full
		}
	}
	if (log_level_check(LOG_LVL_DEBUG2)) {
		DynStr tmps=NULL;
		board_debug_dump(&S->brd, &tmps);
		log_debug2(tmps);
		dstr_free(tmps);
	}
	log_debug("Board: %d tiles, %d edges, %d nodes",
		board_tile_count(&S->brd),
		board_edge_count(&S->brd),
		board_node_count(&S->brd) );
	assert( board_tile_count(&S->brd) == n_tiles );
	assert( board_edge_count(&S->brd) == n_edges );
	assert( board_node_count(&S->brd) == n_nodes );

	// Board payloads (game-specific data)
	vec_resize_zero(S->tiles, n_tiles);
	vec_resize_zero(S->edges, n_edges);
	vec_resize_zero(S->nodes, n_nodes);

	vec_for(S->edges, i, 0)  S->edges[i].pl = -1;
	vec_for(S->nodes, i, 0) {
		S->nodes[i].pl  = -1;
		S->nodes[i].lvl = -1;
		S->nodes[i].tr  = -1;
	}
}

static
void game_generate_tile_assign(Game* S)
{
	int tn = S->rule.thief_num;
	int n_tiles = vec_count(S->tiles);
	int n_desert = n_tiles / 12;

	int dc = S->rule.dice_count;
	int df = S->rule.dice_faces;
	int dice_min = dc;
	int dice_max = dc*df;

	// Determine the quantity of each resource in the board tiles
	// Standard board: 4,4,4,3,3,1
	int res_picks[GAME_RES__COUNT+1] = {0}; //5 res + desert
	{
		int n_opt = GAME_RES__COUNT;
		int n_pos = n_tiles - n_desert;  // Tiles with resources
		int c = n_pos / n_opt;
		int r = n_pos % n_opt;
		for (int i=0; i<n_opt; ++i)
			res_picks[i] = c + (i<r);
		res_picks[n_opt] = n_desert;
	}
	
	// Determine the quantity of each number in the board tiles
	// Standard board: all 2, except 1 for 2 and 12
	int num_picks[dice_max+1];
	MEM_ZERO(num_picks);
	{
		int n_opt = dice_max - dice_min;
		int n_pos = n_tiles - n_desert;  // Tiles with resources
		int c = n_pos / n_opt;
		int r = n_pos % n_opt;
		int r1 = (n_opt - r) / 2;
		int r2 = r1 + r;
		for (int i=0; i<n_opt; ++i) {
			int num = dice_min+i + (dice_min+i >= tn);
			num_picks[num] = c + (r1<=i && i<r2);
		}
	}
	
	// Assign resources and lot numbers to the tiles
	vec_for(S->tiles, t, 0) {
		int r = game_random_pick(S, n_tiles-t, COUNTOF(res_picks), res_picks);
		res_picks[r]--;
		if (r == GAME_RES__COUNT) {  // Desert
			S->tiles[t].res = -1;
			S->tiles[t].num = 0;
			S->thief_tile = t;  // Thief starting in a desert tile
		}
		else {
			S->tiles[t].res = r;
		
			int n = game_random_pick(S, n_tiles-n_desert-t,
						COUNTOF(num_picks), num_picks);
			num_picks[n]--;
			S->tiles[t].num = n;
		}
	}
}

static inline
bool node_2e_is(const Game* S, int n) {
	for (int j=0; j<3; ++j)
		if (S->brd.nodes[n].edges[j] == -1)
			return true;
	return false;
}

static
void game_generate_trade_routes(Game* S)
{
	// Nodes with only two edges
	int n_nodes_2e  = 6 * (S->cfg.board_radius + 1);  //18
	// Nodes delimiting the outer border
	int n_nodes_out = 6 * (S->cfg.board_radius * 2 + 1);  //30
	// Number of trade routes (ships) to distribute
	int n_troute = n_nodes_2e / 2;  //9
	// Number of connections for each trade route
	int n_tr_conn = 2;

	// Create the trade routes
	vec_resize_zero(S->troutes, n_troute+1);
	S->troutes[0] = (GameTradeRoute){ -1, 4 };  // Generic trade route
		//TODO: 4 -> rule / user option

	// Random assignment of the route types
	int picks[GAME_RES__COUNT+1];  //2:1 for each resource + 3:1 generic
	{
		int n = GAME_RES__COUNT+1;
		int c = n_troute / n;
		int r = n_troute % n;
		for (int i=0; i<n; ++i) picks[i] = c;
		picks[n-1] += r;
	}
	// Standard board: 1,1,1,1,1,4
	vec_for(S->troutes, i, 1) {
		int r = game_random_pick(S, n_troute+1-i, GAME_RES__COUNT+1, picks);
		picks[r]--;
		if (r < GAME_RES__COUNT)
			S->troutes[i] = (GameTradeRoute){  r, 2 };
		else
			S->troutes[i] = (GameTradeRoute){ -1, 3 };
	}
	
	// Find an 2e node
	int n = vec_count(S->nodes) - 1;
	while (!node_2e_is(S, n)) n--;
	assert( n >= 0 );

	// Traverse the board border nodes
	bool is_2e = true;
	int i_node=0, i_tr = 1, n_tr_n=0, n_skip=0, np = -1;
	while (i_tr < n_troute+1) {
		// Assign current TR
		if (n_skip == 0) {
			//log_debug("tr%d -> n%d", i_tr, n);
			assert( S->nodes[n].tr == -1 );
			S->nodes[n].tr = i_tr;
			n_tr_n++;
		}
		else n_skip--;
		
		// Two node (harbors) per route
		if (n_tr_n == n_tr_conn) {
			// Set TR ship position in the board (decorative)
			assert( S->troutes[i_tr].x == 0 && np != -1 );
			// Half an edge length outside with respect to the nodes midpoint
			// Nodes midpoint
			float x = (S->brd.nodes[n].x + S->brd.nodes[np].x) * 0.5;
			float y = (S->brd.nodes[n].y + S->brd.nodes[np].y) * 0.5;
			//float f = 1 / sqrt(x*x+y*y);
			
			float dx = S->brd.nodes[n].x - S->brd.nodes[np].x;
			float dy = S->brd.nodes[n].y - S->brd.nodes[np].y;
			float df = (x * dx + y * dy) / (dx * dx + dy * dy);
			
			float nx = x - dx * df;
			float ny = y - dy * df;
			float nf = 1 / sqrt(nx*nx+ny*ny);
			
			S->troutes[i_tr].x = x + nx*0.5*nf;
			S->troutes[i_tr].y = y + ny*0.5*nf;
			
			n_skip = 1 + (((i_node+1) * n_troute) < (i_tr * (n_nodes_out-1)));
			n_tr_n = 0;
			i_tr++;
		}

		// Find next node
		int j;
		for (j=0; j<3; ++j) {
			int n_ = S->brd.nodes[n].nodes[j];
			if (n_ == -1 || n_ == np) continue;
			bool is_2e_ = node_2e_is(S, n_);
			if (!is_2e && !is_2e_) continue;
			np = n;
			n = n_;
			is_2e = is_2e_;
			i_node++;
			break;
		}
		assert( j < 3 );
	}
}

void game_generate(Game* S)
{
	// Initialize the random number generator
	if (S->cfg.seed == 0) S->cfg.seed = timing_timeofday();
	mt19937_init(&S->rng, S->cfg.seed);
	log_debug("Seed: %u", S->cfg.seed);

	if (!(S->rule.dice_count > 0 && S->rule.dice_faces > 0))
		game_default_config_set(S);

	int dc = S->rule.dice_count;
	int df = S->rule.dice_faces;
	int dice_min = dc;
	int dice_max = dc*df;

	IFFALSESET(S->rule.thief_num, (df+1) * dc / 2 );  // Most common value

	// Calculate number probability distribution
	vec_resize_zero(S->rule.num_prob, dice_max+1);
	assert( dc == 2 );
	for (int d1=1; d1<=df; ++d1)
		for (int d2=1; d2<=df; ++d2)
			S->rule.num_prob[d1+d2]++;
	
	vec_for(S->rule.num_prob, i, 0)
		S->rule.num_prob[i] /= df*df;
	
	if (log_line_begin(LOG_LVL_DEBUG)) {
		log_line_strf("Num prob dist (%d-%d):", dice_min, dice_max);
		vec_for(S->rule.num_prob, i, dice_min)
			log_line_strf(" %.2f", S->rule.num_prob[i]);
		log_line_end();
	}
	
	// Board
	game_generate_board(S);
	game_generate_tile_assign(S);
	game_generate_trade_routes(S);

	// Players
	unsigned n_player=0;
	for (int p=0; p<GAME_PLAYER_COUNT; ++p) {
		n_player += !!S->pls[p].type;

		if (!S->pls[p].name[0]) {
			if (S->pls[p].type == GAME_PLAYER_TYPE_EXTERN)
				sprintf(S->pls[p].name, "Player %d", p+1);
			else
				sprintf(S->pls[p].name, "Player %d (AI)", p+1);
		}

		for (unsigned j=0; j<COUNTOF(S->pls[p].devcards); ++j)
			S->pls[p].devcards[j] = -1;

		for (unsigned j=0; j<COUNTOF(S->pls[p].troutes); ++j)
			S->pls[p].troutes[j] = -1;
		
		S->pls[p].troutes[0] = 0;
		for (int r=0; r<GAME_RES__COUNT; ++r)
			S->pls[p].res_tq[r] = S->troutes[0].qty;
	}

	// Game start
	game_log_clean(S);

	S->n_turn = 0;
	S->n_round = 0;
	S->larmy_pl = -1;
	S->lroad_pl = -1;
	vec_resize(S->lroad_edges, 0);

	if (!n_player) {
		S->pl_curr = 0;
		S->state = 0;
		return;
	}

	S->pl_curr = 0;
	S->state = GAME_STATE_SETUP;
	game_log(S, GAME_LET_TURN_START, S->pl_curr, S->n_turn);

	if (S->cfg.flags & GAME_CF_AUTO_SETUP) {
		while (S->state == GAME_STATE_SETUP) {
			for (int p=0; p<GAME_PLAYER_COUNT; ++p) {
				if (!S->pls[p].type) continue;
				game_player_action_ai(S, p);
				game_player_action(S, p, GAME_PA_END_TURN, 0, 0);
			}
		}
	}

	if (S->pls[S->pl_curr].type == GAME_PLAYER_TYPE_AI) {
		game_player_action_ai(S, S->pl_curr);
	}
}

void game_tile_produce(Game* S, int i_tile)
{
	int res = S->tiles[i_tile].res;
	BoardHexTile ti = board_hex_tile_get(&S->brd, i_tile);

	for (int j=0; j<6; ++j) {
		int n = ti.nodes[j];
		if (n != -1 && S->nodes[n].pl != -1) {
			int pl = S->nodes[n].pl;
			int qty = S->nodes[n].lvl + 1;
			S->pls[pl].res[res] += qty;
			game_log(S, GAME_LET_RES_PROD, pl, res, qty, i_tile, n);
		}
	}
}

void game_node_produce(Game* S, int i_node)
{
	int pl = S->nodes[i_node].pl;
	if (pl == -1) return;
	int qty = S->nodes[i_node].lvl + 1;
	
	BoardHexNode ni = board_hex_node_get(&S->brd, i_node);
	for (unsigned i=0; i<COUNTOF(ni.tiles); ++i) {
		int tile = ni.tiles[i];
		int res = S->tiles[tile].res;
		if (res != -1) {
			S->pls[pl].res[res] += qty;
			game_log(S, GAME_LET_RES_PROD, pl, res, qty, tile, i_node);
		}
	}
}

void game_players_update(Game* S)
{
	for (int p=0; p<GAME_PLAYER_COUNT; ++p) {
		GamePlayer * player = &S->pls[p];
		if (!player->type) continue;

		player->n_res = game_res_count(player->res);
		
		player->n_devcard = 0;
		for (unsigned j=0; j<COUNTOF(player->devcards); ++j)
			player->n_devcard += (player->devcards[j] != -1);
		
		// Update victory points
		int vp = 0;
		for (int i=0; i<GAME_SETTLE_LEVEL_COUNT; ++i)
			vp += player->n_node_lvl[i] * (i+1);
		if (S->lroad_pl == p)
			vp += S->rule.lroad_vp;
		if (S->larmy_pl == p)
			vp += S->rule.larmy_vp;
		
		if (vp != player->victory_points) {
			game_log(S, GAME_LET_VP_CHG, p, vp - player->victory_points);
			player->victory_points = vp;
		}
	}
}

typedef struct {
	int e, n, j;
} lroad_stack_t;

static inline
bool lroad_stack_edge_check(const lroad_stack_t* stack, int i, int e) {
	for (; i>=0; --i) if (stack[i].e == e) return true;
	return false;
}

/* Determines the longest road starting at edge <e> and moving toward the
 * local node <jn> = 0 or 1.
 */
static
int game_lroad_from(const Game* S, int e, int jn, lroad_stack_t* stack,
	int** pedges_global)
{
	int pl = S->edges[e].pl;
	assert( pl >= 0 );

	int l_best = 0;
	
	int n0 = S->brd.edges[e].nodes[jn];  // Current node
	stack[0] = (lroad_stack_t){e, n0, 0};

	int i=0;
	while (i >= 0) {
		assert( i < S->pls[pl].n_edge );
		int n = stack[i].n;
		int j = stack[i].j;
		// Find next valid edge
		for (; j<3; ++j) {
			int e_ = S->brd.nodes[n].edges[j];
			if (S->edges[e_].pl != pl) continue;
			if (lroad_stack_edge_check(stack, i, e_)) continue;
			break;
		}
		
		if (j < 3) {  // Add new step
			stack[i].j = j+1;
			i++;
			int e_ = S->brd.nodes[n].edges[j];
			int n_ = S->brd.nodes[n].nodes[j];
			stack[i] = (lroad_stack_t){e_, n_, 0};
		}
		else {  // End of this candidate road
			if (i+1 > (int)vec_count(*pedges_global)) {  // New global best
				vec_resize(*pedges_global, i+1);
				vec_for(*pedges_global, ii, 0) {
					(*pedges_global)[ii] = stack[ii].e;
				}
			}
			MAXSET(l_best, i+1);
			i--;
		}
	}

	return l_best;  // Length of the longest road
}

/* Determines the longest road of any player.
 * Stores for each player the length of their longest road.
 */
void game_lroad_update(Game* S)
{
	lroad_stack_t * stack = NULL;

	int n_edge_max=0;
	for (int p=0; p<GAME_PLAYER_COUNT; ++p) {
		MAXSET(n_edge_max, S->pls[p].n_edge);
	}
	vec_resize(stack, n_edge_max);

	vec_for(S->edges, e, 0) {
		int pl = S->edges[e].pl;
		if (pl == -1) continue;
		for (int j=0; j<2; ++j) {
			int l = game_lroad_from(S, e, j, stack, &S->lroad_edges);
			MAXSET(S->pls[pl].longest_road, l);
		}
	}

	// Player with the longest road.
	if ( (int)vec_count(S->lroad_edges) >= S->rule.lroad_min ) {
		S->lroad_pl = S->edges[ S->lroad_edges[0] ].pl;
		//TODO: log change
	}

	vec_free(stack);
}

// Get the next player, skipping empty slots.
static inline
int game_player_next_get(const Game* S, int pl)
{
	assert( 0 <= pl && pl < GAME_PLAYER_COUNT );
	if (!S->pls[pl].type) return pl;  //No players
	while (1) {
		pl++;
		if (pl == GAME_PLAYER_COUNT) pl = 0;
		if (S->pls[pl].type) return pl;
	}
}

static inline
int game_player_prev_get(const Game* S, int pl)
{
	assert( 0 <= pl && pl < GAME_PLAYER_COUNT );
	if (!S->pls[pl].type) return pl;  //No players
	while (1) {
		pl--;
		if (pl == -1) pl = GAME_PLAYER_COUNT - 1;
		if (S->pls[pl].type) return pl;
	}
}

int game_player_action(Game* S, int pl, int action, int p1, int p2)
{
	switch (action) {
	case GAME_PA_END_TURN: {
		if (pl != S->pl_curr) return -1;
		if (!(S->state == GAME_STATE_NORMAL || S->state == GAME_STATE_SETUP))
			return -1;

		// During each setup turn, the player must place one settlement and
		// one road.
		if (S->state == GAME_STATE_SETUP &&
			!(S->pls[S->pl_curr].n_node == S->n_round+1 &&
			  S->pls[S->pl_curr].n_edge == S->n_round+1) )
		{
			return -1;
		}
		
		S->n_turn++;
		
		bool b_new_round=false;
		if (S->state == GAME_STATE_SETUP && S->n_round%2 == 1) {
			// Reversed order in the second setup round.
			int p = game_player_prev_get(S, S->pl_curr);
			if (p >= S->pl_curr) b_new_round = true;
			S->pl_curr = p;
		}
		else {
			int p = game_player_next_get(S, S->pl_curr);
			if (p <= S->pl_curr) b_new_round = true;
			S->pl_curr = p;
		}

		if (b_new_round) {  // End of round
			S->n_round++;
			
			if (S->state == GAME_STATE_SETUP && S->n_round == 1) {
				S->pl_curr = game_player_prev_get(S, S->pl_curr);  //Last
			}
			else if (S->state == GAME_STATE_SETUP && S->n_round == 2) {
				S->pl_curr = game_player_next_get(S, S->pl_curr);  //First
				S->state = GAME_STATE_NORMAL;

				// Make the last settlements of each player produce
				for (int p=0; p<GAME_PLAYER_COUNT; ++p) {
					if (!S->pls[p].type) continue;
					game_node_produce(S, S->pls[p].last_node);
				}
			}
		}
		
		//TODO: start turn function
		S->pls[S->pl_curr].turn_flags = 0;
		MEM_ZERO(S->pls[S->pl_curr].devcards_lock);
		game_log(S, GAME_LET_TURN_START, S->pl_curr, S->n_turn);
		
		if (S->state == GAME_STATE_NORMAL) {
			S->lot_curr = game_dice_throw(S);
			game_log(S, GAME_LET_DICE_THROW, S->pl_curr, S->lot_curr);
		}
		
		if (S->state == GAME_STATE_SETUP) {
		}
		else if (S->lot_curr == S->rule.thief_num) {
			S->state = GAME_STATE_THIEF_MOVE;

			// Everyone loses half of their resources if they have too much
			for (int p=0; p<GAME_PLAYER_COUNT; ++p) {
				if (!S->pls[p].type) continue;
				if (S->pls[p].n_res >= S->rule.thief_res_min) {
					int n_remove = S->pls[p].n_res / 2;
					//TODO: let players choose
					int cost[GAME_RES__COUNT] = {0};
					game_ai_thief_res(S, p, cost, n_remove);
					assert( game_res_count(cost) == n_remove );
					game_res_sub(S->pls[p].res, cost);
					game_log(S, GAME_LET_RES_PAY, p, cost);
					//TODO: GAME_LET_RES_LOSS ? or pay to ...
				}
			}
			
			// Auto-move thief in solitary
			unsigned n_player=0;
			for (int p=0; p<GAME_PLAYER_COUNT; ++p) n_player += !!S->pls[p].type;
			if (n_player == 1) {
				int i = game_randint(S, 1, vec_count(S->tiles)-2);
				if (i == S->thief_tile) i++;
				game_log(S, GAME_LET_THIEF_MOVE, S->pl_curr, S->thief_tile, i);
				S->thief_tile = i;
				S->state = GAME_STATE_NORMAL;
			}
		}
		else {
			S->state = GAME_STATE_NORMAL;

			// Resource production/harvest
			vec_for(S->tiles, t, 0) {
				if (S->tiles[t].num == S->lot_curr && S->tiles[t].res >= 0 &&
					S->thief_tile != (int)t)
				{
					game_tile_produce(S, t);
				}
			}
		}
		break;
	}
	
	case GAME_PA_BUILD_EDGE: {
		if (pl != S->pl_curr) return -1;
		if (!vec_idx_check(S->edges, p1)) return -1;
		if (!(S->state == GAME_STATE_NORMAL || S->state == GAME_STATE_SETUP))
			return -1;
		
		GamePlayer * player = &S->pls[pl];
		GameEdge * edge = &S->edges[p1];
		if (edge->pl != -1) return -1;

		const int * cost = NULL;
		if (S->state == GAME_STATE_SETUP) {
			if (player->n_node != S->n_round+1) return -1;
			if (player->n_edge != S->n_round) return -1;

			// Check that the edge is connect to the last node
			BoardHexEdge ei = board_hex_edge_get(&S->brd, p1);
			unsigned i;
			for (i=0; i<COUNTOF(ei.nodes); ++i) {
				if (ei.nodes[i] == player->last_node)
					break;
			}
			if (i == COUNTOF(ei.nodes)) return -1;
		} else {
			cost = S->rule.cost_edge;
			if (!game_res_check(player->res, cost)) return -1;

			// Check edge adjacency
			BoardHexEdge ei = board_hex_edge_get(&S->brd, p1);
			unsigned i;
			for (i=0; i<COUNTOF(ei.edges); ++i) {
				if (ei.edges[i] != -1 && S->edges[ei.edges[i]].pl == pl)
					break;
			}
			if (i == COUNTOF(ei.edges)) return -1;
		}
		
		if (cost) {
			game_res_sub(player->res, cost);
			game_log(S, GAME_LET_RES_PAY, pl, cost);
		}

		edge->pl = pl;
		player->n_edge++;
		game_log(S, GAME_LET_BUILD_EDGE, pl, p1);

		game_lroad_update(S);
		break;
	}
	
	case GAME_PA_BUILD_NODE: {
		if (pl != S->pl_curr) return -1;
		if (!vec_idx_check(S->nodes, p1)) return -1;
		if (!(S->state == GAME_STATE_NORMAL || S->state == GAME_STATE_SETUP))
			return -1;

		GamePlayer * player = &S->pls[pl];
		GameNode * node = &S->nodes[p1];
		int lvl = (node->pl == -1) ? 0 : node->lvl+1;
		assert( lvl >= 0 );
		if (!(node->pl == -1 || (node->pl == pl &&
			lvl < GAME_SETTLE_LEVEL_COUNT) )) return -1;
		
		const int * cost = NULL;
		if (S->state == GAME_STATE_SETUP) {
			if (player->n_node != S->n_round) return -1;
			if (node->pl != -1) return -1;
		}
		else {
			cost = S->rule.cost_node[lvl];
			if (!game_res_check(player->res, cost)) return -1;
		}
		
		BoardHexNode ni = board_hex_node_get(&S->brd, p1);
		unsigned i;

		if (S->state != GAME_STATE_SETUP) {
			// Check edge adjacency
			for (i=0; i<COUNTOF(ni.edges); ++i) {
				if (ni.edges[i] != -1 && S->edges[ni.edges[i]].pl == pl)
					break;
			}
			if (i == COUNTOF(ni.edges)) return -1;
		}

		// Check settlement farness
		for (i=0; i<COUNTOF(ni.nodes); ++i) {
			if (ni.nodes[i] != -1 && S->nodes[ni.nodes[i]].pl != -1)
				break;
		}
		if (i != COUNTOF(ni.nodes)) return -1;
		
		if (cost) {
			game_res_sub(player->res, cost);
			game_log(S, GAME_LET_RES_PAY, pl, cost);
		}

		node->pl = pl;
		node->lvl = lvl;
		
		if (lvl > 0) player->n_node_lvl[lvl-1]--;  //Upgrade
		else player->n_node++;
		player->n_node_lvl[lvl]++;
		player->last_node = p1;
		game_log(S, GAME_LET_BUILD_NODE, pl, p1, lvl);
		
		if (node->tr >= 0)
			game_player_troute_add(S, pl, node->tr);
		break;
	}

	case GAME_PA_DEVCARD_BUY: {
		if (pl != S->pl_curr) return -1;
		if (!(S->state == GAME_STATE_NORMAL)) return -1;
		GamePlayer * player = &S->pls[pl];
		
		unsigned pos=0;
		for (pos=0; pos<COUNTOF(player->devcards); ++pos) {
			if (player->devcards[pos] == -1)
				break;
		}
		if (pos == COUNTOF(player->devcards)) return -1;
		
		const int * cost = S->rule.cost_devcard;
		if (!game_res_check(player->res, cost)) return -1;
		game_res_sub(player->res, cost);
		game_log(S, GAME_LET_RES_PAY, pl, cost);

		player->devcards[pos] = 0;  //TODO: multiple card types and random draw
		player->devcards_lock[pos] = 1;
		game_log(S, GAME_LET_DEVCARD_BUY, pl);
		break;
	};

	case GAME_PA_DEVCARD_USE: {
		if (pl != S->pl_curr) return -1;
		if (!(S->state == GAME_STATE_NORMAL)) return -1;
		GamePlayer * player = &S->pls[pl];
		if (player->turn_flags & GAME_PTF_DEVCARD_USED) return -1;

		if (!(0 <= p1 && (unsigned)p1 < COUNTOF(player->devcards))) return -1;
		if (player->devcards[p1] == -1) return -1;
		if (player->devcards_lock[p1]) return -1;

		if (player->devcards[p1] == 0) {  // Army
			player->n_army++;
			int n = player->n_army;
			if (n >= S->rule.larmy_min && n > S->larmy_n) {
				S->larmy_n = n;
				S->larmy_pl = pl;
				//TODO: log change
			}
			S->state = GAME_STATE_THIEF_MOVE;
			game_log(S, GAME_LET_DEVCARD_USE, pl, 0);
		}

		player->devcards[p1] = -1;
		player->turn_flags |= GAME_PTF_DEVCARD_USED;
		break;
	}
	
	case GAME_PA_TRADE_RES: {
		if (pl != S->pl_curr) return -1;
		if (!(S->state == GAME_STATE_NORMAL)) return -1;
		if (!(0 <= p1 && p1 < GAME_RES__COUNT)) return -1;
		if (!(0 <= p2 && p2 < GAME_RES__COUNT)) return -1;
		if (p1 == p2) return -1;
		
		GamePlayer * player = &S->pls[pl];
		int q1 = player->res_tq[p1];
		if (!(player->res[p1] >= q1)) return -1;
		player->res[p1] -= q1;
		player->res[p2] += 1;
		game_log(S, GAME_LET_RES_TRADE, pl, p1, p2, q1, 1);
		break;;
	}

	case GAME_PA_THIEF_MOVE: {
		if (pl != S->pl_curr) return -1;
		if (!vec_idx_check(S->tiles, p1)) return -1;
		if (!(S->state == GAME_STATE_THIEF_MOVE)) return -1;
		if (p1 == S->thief_tile) return -1;
		game_log(S, GAME_LET_THIEF_MOVE, S->pl_curr, S->thief_tile, p1);
		S->thief_tile = p1;
		
		// Steal from the richest player with a settlement in the tile
		//TODO: allow to choose target player
		int best_n_res = 0;
		int best_pl = -1;
		const BoardHexTile ti = board_hex_tile_get(&S->brd, p1);
		for (int j=0; j<6; ++j) {
			int n = ti.nodes[j];
			int pln = S->nodes[n].pl;
			if (pln != -1 && pln != pl && pln != best_pl) {
				int n_res = S->pls[pln].n_res;
				if (n_res > 0 && n_res > best_n_res) {
					best_n_res = n_res;
					best_pl = pln;
				}
			}
		}
		// Steal a random resource
		if (best_pl != -1) {
			int i = game_randint(S, 0, best_n_res-1);
			for (int r=0; r<GAME_RES__COUNT; ++r) {
				i -= S->pls[best_pl].res[r];
				if (i < 0) {
					S->pls[best_pl].res[r]--;
					S->pls[pl].res[r]++;
					game_log(S, GAME_LET_RES_STEAL, pl, best_pl, r, 1);
					break;
				}
			}
		}

		S->state = GAME_STATE_NORMAL;
		break;
	}
	}

	game_players_update(S);
	return 1;
}
