/* Copyright 2025-2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: Zlib
 */
#include "textrender.h"
#include "image_draw.h"

#ifndef TEXTRENDER_NO_UNICODE
#include "unicode.h"
#endif

#define MIN_(X,Y) ((X)<(Y) ? (X) : (Y))
#define MAX_(X,Y) ((X)>(Y) ? (X) : (Y))

typedef unsigned char Uint8;

/* Text decoder */

static
unsigned text_decode_next(const char** ptext, const char* text_end, int flags)
{
#ifndef TEXTRENDER_NO_UNICODE
	if (!(flags & TR_RF_NO_UTF8))
		return utf8_decode_next(ptext, text_end);
#endif
	if (!(*ptext < text_end)) return 0;
	unsigned c = **ptext;
	(*ptext)++;
	return c;
}

/* Writer
 * Determines the next glyph and where it should be drawn.
 */

typedef struct TR_Writer {
	const TextRenderGlyph *	g;
	ImgPoint				p0;
	ImgPoint				pen;
	ImgRectP				bbox;
} TR_Writer;

static
void trwriter_init(TR_Writer* W, TextRender* S, const ImgPoint origin)
{
	*W = (TR_Writer){0};
	W->pen = origin;
	W->bbox.x1 = W->bbox.x2 = W->pen.x;
	W->bbox.y1 = W->bbox.y2 = W->pen.y;
	W->pen.y += S->metrics.ascent;
}

static
void trwriter_next(TR_Writer* W, TextRender* S, unsigned c, int flags)
{
	unsigned prev_glyph = 0;
	if (W->g) {
		prev_glyph = W->g->glyph_index;
		W->g = 0;
	}

	switch (c) {
	case 0:
		return;
	case '\r':
		W->pen.x = W->bbox.x1;
		return;
	case '\n':
		W->pen.x = W->bbox.x1;
		W->pen.y += S->metrics.height;
		return;
	case '\t':
		W->pen.x = ((W->pen.x + S->metrics.tab_width - 1)
			/ S->metrics.tab_width) * S->metrics.tab_width;
		return;
	}

	W->g = S->cls->glyph_get(S, c);
	if (!W->g) return;

	// Optional: kerning
	if (!(flags & TR_RF_NO_KERNING) && prev_glyph && S->cls->kerning_get)
	{
		ImgPoint k = S->cls->kerning_get(S, prev_glyph, W->g->glyph_index);
		W->pen.x += k.x;
	}

	// Origin of the glyph on screen
	W->p0.x = W->pen.x + W->g->bitmap_left;
	W->p0.y = W->pen.y - W->g->bitmap_top;
		
	// Expand the boundaries of the text
	// Some styles can widen the glyphs (i.e: slanted)
	int xe = W->pen.x +
		MAX_(W->g->advance_x, W->g->bitmap_left + W->g->bitmap_width);
		//TODO: +1 ?
	if (W->bbox.x2 < xe)
		W->bbox.x2 = xe;
	
	int ye = W->pen.y + S->metrics.height - S->metrics.ascent;
	if (W->bbox.y2 < ye)
		W->bbox.y2 = ye;

	// Advance the pen position
	W->pen.x += W->g->advance_x;
}

/* Text Renderer */

void textrender_destroy(TextRender* obj)
{
	if (obj && obj->cls && obj->cls->destroy)
		obj->cls->destroy(obj);
}

int textrender_size_get(TextRender* tr, ImgPoint* size,
	const char* text, const char* text_end, int flags)
{
	flags |= tr->c.rflags;

	if (!text_end)
		for (text_end=text; *text_end; ++text_end);
	
	TR_Writer w;
	trwriter_init(&w, tr, (ImgPoint){0,0});

	unsigned c;
	while ((c = text_decode_next(&text, text_end, flags)))
		trwriter_next(&w, tr, c, flags);
	
	if (size) {
		*size = (ImgPoint){ w.bbox.x2 - w.bbox.x1, w.bbox.y2 - w.bbox.y1 };
	}

	return 0;
}

int textrender_render(TextRender* tr, Image* img, ImgRect* dstrect,
	const char* text, const char* text_end, const ImgColor col, int flags)
{
	if (!text) {
		if (dstrect) {
			dstrect->w = 0;
			dstrect->h = 0;
		}
		return 0;
	}

	flags |= tr->c.rflags;

	if (!text_end)
		for (text_end=text; *text_end; ++text_end);

	ImgRect rect = {0};
	if (dstrect) rect = *dstrect;
	else if (img) { rect.w = img->w; rect.h = img->h; }
	
	// Determine the position of aligned text
	if (img && flags & TR_RF_MASK_ALIGN)
	{
		ImgPoint sz;
		int r = textrender_size_get(tr, &sz, text, text_end, flags);
		if (r<0) return r;

		if (flags & TR_RF_ALIGN_RIGHT)
			rect.x += rect.w - sz.x;
		else if (flags & TR_RF_ALIGN_HCENTER)
			rect.x += (rect.w - sz.x) / 2;

		if (flags & TR_RF_ALIGN_BOTTOM)
			rect.y += rect.h - sz.y;
		else if (flags & TR_RF_ALIGN_VCENTER)
			rect.y += (rect.h - sz.y) / 2;

		//TODO: per line alignment ?
	}

	// Walk over the text characters and render
	TR_Writer w={0};
	trwriter_init(&w, tr, img_rect_po(&rect));

	unsigned c;
	while ((c = text_decode_next(&text, text_end, flags)))
	{
		trwriter_next(&w, tr, c, flags);
		if (!w.g) continue;
		if (!img) continue;

		Image src = { .data=w.g->bitmap_data, .w=w.g->bitmap_width,
			.h=w.g->bitmap_height, .bypp=1, .pitch=w.g->bitmap_width,
			.format=IMG_FORMAT_GRAY };

		imgdraw_alpha_mask(img, w.p0, &src, col, flags & TR_RF_FAST_BLIT);
	}

	// Return the bounding box
	if (dstrect) *dstrect = img_rect_p_to_s(&w.bbox);
	
	return 0;
}

int textrender_render_new(TextRender* obj, Image* img, ImgFormat fmt,
	const char* text, const char* text_end,
	const ImgColor col_fg, const ImgColor col_bg, int flags)
{
	flags &= ~TR_RF_MASK_ALIGN;  //Alignment is meaningless here

	ImgPoint sz;
	int r = textrender_size_get(obj, &sz, text, text_end, flags);
	if (r<0) return r;

	if (!fmt) fmt = (col_bg.a == 255) ? IMG_FORMAT_RGB : IMG_FORMAT_RGBA;
	r = img_resize(img, sz.x, sz.y, fmt, 0);
	if (r<0) return r;

	img_fill(img, col_bg);

	return textrender_render(obj, img, NULL, text, text_end, col_fg, flags);
}

#define MEAN_WIDTH_GET(TXT) \
	textrender_size_get(obj, &sz, TXT, TXT+sizeof(TXT), 0) \
	? 1 : sz.x / sizeof(TXT)

void textrender_metrics_update(TextRender* obj)
{
	ImgPoint sz = {0,0};
	obj->metrics.mean_width = MEAN_WIDTH_GET(
		"abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPRSTUVWXYZ0123456789");
	obj->metrics.num_width  = MEAN_WIDTH_GET("0123456789");
	obj->metrics.m_width    = MEAN_WIDTH_GET("m");
	obj->metrics.tab_width  = obj->metrics.mean_width * 8;
}

/* "C" text renderer */

const TextRenderGlyph* textrender_c_glyph_get(TextRenderC* obj, unsigned cp)
{
	if (obj->cp_first <= cp && cp <= obj->cp_last)
		return &obj->glyphs[ cp - obj->cp_first ];
	else
		return &obj->glyphs[0];
}

const TextRenderClass textrender_class_c = {
	(const TextRenderGlyph* (*)(TextRender*, unsigned)) textrender_c_glyph_get,
	NULL,
	NULL,
};

/*TextRender* textrender_c_new(const TextRenderC* src)
{
	TextRenderC* obj = malloc(sizeof(TextRenderC));
	if (obj) {
		*obj = *src;
		textrender_metrics_update((TextRender*)obj);
	}
	return (TextRender*)obj;
}*/
