/* Copyright 2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: Zlib
 *
 * Simple sound synthetizer that can be programmed with a string.
 *
 * Commands start with a letter o a symbol.
 * Whitespace is ignored and can be used to separate commands.
 * Commands that change parameters remain in effect until the end the group.
 *
 * Commands:
 *   N     Play MIDI note number N (0-127). 69 = A4 = 440Hz.
 *   -     Rest (play nothing for the duration of one note).
 *   bN    Change to N beats per minute. N notes fill in one minute.
 *   '     Halves the durection of next note. Can be repeated.
 *   _     Doubles the durection of next note. Can be repeated.
 *   |     Go back in time to the previous note.
 *         This allow chord to be made.
 *         It is affected by ', allowing partial overlap.
 *   wI    Change wave form or instrument to I. Options for I:
 *           sin   Sine wave
 *           tri   Triangle wave
 *           sqr   Square wave
 *           saw   Sawtooth
 *           sisi  sin(wt + sin(wt))
 *   nI    Play noise of type I. Options:
 *           w  White noise.
 *           p  Pink noise
 *   aN    Changes the amplitude (volume). N=0-100.
 *   eLN   Envelope options. For L:
 *           a   N% attack duration 
 *           d   N% decay duration 
 *           s   N% sustain duration 
 *           A   N% attack amplitude 
 *           o   N% overlap between consecutive notes
 *           c   Curve type:
 *                 l   Linear
 *                 e   Exponential
 *   #     Line comment. Ignores everything until the next newline.
 *
 * Not yet implemented:
 *   {     Begin group.
 *   }     End group.
 *   (     Begin chord (notes played simultaneously).
 *   )     End chord.
 *   sI    Play sample with id I.
 *
 * Examples:
 *   "69 70 71"
 */
#pragma once
#include "sound.h"
#include "strslice.h"

typedef struct SoundSynth {
	struct SoundSynthSample {
		unsigned id;
		Sound snd;
	} * samples;  //vector, sorted
} SoundSynth;

void sndsynth_free(SoundSynth* S);

// Execute the commands in <slices> and store the generated sound in <out>.
// If <out> already has a format, it will be kept.
// Return the length in sample or a negative error code.
int sndsynth_synth_m(SoundSynth* S, Sound* out, unsigned n_slice,
	const StrSlice* slices, int flags);

enum {
	SND_SYNTH_F_NO_OUTPUT = 1,
};

static inline
int sndsynth_synth(SoundSynth* S, Sound* out, const char* cmds)
{
	StrSlice sl = strsl_fromz(cmds);
	return sndsynth_synth_m(S, out, 1, &sl, 0);
}

static inline
int sndsynth_length_get(SoundSynth* S, unsigned srate, const char* cmds)
{
	Sound snd = { .freq=srate };
	StrSlice sl = strsl_fromz(cmds);
	return sndsynth_synth_m(S, &snd, 1, &sl, SND_SYNTH_F_NO_OUTPUT);
}
