/* Copyright 2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: Zlib
 */
#include "sound_synth.h"
#include "vector.h"
#include <math.h>

void sndsynth_free(SoundSynth* S)
{
	vec_for(S->samples, i, 0) {
		snd_free(&S->samples[i].snd);
	}
	vec_free(S->samples);
}

/* Text decoder */

typedef struct TextDecodePos {
	const char *cur, *end;  // Current slice
	const StrSlice *texts;  // Pending slices
	unsigned n_text;
	unsigned unget;
} TextDecodePos;

static
void text_decode_init(TextDecodePos* pos, unsigned n_text, const StrSlice* texts)
{
	const char *cur=NULL, *end=NULL;
	if (n_text > 0) {
		cur = strsl_begin(texts[0]);
		end = strsl_end  (texts[0]);
	}
	*pos = (TextDecodePos){ cur, end, texts+1, n_text-1 };
}

static
unsigned text_decode_get(TextDecodePos* pos)
{
	if (pos->unget) {
		unsigned c = pos->unget;
		pos->unget = 0;
		return c;
	}

	while (!(pos->cur < pos->end)) {
		if (!pos->n_text) return 0;
		pos->cur = strsl_begin(pos->texts[0]);
		pos->end = strsl_end  (pos->texts[0]);
		pos->texts++;
		pos->n_text--;
	}

	unsigned c = (unsigned char) *pos->cur;
	pos->cur++;
	return c;
}

static
void text_decode_unget(TextDecodePos* pos, unsigned c)
{
	assert( pos->unget == 0 );
	pos->unget = c;
}

static inline
bool chr_digit_is(unsigned c) {
	return '0' <= c && c <= '9';
}

static inline
bool chr_alphanum_is(unsigned c) {
	return ('0' <= c && c <= '9') ||
	       ('A' <= c && c <= 'Z') ||
	       ('a' <= c && c <= 'z') ;
}

//static inline
//bool chr_space_is(unsigned c) {
//	return (c == ' ' || c == '\t' || c == '\r' || c == '\n');
//}

static
unsigned text_decode_number_get(TextDecodePos* pos, unsigned c)
{
	unsigned v = c ? c - '0' : 0;
	while (chr_digit_is(c = text_decode_get(pos)))
		v = v*10 + (c - '0');
	text_decode_unget(pos, c);
	return v;
}

static
void text_decode_label_get(TextDecodePos* pos, unsigned max, char* label)
{
	unsigned i=0;
	for (; i<max; ++i) {
		unsigned c = text_decode_get(pos);
		if (!chr_alphanum_is(c)) {
			text_decode_unget(pos, c);
			break;
		}
		label[i] = c;
	}
	for (; i<max; ++i) label[i] = 0;
}

/* Synth context and commands parser */

enum {
	SSC_ACTION_END		= 0,
	SSC_ACTION_IGNORE	= 1,
	SSC_ACTION_NOTE		= 2,
	SSC_ACTION_NOISE	= 3,
};

typedef struct SoundSynthCtx {
	// "Style" / "Formating"
	struct SoundSynthCtxState {
		unsigned len_beat;
		double freq, amp;
		SndWaveType wave_type;
		SndNoiseType noise_type;
		SndCurveType fade_curve;
		double f_dur_next;
		// Envelope / ADSR
		double f_attack, f_decay, f_sustain, f_overlap;
		double m_attack;
	} s, *sstack;

	// Current note values
	unsigned len, pos, pos_next, pos_end;
	double t_attack, t_decay, t_sustain;

	// Config
	unsigned srate;
} SoundSynthCtx;

void ssctx_free(SoundSynthCtx* C)
{
	vec_free(C->sstack);
}

void ssctx_init(SoundSynthCtx* C, unsigned srate)
{
	C->srate = srate;
	C->s.len_beat = srate;  //1s, 60bpm
	C->s.freq = 440.0;
	C->s.amp = 0.5;
	C->s.wave_type = SND_WT_SINE;
	C->s.noise_type = SND_NT_WHITE;
	C->s.fade_curve = SND_CT_LINEAR;
	C->s.f_dur_next = 1;
	C->s.m_attack = 1;
}

static
void ssctx_pos_advance(SoundSynthCtx* C)
{
	C->len = C->s.len_beat;
	if (C->s.f_dur_next != 1) {
		C->len = C->len * C->s.f_dur_next;
		C->s.f_dur_next = 1;
	}

	C->t_attack  = C->len * C->s.f_attack  / C->srate;
	C->t_decay   = C->len * C->s.f_decay   / C->srate;
	C->t_sustain = C->len * C->s.f_sustain / C->srate;
	
	C->pos = C->pos_next;
	C->pos_next += C->len;
	
	if (C->s.f_overlap > 0)
		C->len = C->len * (1 + C->s.f_overlap);
	
	MAXSET(C->pos_end, C->pos + C->len);
}

static
int ssctx_next(SoundSynthCtx* C, TextDecodePos* pos)
{
	int R=-1;

	unsigned c = text_decode_get(pos);
	if (c == 0) return 0;

	switch (c) {
	case ' ': case '\t': case '\r': case '\n':
		R = SSC_ACTION_IGNORE;
		break;
	case '#':
		while ((c = text_decode_get(pos)) && c != '\r' && c != '\n') ;
		R = SSC_ACTION_IGNORE;
		break;
	case '{':
		vec_push(C->sstack, C->s);
		R = SSC_ACTION_IGNORE;
		break;
	case '}':
		if (!vec_count(C->sstack)) RETURN(-1);
		C->s = vec_pop(C->sstack);
		R = SSC_ACTION_IGNORE;
		break;
	case '0': case '1': case '2': case '3': case '4':
	case '5': case '6': case '7': case '8': case '9': {
		unsigned n = text_decode_number_get(pos, c);
		C->s.freq = 440.0 * exp2((n - 67.0)/12.0);
		ssctx_pos_advance(C);
		R = SSC_ACTION_NOTE;
		break;
	}
	case 'n': {
		c = text_decode_get(pos);
		if      (c == 'w') C->s.noise_type = SND_NT_WHITE;
		else if (c == 'p') C->s.noise_type = SND_NT_PINK;
		ssctx_pos_advance(C);
		R = SSC_ACTION_NOISE;
		break;
	}
	case '-': {
		ssctx_pos_advance(C);
		R = SSC_ACTION_IGNORE;
		break;
	}
	case '|': {
		if (C->s.f_dur_next != 1) {
			C->pos_next -= (int)(C->pos_next - C->pos) * C->s.f_dur_next;
			if ((int)C->pos_next < 0) C->pos_next = 0;
			C->s.f_dur_next = 1;
		} else
			C->pos_next = C->pos;
		R = SSC_ACTION_IGNORE;
		break;
	}
	case '\'': {
		C->s.f_dur_next /= 2;
		R = SSC_ACTION_IGNORE;
		break;
	}
	case '_': {
		C->s.f_dur_next *= 2;
		R = SSC_ACTION_IGNORE;
		break;
	}
	case 'b': {
		unsigned n = text_decode_number_get(pos, 0);
		C->s.len_beat = C->srate * 60.0 / n;
		R = SSC_ACTION_IGNORE;
		break;
	}
	case 'w': {
		char label[4];
		text_decode_label_get(pos, 4, label);
		if      (!memcmp(label, "sin" , 4))  C->s.wave_type = SND_WT_SINE;
		else if (!memcmp(label, "tri" , 4))  C->s.wave_type = SND_WT_TRIANGLE;
		else if (!memcmp(label, "sqr" , 4))  C->s.wave_type = SND_WT_SQUARE;
		else if (!memcmp(label, "saw" , 4))  C->s.wave_type = SND_WT_SAWTOOTH;
		else if (!memcmp(label, "sisi", 4))  C->s.wave_type = SND_WT_SINE_PM_SINE;
		R = SSC_ACTION_IGNORE;
		break;
	}
	case 'a': {
		unsigned n = text_decode_number_get(pos, 0);
		C->s.amp = n / 100.0;
		R = SSC_ACTION_IGNORE;
		break;
	}
	case 'e': {
		unsigned n;
		c = text_decode_get(pos);
		switch (c) {
		case 'a':
			n = text_decode_number_get(pos, 0);
			C->s.f_attack = n / 100.0;
			break;
		case 'd':
			n = text_decode_number_get(pos, 0);
			C->s.f_decay = n / 100.0;
			break;
		case 's':
			n = text_decode_number_get(pos, 0);
			C->s.f_sustain = n / 100.0;
			break;
		case 'A':
			n = text_decode_number_get(pos, 0);
			C->s.m_attack = n / 100.0;
			break;
		case 'o':
			n = text_decode_number_get(pos, 0);
			C->s.f_overlap = n / 100.0;
			break;
		case 'c':
			c = text_decode_get(pos);
			if      (c == 'l') C->s.fade_curve = SND_CT_LINEAR;
			else if (c == 'e') C->s.fade_curve = SND_CT_EXP;
			break;
		}
		R = SSC_ACTION_IGNORE;
		break;
	}
	}

end:
	return R;
}

int sndsynth_synth_m(SoundSynth* S, Sound* out, unsigned n_slice,
	const StrSlice* slices, int flags)
{
	int R=1, r;
	Sound tmpsnd={0}, vw={0};

	// High quality defaults
	if (!out->freq) out->freq = 48000;
	if (!out->ch) out->ch = 1;
	if (!out->format) out->format = SND_FMT_F32;
	out->len = 0;

	if (!(flags & SND_SYNTH_F_NO_OUTPUT)) {
		r = sndsynth_synth_m(S, out, n_slice, slices, SND_SYNTH_F_NO_OUTPUT);
		if (r < 0) return r;
		snd_resize(out, r, 0, 0, 0);
		snd_zero(out);
	}

	SoundSynthCtx ctx={0};
	ssctx_init(&ctx, out->freq);
	
	TextDecodePos pos={0};
	text_decode_init(&pos, n_slice, slices);

	while ((r = ssctx_next(&ctx, &pos)) > 0) {
		if (r == SSC_ACTION_IGNORE) continue;
		if (flags & SND_SYNTH_F_NO_OUTPUT) continue;
		
		// Generate sample
		snd_resize(&tmpsnd, ctx.len, out->ch, out->freq, out->format);
		snd_zero(&tmpsnd);
		
		if (r == SSC_ACTION_NOISE)
			snd_noise_add(&tmpsnd, ctx.s.noise_type, ctx.s.amp);
		else
			snd_wave_add(&tmpsnd, ctx.s.wave_type, ctx.s.freq, ctx.s.amp, 0);
		
		snd_adsr(&tmpsnd, ctx.t_attack, ctx.t_decay, ctx.t_sustain,
			ctx.s.m_attack, 1, 1, ctx.s.fade_curve);

		// Add sample
		assert( ctx.pos + ctx.len <= out->len );
		vw = snd_slice_get(out, ctx.pos, ctx.pos + ctx.len);
		snd_add(&vw, &tmpsnd, 1, 1);
	}
	if (r<0) R=r;
	else R = ctx.pos_end;

	snd_free(&tmpsnd);
	ssctx_free(&ctx);
	return R;
}
