/* Copyright 2024, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: Zlib
 *
 * Mersenne Twister random number generator.
 *
 * Comparison with std::mt19937 (glibc 2.40+r16+gaa533d58ff-2):
 *  uint32 matches operator()
 *  randint almost matches uniform_int_distribution<uint32_t>
 *  rand matches uniform_real_distribution<double>(0,1)
 *  randn matches normal_distribution(0,1)
 */
#pragma once
#include <stdint.h>
#include <assert.h>

typedef struct {
    uint32_t array[624];
    uint32_t index;
	double saved;
} MT19937State;

// Suggested initial seed: 19650218UL
void mt19937_init(MT19937State* S, uint32_t seed);

// Returns a random 32bits integer
uint32_t mt19937_uint32(MT19937State* S);

// Returns a random 64bits integer
static inline
uint64_t mt19937_uint64(MT19937State* S) {
	return (uint64_t)mt19937_uint32(S) + ((uint64_t)mt19937_uint32(S) << 32);
}

// Returns a random integer in [low,high]
static inline
int32_t mt19937_randint(MT19937State* S, int32_t low, int32_t high) {
	assert( low <= high );
	uint64_t z = mt19937_uint32(S),
	         m = high - low;
	return (int32_t)((z*m+0x7fffffff)/0xffffffff) + low;
}

// Returns a random number in [0,1)
static inline
float mt19937_randf(MT19937State* S) {
	//return (float)mt19937_uint32(S) * (1.0f / (~(uint32_t)0));
	return (float)mt19937_uint32(S) * 2.3283064365386963e-10;  //1/2^32
}

// Returns a random number in [0,1)
static inline
double mt19937_rand(MT19937State* S) {
	//return (double)mt19937_uint64(S) * (1.0 / (~(uint64_t)0));
	return (double)mt19937_uint64(S) * 5.421010862427522e-20;  //1/2^64
}

// Returns a normally-distributed random number
// Uses Box-Muller method
double mt19937_randn_bm(MT19937State* S);

// Returns a normally-distributed random number
// Uses Marsaglia polar method
double mt19937_randn_mp(MT19937State* S);

#define mt19937_randn mt19937_randn_mp
