/* Copyright 2025-2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: Zlib
 */
#include "richtext.h"
#include "bisect.h"
#include "vector.h"
#include "unicode.h"
#include "image_draw.h"

void richtext_free(RichTextEngine* S)
{
	vec_free(S->images);
	vec_free(S->colors);
	vec_free(S->fonts);
}

void richtext_default_colors_set(RichTextEngine* S)
{
	richtext_color_set(S, 'B', (ImgColor){0,0,0,255});
	richtext_color_set(S, 'G', (ImgColor){127,127,127,255});
	richtext_color_set(S, 'w', (ImgColor){255,255,255,255});
	richtext_color_set(S, 'r', (ImgColor){255,31,31,255});
	richtext_color_set(S, 'g', (ImgColor){31,200,31,255});
	richtext_color_set(S, 'b', (ImgColor){31,31,255,255});
	richtext_color_set(S, 'y', (ImgColor){31,200,200,255});
}

void richtext_font_set(RichTextEngine* S, unsigned id, TextRender* font)
{
	BISECT_RIGHT_DECL(found, idx, 0, vec_count(S->fonts), S->fonts[i_].id - id);
	if (!found) vec_insert(S->fonts, idx, 1, NULL);
	S->fonts[idx] = (struct RichTextFont){ id, font };
	if (!S->c.def_font) S->c.def_font = font;
}

TextRender* richtext_font_get(RichTextEngine* S, unsigned id)
{
	BISECT_RIGHT_DECL(found, idx, 0, vec_count(S->fonts), S->fonts[i_].id - id);
	if (found) return S->fonts[idx].font;
	return S->c.def_font;
}

void richtext_color_set(RichTextEngine* S, unsigned id, const ImgColor color)
{
	BISECT_RIGHT_DECL(found, idx, 0, vec_count(S->colors), S->colors[i_].id - id);
	if (!found) vec_insert(S->colors, idx, 1, NULL);
	S->colors[idx] = (struct RichTextColor){ id, color };
	if (!S->c.def_color.a) S->c.def_color = color;
}

ImgColor richtext_color_get(RichTextEngine* S, unsigned id)
{
	BISECT_RIGHT_DECL(found, idx, 0, vec_count(S->colors), S->colors[i_].id - id);
	if (found) return S->colors[idx].color;
	return S->c.def_color;
}

void richtext_image_set(RichTextEngine* S, unsigned id, const Image* image)
{
	BISECT_RIGHT_DECL(found, idx, 0, vec_count(S->images), S->images[i_].id - id);
	if (!found) vec_insert(S->images, idx, 1, NULL);
	S->images[idx] = (struct RichTextImage){ id, image };
}

const Image* richtext_image_get(RichTextEngine* S, unsigned id)
{
	BISECT_RIGHT_DECL(found, idx, 0, vec_count(S->images), S->images[i_].id - id);
	if (found) return S->images[idx].image;
	return NULL;
}

int richtext_size_get(RichTextEngine* S, const StrSlice text, ImgPoint* psz)
{
	ImgRect rect={0,0,0,0};
	TRYR( richtext_render_m(S, NULL, &rect, 1, &text) );
	if (psz) *psz = (ImgPoint){ rect.w, rect.h };
	return 0;
}

int richtext_render(RichTextEngine* S, Image* img, ImgRect* rect,
	const StrSlice text)
{
	return richtext_render_m(S, img, rect, 1, &text);
}

/* Text decoder */

typedef struct TextDecodePos {
	const char *cur, *end;  // Current slice
	const StrSlice *texts;  // Pending slices
	unsigned n_text;
} TextDecodePos;

static
void text_decode_init(TextDecodePos* pos, unsigned n_text, const StrSlice* texts)
{
	const char *cur=NULL, *end=NULL;
	if (n_text > 0) {
		cur = strsl_begin(texts[0]);
		end = strsl_end  (texts[0]);
	}
	*pos = (TextDecodePos){ cur, end, texts+1, n_text-1 };
}

static
unsigned text_decode_next(TextDecodePos* pos, int flags)
{
	while (!(pos->cur < pos->end)) {
		if (!pos->n_text) return 0;
		pos->cur = strsl_begin(pos->texts[0]);
		pos->end = strsl_end  (pos->texts[0]);
		pos->texts++;
		pos->n_text--;
	}

	if (!(flags & RTE_CF_NO_UTF8)) {
		return utf8_decode_next(&pos->cur, pos->end);
	} else {
		unsigned c = (unsigned char) *pos->cur;
		pos->cur++;
		return c;
	}
}

/* Render context and style parser */

enum {
	RT_RESULT_ERROR			= -1,
	RT_RESULT_END			= 0,
	RT_RESULT_SKIP			= 1,
	RT_RESULT_DRAW_DIRECT	= 2,
	RT_RESULT_DRAW_MASK		= 3,
};

typedef struct RichTextPen {
	ImgPoint pos, prev;
	ImgRectP sbox;  //Size box
	TextDecodePos text_begin;
	unsigned i_sstack;
	int flags;
} RichTextPen;

enum {
	RTP_F_BASELINE		= 0x0001,
	RTP_F_NO_WRAP		= 0x0002,
	
	RTP_F_ALIGN_TOP		= 0,
	RTP_F_ALIGN_LEFT	= 0,
	RTP_F_ALIGN_BOTTOM	= 0x0100,
	RTP_F_ALIGN_RIGHT	= 0x0200,
	RTP_F_ALIGN_VCENTER	= 0x0400,
	RTP_F_ALIGN_HCENTER	= 0x0800,
	RTP_F_ALIGN_V_MASK	= RTP_F_ALIGN_BOTTOM | RTP_F_ALIGN_VCENTER,
	RTP_F_ALIGN_H_MASK	= RTP_F_ALIGN_RIGHT  | RTP_F_ALIGN_HCENTER,
	RTP_F_ALIGN_MASK	= RTP_F_ALIGN_V_MASK | RTP_F_ALIGN_H_MASK,
	//RTP_F_ALIGN_DONE	= 0x1000,
};

typedef struct RichTextStyle {
	TextRender * font;
	ImgColor col_fg, col_bg;
	ImgRectP bbox;
	int flags;
} RichTextStyle;

enum {
	RTS_F_PHANTOM	= 0x0001,
};

typedef struct RichTextCtx {
	Image glyph;  //Glyph/image to draw
	ImgPoint p0;  //Draw origin of the glyph
	
	unsigned glyph_idx_prev;  //Previous glyph index, used for kerning

	RichTextPen p;  //Current pen position and size box
	RichTextPen * pstack;  //vector

	RichTextStyle s;  //Current style
	RichTextStyle * sstack;  //vector

	unsigned ec;  //Escape character
} RichTextCtx;

static
void rtctx_free(RichTextCtx* R)
{
	vec_free(R->pstack);
	vec_free(R->sstack);
}

static
void rtctx_init(RichTextCtx* C, RichTextEngine* E, const ImgRect rect)
{
	C->s.font = E->c.def_font;
	C->s.col_fg = E->c.def_color;
	C->s.bbox = img_rect_s_to_p(&rect);
	
	C->p.prev = C->p.pos = img_rect_po(&rect);
	C->p.sbox.x1 = C->p.sbox.x2 = C->p.pos.x;
	C->p.sbox.y1 = C->p.sbox.y2 = C->p.pos.y;
	// Text wrapping disabled by default because it done at glyph level and does
	// not looks so good.
	C->p.flags |= RTP_F_NO_WRAP;

	C->ec = IFFALSE(E->c.ec, 0x7f);
}

static
void rtctx_align_begin(RichTextCtx* C, TextDecodePos* pos, int aflags)
{
	// Aligment already calculated for this block
	//if (C->p.flags & RTP_F_ALIGN_DONE) return;

	// No bounding box defined -> no alignment possible
	if (C->s.bbox.x1 == C->s.bbox.x2) aflags &= ~RTP_F_ALIGN_H_MASK;
	if (C->s.bbox.y1 == C->s.bbox.y2) aflags &= ~RTP_F_ALIGN_V_MASK;
	
	// Aligment starting, no text yet: just update
	if (C->p.flags & RTP_F_ALIGN_MASK && C->p.sbox.x1 == C->p.sbox.x2) {
		if (!aflags) {
			assert( vec_count(C->pstack) );
			C->p = vec_pop(C->pstack);
		} else {
			C->p.flags &= ~RTP_F_ALIGN_MASK;
			C->p.flags |= aflags;
			C->p.text_begin = *pos;
			assert( vec_idx_check(C->sstack, C->p.i_sstack) );
			C->sstack[ C->p.i_sstack ] = C->s;
		}
		return;
	}
	
	bool v_same =
		(C->p.flags & RTP_F_ALIGN_V_MASK) == (aflags & RTP_F_ALIGN_V_MASK);
	bool h_same =
		(C->p.flags & RTP_F_ALIGN_H_MASK) == (aflags & RTP_F_ALIGN_H_MASK);

	// No change
	if (v_same && h_same) return;
	
	// Finish the vertical size calculation before changing the h alignment
	if (v_same && C->p.flags & RTP_F_ALIGN_V_MASK) return;

	// No aligment required
	if (!aflags) return;

	vec_push(C->pstack, C->p);
	C->p.flags &= ~RTP_F_ALIGN_MASK;
	C->p.flags |= aflags;
	C->p.prev = C->p.pos;
	C->p.sbox.x1 = C->p.sbox.x2 = C->p.pos.x;
	C->p.sbox.y1 = C->p.sbox.y2 = C->p.pos.y;
	C->p.text_begin = *pos;
	C->p.i_sstack = vec_count(C->sstack);
	vec_push(C->sstack, C->s);
}

static
int rtctx_align_end(RichTextCtx* C, TextDecodePos* pos)
{
	if (!vec_count(C->pstack)) return 0;

	*pos = C->p.text_begin;
	
	assert( vec_idx_check(C->sstack, C->p.i_sstack) );
	C->s = C->sstack[ C->p.i_sstack ];
	vec_resize(C->sstack, C->p.i_sstack);

	ImgPoint size = { C->p.sbox.x2 - C->p.sbox.x1, C->p.sbox.y2 - C->p.sbox.y1 };
	int aflags = C->p.flags;

	C->p = vec_pop(C->pstack);
	//C->p.flags |= RTP_F_ALIGN_DONE;
	
	if (aflags & RTP_F_ALIGN_RIGHT)
		C->p.pos.x = C->s.bbox.x2 - size.x;
	else if (aflags & RTP_F_ALIGN_HCENTER)
		C->p.pos.x = (C->s.bbox.x1 + C->s.bbox.x2 - size.x) / 2;
	
	if (aflags & RTP_F_ALIGN_BOTTOM)
		C->p.pos.y = C->s.bbox.y2 - size.y;
	else if (aflags & RTP_F_ALIGN_VCENTER)
		C->p.pos.y = (C->s.bbox.y1 + C->s.bbox.y2 - size.y) / 2;
	
	return 1;
}

static
void rtctx_text_wrap(RichTextCtx* C, int advance_x)
{
	const TextRender * F = C->s.font;

	// Text wrapping
	// When the right margin is reached by glyph, text will continue in the
	// following, if there is enough space below.
	if (!(C->p.flags & RTP_F_NO_WRAP) &&
		C->p.flags & RTP_F_BASELINE &&
		C->s.bbox.x2 > C->s.bbox.x1 + advance_x &&
		C->p.pos.x + advance_x > C->s.bbox.x2 &&
		C->s.bbox.y2 >= C->p.pos.y + (int)F->metrics.height )
	{
		C->p.pos.x = C->s.bbox.x1;
		C->p.pos.y += F->metrics.height;
	}
}

static
int rtctx_glyph_setup(RichTextCtx* C, unsigned cp, int flags)
{
	TextRender * F = C->s.font;

	const TextRenderGlyph * g = F->cls->glyph_get(F, cp);
	if (!g) return RT_RESULT_SKIP;

	C->glyph = (Image){ .data=g->bitmap_data, .w=g->bitmap_width,
		.h=g->bitmap_height, .pitch=g->bitmap_width, .bypp=1,
		.format=IMG_FORMAT_GRAY };
	
	if (!(C->p.flags & RTP_F_BASELINE)) {
		C->p.flags |= RTP_F_BASELINE;
		C->p.pos.y += F->metrics.ascent;
		C->glyph_idx_prev = 0;
	}

	C->p.prev = C->p.pos;

	rtctx_text_wrap(C, g->advance_x);

	// Optional: kerning
	if (!(flags & RTE_CF_NO_KERNING) && C->glyph_idx_prev && F->cls->kerning_get)
	{
		ImgPoint k = F->cls->kerning_get(F, C->glyph_idx_prev, g->glyph_index);
		C->p.pos.x += k.x;
	}
	C->glyph_idx_prev = g->glyph_index;

	// Origin of the glyph on screen
	C->p0.x = C->p.pos.x + g->bitmap_left;
	C->p0.y = C->p.pos.y - g->bitmap_top;
		
	// Expand the boundaries of the text
	int xe = C->p.pos.x +
		// Some styles can widen the glyphs (i.e: slanted)
		ccMAX(g->advance_x, g->bitmap_left + g->bitmap_width);
		//TODO: +1 ?
	if (C->p.sbox.x2 < xe)
		C->p.sbox.x2 = xe;
	
	int ye = C->p.pos.y + F->metrics.height - F->metrics.ascent;
	if (C->p.sbox.y2 < ye)
		C->p.sbox.y2 = ye;

	// Advance the pen position
	C->p.pos.x += g->advance_x;

	return RT_RESULT_DRAW_MASK;
}

static
int rtctx_image_setup(RichTextCtx* C, const Image* img, int flags, bool b_baseline)
{
	ccUNUSED(flags);
	C->glyph = *img;
	C->glyph.flags = 0;
	
	C->glyph_idx_prev = 0;
	
	const TextRender * F = C->s.font;
	if (b_baseline) {
		if (!(C->p.flags & RTP_F_BASELINE)) {
			C->p.flags |= RTP_F_BASELINE;
			C->p.pos.y += F->metrics.ascent;
		}
	} else {
		if (C->p.flags & RTP_F_BASELINE) {
			C->p.flags &= ~RTP_F_BASELINE;
			C->p.pos.y -= F->metrics.ascent;
		}
	}
	
	C->p.prev = C->p.pos;

	rtctx_text_wrap(C, img->w);
	
	C->p0.x = C->p.pos.x;
	C->p0.y = C->p.pos.y - img->h * (C->p.flags & RTP_F_BASELINE);
	
	MAXSET(C->p.sbox.x2, C->p0.x + (int)img->w);
	MAXSET(C->p.sbox.y2, C->p0.y + (int)img->h);

	C->p.pos.x += img->w;
	
	return RT_RESULT_DRAW_DIRECT;
}

int rtctx_ansi_cmd_parse(RichTextCtx* C, unsigned c, int flags)
{
	const TextRender * F = C->s.font;
	ccUNUSED(flags);

	switch (c) {
	case '\r':
		C->p.pos.x = C->s.bbox.x1;
		C->p.prev = C->p.pos;
		return RT_RESULT_SKIP;
	case '\n':
		C->p.pos.x = C->s.bbox.x1;
		C->p.pos.y += F->metrics.height;  //TODO: custom lineskip?
		if (C->p.flags & RTP_F_BASELINE) {
			C->p.flags &= ~RTP_F_BASELINE;
			C->p.pos.y -= F->metrics.ascent;
		}
		C->p.prev = C->p.pos;
		return RT_RESULT_SKIP;
	case '\t': {
		int tw = F->metrics.tab_width;  //TODO: global?
		C->p.pos.x = ((C->p.pos.x + tw - 1) / tw) * tw;
		return RT_RESULT_SKIP;
	}
	case '\v':
		C->p.pos.y += F->metrics.height;
		if (C->p.flags & RTP_F_BASELINE) {
			C->p.flags &= ~RTP_F_BASELINE;
			C->p.pos.y -= F->metrics.ascent;
		}
		C->p.prev = C->p.pos;
		return RT_RESULT_SKIP;
	case '\b':
		C->p.prev = C->p.pos;
		return RT_RESULT_SKIP;
	}

	return 0;  //Not an escape code
}

static
int rtctx_cmd_parse(RichTextCtx* C, RichTextEngine* E, TextDecodePos* pos,
	int flags)
{
	unsigned c = text_decode_next(pos, flags);
	if (c == 0) return RT_RESULT_ERROR;
	if (c == C->ec) return 0;
	switch (c) {
	case '{':
		vec_push(C->sstack, C->s);
		break;
	case '}':
		if (rtctx_align_end(C, pos)) break;
		if (!vec_count(C->sstack)) return -1;
		C->s = vec_pop(C->sstack);
		break;
	case 'p':
		C->s.flags |= RTS_F_PHANTOM;
		break;
	case 'c':
		c = text_decode_next(pos, flags);
		C->s.col_fg = richtext_color_get(E, c);
		break;
	case 'b':
		c = text_decode_next(pos, flags);
		C->s.col_bg = richtext_color_get(E, c);
		break;
	case 'f':
		c = text_decode_next(pos, flags);
		C->s.font = richtext_font_get(E, c);
		break;
	case 'i':
	case 'I': {
		bool b_baseline = (c == 'i');
		c = text_decode_next(pos, flags);
		const Image * img = richtext_image_get(E, c);
		if (!img) return RT_RESULT_SKIP;
		return rtctx_image_setup(C, img, flags, b_baseline);
	}
	case 'C': {
		int m = C->p.flags & RTP_F_BASELINE ? 1 : 0;
		const TextRender * F = C->s.font;
		c = text_decode_next(pos, flags);
		switch (c) {
		case 'l':  C->p.pos.x = C->s.bbox.x1;  break;
		case 'r':  C->p.pos.x = C->s.bbox.x2;  break;
		case 't':  C->p.pos.y = C->s.bbox.y1 + F->metrics.ascent*m;  break;
		case 'b':  C->p.pos.y = C->s.bbox.y2 - F->metrics.descent*m;  break;
		case 'L':  C->p.pos.x = C->p.sbox.x1;  break;
		case 'R':  C->p.pos.x = C->p.sbox.x2;  break;
		case 'T':  C->p.pos.y = C->p.sbox.y1 + F->metrics.ascent*m;  break;
		case 'B':  C->p.pos.y = C->p.sbox.y2 - F->metrics.descent*m;  break;
		default:   return RT_RESULT_ERROR;
		}
		//TODO: pen_set_pend=true for changes in y ?
		break;
	}
	case 'B': {
		int m = C->p.flags & RTP_F_BASELINE ? 1 : 0;
		const TextRender * F = C->s.font;
		c = text_decode_next(pos, flags);
		switch (c) {
		case 'l':  C->s.bbox.x1 = C->p.pos.x;  break;
		case 'r':  C->s.bbox.x2 = C->p.pos.x;  break;
		case 't':  C->s.bbox.y1 = C->p.pos.y - F->metrics.ascent*m;  break;
		case 'b':  C->s.bbox.y2 = C->p.pos.y + F->metrics.descent*m;  break;
		default:   return RT_RESULT_ERROR;
		}
		break;
	}
	case 'O':
		c = text_decode_next(pos, flags);
		switch (c) {
		case 'w':  C->p.flags &= ~RTP_F_NO_WRAP;  break;
		case 'W':  C->p.flags |=  RTP_F_NO_WRAP;  break;
		default:   return RT_RESULT_ERROR;
		}
		break;
	case 'v': {
		int aflags = C->p.flags & RTP_F_ALIGN_H_MASK;
		c = text_decode_next(pos, flags);
		switch (c) {
		case 't':  aflags |= RTP_F_ALIGN_TOP;      break;
		case 'b':  aflags |= RTP_F_ALIGN_BOTTOM;   break;
		case 'c':  aflags |= RTP_F_ALIGN_VCENTER;  break;
		default:   return RT_RESULT_ERROR;
		}
		rtctx_align_begin(C, pos, aflags);
		break;
	}
	case 'h': {
		int aflags = C->p.flags & RTP_F_ALIGN_V_MASK;
		c = text_decode_next(pos, flags);
		switch (c) {
		case 'l':  aflags |= RTP_F_ALIGN_LEFT;     break;
		case 'r':  aflags |= RTP_F_ALIGN_RIGHT;    break;
		case 'c':  aflags |= RTP_F_ALIGN_HCENTER;  break;
		default:   return RT_RESULT_ERROR;
		}
		rtctx_align_begin(C, pos, aflags);
		break;
	}
	default:
		return RT_RESULT_ERROR;
	}
	return RT_RESULT_SKIP;
}

int rtctx_next(RichTextCtx* C, RichTextEngine* E, TextDecodePos* pos, int flags)
{
	int R=0;

	unsigned c = text_decode_next(pos, flags);
	if (c == 0) {
		if (rtctx_align_end(C, pos)) return RT_RESULT_SKIP;
		return 0;
	}

	// Rich text command?
	if (c == C->ec) {
		R = rtctx_cmd_parse(C, E, pos, flags);
		if (R != 0) goto end;
	}

	// ANSI escape code?
	R = rtctx_ansi_cmd_parse(C, c, flags);
	if (R != 0) goto end;

	// A normal character
	R = rtctx_glyph_setup(C, c, flags);

end:
	if (C->s.flags & RTS_F_PHANTOM) R = RT_RESULT_SKIP;
	if (C->p.flags & RTP_F_ALIGN_MASK) R = RT_RESULT_SKIP;
	return R;
}

/* Renderer */

int richtext_render_m(RichTextEngine* S, Image* dst, ImgRect* dstrect,
	unsigned n_text, const StrSlice* texts)
{
	int R=1, r;
	int flags = S->c.flags;
	bool fast_blit = flags & RTE_CF_FAST_BLIT;
	
	ImgRect rect = {0};
	if (dstrect) rect = *dstrect;
	else { rect.w = dst->w; rect.h = dst->h; }

	RichTextCtx ctx={0};
	rtctx_init(&ctx, S, rect);
	
	TextDecodePos pos={0};
	text_decode_init(&pos, n_text, texts);

	while ((r = rtctx_next(&ctx, S, &pos, flags)) > 0) {
		if (r == RT_RESULT_SKIP) continue;
		if (!dst) continue;
		
		// Draw background
		if (ctx.s.col_bg.a) {
			//TODO: hackish...
			const TextRender * F = ctx.s.font;
			imgdraw_rect_fill(dst, (ImgRect){ ctx.p.prev.x,
				ctx.p.pos.y - F->metrics.ascent * (ctx.p.flags & RTP_F_BASELINE),
				ctx.p.pos.x - ctx.p.prev.x, F->metrics.height }, ctx.s.col_bg);
		}
		
		// Draw glyph / image
		if (r == RT_RESULT_DRAW_DIRECT) {
			imgdraw_blit(dst, ctx.p0, &ctx.glyph, !fast_blit);
		}
		else if (r == RT_RESULT_DRAW_MASK) {
			imgdraw_alpha_mask(dst, ctx.p0, &ctx.glyph, ctx.s.col_fg, fast_blit);
		}
	}
	if (r<0) R=r;

	// Return the bounding box
	if (dstrect) *dstrect = img_rect_p_to_s(&ctx.p.sbox);
	
//end:
	rtctx_free(&ctx);
	return R;
}
