/* Copyright 2024-2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: Zlib
 */
#pragma once
#include "ccommon.h"
#include <stdint.h>

/* One pixel blit */
#define Rget(d) d[0] 
#define Gget(d) d[1] 
#define Bget(d) d[2] 
#define Aget(d) d[3] 

static inline
void imgdraw_pixel_set_gray(uint8_t* d, int r, int g, int b, int a)
{
	ccUNUSED(a);
	if (g > r) r = g;
	if (b > r) r = b;
	*d = r;
}

static inline
void imgdraw_pixel_set_gray_a(uint8_t* d, int r, int g, int b, int a)
{
	ccUNUSED(r);  ccUNUSED(g);  ccUNUSED(b);
	*d = a;
}

static inline
void imgdraw_pixel_set_rgb(uint8_t* d, int r, int g, int b, int a)
{
	ccUNUSED(a);
	Rget(d) = r; Gget(d) = g; Bget(d) = b;
}

static inline
void imgdraw_pixel_set_rgba(uint8_t* d, int r, int g, int b, int a)
{
	Rget(d) = r; Gget(d) = g; Bget(d) = b; Aget(d) = a;
}

static inline
void imgdraw_pixel_blend_gray(uint8_t* d, int r, int g, int b, int a)
{
	ccUNUSED(r);  ccUNUSED(g);  ccUNUSED(b);
	*d += a - ((*d * a + 255) >> 8);
}

static inline
void imgdraw_pixel_blend_rgb(uint8_t* d, int r, int g, int b, int a)
{
	if (a == 0) return;
	if (a == 255) imgdraw_pixel_set_rgb(d, r, g, b, 255);
	else {
		Rget(d) += ((r - Rget(d)) * a + 255) >> 8;
		Gget(d) += ((g - Gget(d)) * a + 255) >> 8;
		Bget(d) += ((b - Bget(d)) * a + 255) >> 8;
	}
}

static inline
void imgdraw_pixel_blend_rgba(uint8_t* d, int r, int g, int b, int a)
{
	if (a == 0) return;
	if (a == 255) imgdraw_pixel_set_rgba(d, r, g, b, 255);
	else {
		imgdraw_pixel_blend_rgb(d, r, g, b, a);
		Aget(d) += a - ((Aget(d) * a + 255) >> 8);
	}
}

#undef Rget
#undef Gget
#undef Bget
#undef Aget

/* Fills */
//vars: drow, col
#define IMGDRAW_FILL_COLOR(B,F) \
	imgdraw_pixel_##B##_##F(drow, col.r, col.g, col.b, col.a);

//vars: drow, col
#define IMGDRAW_FILL_COLOR_A(B,F) \
	imgdraw_pixel_##B##_##F(drow, col.r, col.g, col.b, a);

/* Pixel set */
//vars: dst
#define IMGDRAW_PIXEL_SET(FILL,X,Y) \
	{ const int x_=(X), y_=(Y); uint8_t* drow=&IMG_INDEX(*dst,x_,y_); FILL }

//vars: dst
#define IMGDRAW_PIXEL_SET_CLIP(FILL,X,Y) { \
	const int x_=(X), y_=(Y); \
	if (0 <= x_ && x_ < w && 0 <= y_ && y_ < h) { \
		uint8_t* drow=&IMG_INDEX(*dst,x_,y_); \
		FILL \
	} \
}

/* Forms */
//vars: dst, x, y
#define IMGDRAW_FORM_PIXEL(dsx, FILL) \
	IMGDRAW_PIXEL_SET(FILL,x,y)

//vars: dst, x1, x2, y
#define IMGDRAW_FORM_HLINE(dsx, FILL) \
	uint8_t* drow = dst->data + dst->pitch * y + dsx * x1; \
	for (int x=x1; x<=x2; ++x, drow+=dsx) { \
		FILL \
	} \

//vars: dst, x1, x2, y1, y2
#define IMGDRAW_FORM_RECT(dsx, FILL) \
	for (int y=y1; y<y2; ++y) { \
		uint8_t* drow = dst->data + dst->pitch * y + dsx * x1; \
		for (int x=x1; x<x2; ++x, drow+=dsx) { \
			FILL \
		} \
	}

//vars: dst
#define IMGDRAW_FORM_RECT_DS(dsx) \
	for (int y=0, w=dst->w, h=dst->h; y<h; ++y) { \
		uint8_t* drow = dst->data + dst->pitch * y; \
		const uint8_t* srow = sdata + ssy * y; \
		for (int x=0; x<w; ++x, drow+=dsx, srow+=ssx) {

//vars: dst, col, x1, y1, x2, y2, cx, cy, fx, fy
#define IMGDRAW_FORM_DOT_SMOOTH(dsx, FILL) \
	for (int y=y1; y<y2; ++y) { \
		uint8_t* drow = dst->data + dst->pitch * y + dsx * x1; \
		for (int x=x1; x<x2; ++x, drow+=dsx) { \
			float p = (x-cx)*(x-cx)*fx + (y-cy)*(y-cy)*fy; \
			if (p >= 1.0) continue; \
			int a = col.a * (1.0 - p); \
			FILL \
		} \
	}
	
//vars: dst, x1, y1, x2, y2, thick
//TODO: optimize: fast sqrt ?
#define IMGDRAW_FORM_RING_SMOOTH(dsx, FILL) \
	for (int y=y1; y<y2; ++y) { \
		uint8_t* drow = dst->data + dst->pitch * y + dsx * x1; \
		for (int x=x1; x<x2; ++x, drow+=dsx) { \
			float s = sqrt((x-cx)*(x-cx) + (y-cy)*(y-cy)); \
			float s1 = rad + 0.5 - s; \
			float s2 = s1 - thi; \
			s1 = s1<0 ? 0 : s1>1 ? 1 : s1; \
			s2 = s2<0 ? 0 : s2>1 ? 1 : s2; \
			int a = col.a * (s1 - s2); \
			FILL \
		} \
	}

//vars: dst, y1, y2, m, b
#define IMGDRAW_FORM_LINE(dsx, FILL) \
	for (int y=y1; y<=y2; ++y) { \
		int x = y*m +b +0.5; \
		IMGDRAW_PIXEL_SET(FILL,x,y) \
	}
	
//vars: dst, y1, y2, m, b
#define IMGDRAW_FORM_LINE_T(dsx, FILL) \
	for (int x=y1; x<=y2; ++x) { \
		int y = x*m +b +0.5; \
		IMGDRAW_PIXEL_SET(FILL,x,y) \
	}

//vars: dst, x, y, r
#define IMGDRAW_FORM_CIRCLE_1PX(dsx, FILL) \
	const int w=dst->w, h=dst->h; \
	IMGDRAW_PIXEL_SET_CLIP(FILL, x, y+r) \
	IMGDRAW_PIXEL_SET_CLIP(FILL, x, y-r) \
	IMGDRAW_PIXEL_SET_CLIP(FILL, x+r, y) \
	IMGDRAW_PIXEL_SET_CLIP(FILL, x-r, y) \
	\
    int u=0, v=r, f=1-r; \
    while (u < v) { \
        if (f >= 0) { \
            v--; \
            f -= v*2; \
        } \
        u++; \
        f += u*2+1; \
        IMGDRAW_PIXEL_SET_CLIP(FILL, x + u, y + v); \
        IMGDRAW_PIXEL_SET_CLIP(FILL, x - u, y + v); \
        IMGDRAW_PIXEL_SET_CLIP(FILL, x + u, y - v); \
        IMGDRAW_PIXEL_SET_CLIP(FILL, x - u, y - v); \
        IMGDRAW_PIXEL_SET_CLIP(FILL, x + v, y + u); \
        IMGDRAW_PIXEL_SET_CLIP(FILL, x - v, y + u); \
        IMGDRAW_PIXEL_SET_CLIP(FILL, x + v, y - u); \
        IMGDRAW_PIXEL_SET_CLIP(FILL, x - v, y - u); \
    }

#define IMGDRAW_FORM_END }}

/* Dispatch */
#define IMGDRAW_FMT_SWITCH(FORM, FILL, B) \
	switch (dst->format) { \
	case IMG_FORMAT_NULL: \
		break; \
	case IMG_FORMAT_GRAY: \
		{ FORM(1, FILL(B, gray) ) }  \
		break; \
	case IMG_FORMAT_RGB: \
		{ FORM(3, FILL(B, rgb) ) } \
		break; \
	case IMG_FORMAT_RGBA: \
		{ FORM(4, FILL(B, rgba) ) } \
		break; \
	}

#define IMGDRAW_INNER(FORM, FILL) \
	if (blend) IMGDRAW_FMT_SWITCH(FORM, FILL, blend) \
	else IMGDRAW_FMT_SWITCH(FORM, FILL, set)
