/* Copyright 2024-2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: Zlib
 *
 * Software bitmap draw functions.
 * "_inner" functions do not clip, use with care or do not use at all.
 */
#pragma once
#include <stdint.h>
#include "ccommon.h"
#include "image.h"

//TODO: draw context with line and fill patterns, etc.

void imgdraw_pixel_inner(Image* dst, int x, int y, ImgColor col, bool blend);

static inline
void imgdraw_pixel(Image* dst, ImgPoint p, ImgColor col)
{
	if (!(0 <= p.x && p.x < (int)dst->w && 0 <= p.y && p.y < (int)dst->h)) return;
	imgdraw_pixel_inner(dst, p.x, p.y, col, col.a != 255);
}

void imgdraw_rect_color_inner(Image* dst, int x1, int y1, int x2, int y2,
	ImgColor col, bool blend);

static inline
void imgdraw_rect_fill(Image* img, ImgRect rect, ImgColor col)
{
	int x1=rect.x, y1=rect.y, x2=rect.x + rect.w, y2=rect.y + rect.h;
	if (x1 > x2) ccSWAPT(int, x1, x2);
	if (y1 > y2) ccSWAPT(int, y1, y2);
	if (!(x2 > 0 && y2 > 0)) return;
	if (x1 < 0) x1 = 0;
	if (y1 < 0) y1 = 0;
	if (x2 > (int)img->w) x2 = img->w;
	if (y2 > (int)img->h) y2 = img->h;
	imgdraw_rect_color_inner(img, x1, y1, x2, y2, col, col.a != 255);
}

static inline
void imgdraw_hline(Image* img, int x1, int x2, int y, ImgColor col, int lw)
{
	imgdraw_rect_fill(img, (ImgRect){x1, y, x2-x1+1, lw}, col);
}

static inline
void imgdraw_vline(Image* img, int x, int y1, int y2, ImgColor col, int lw)
{
	imgdraw_rect_fill(img, (ImgRect){x, y1, lw, y2-y1+1}, col);
}

static inline
void imgdraw_rect_border(Image* img, ImgRect rect, ImgColor col, int lw)
{
	imgdraw_rect_fill(img, (ImgRect){rect.x, rect.y, rect.w, lw}, col);
	imgdraw_rect_fill(img, (ImgRect){rect.x, rect.y+lw, lw, rect.h-lw*2}, col);
	imgdraw_rect_fill(img, (ImgRect){rect.x+rect.w-lw, rect.y+lw, lw, rect.h-lw*2}, col);
	imgdraw_rect_fill(img, (ImgRect){rect.x, rect.y+rect.h-lw, rect.w, lw}, col);
}

void imgdraw_line(Image* img, ImgPoint p1, ImgPoint p2, ImgColor col, int lw);

static inline
void imgdraw_polygon_border(Image* img, unsigned npts, const ImgPoint* pts,
	ImgColor col, int lw)
{
	unsigned ip=npts-1;
	for (unsigned i=0; i<npts; ip=i, ++i)
		imgdraw_line(img, pts[ip], pts[i], col, lw);
}

void imgdraw_polygon_fill(Image* img, unsigned npts, const ImgPoint* pts,
	ImgColor col);

void imgdraw_circle_border_inner(Image* dst, int x, int y, int r,
	ImgColor col, bool blend);

static inline
void imgdraw_circle_border(Image* img, ImgPoint cen, int rad, ImgColor col)
{
	imgdraw_circle_border_inner(img, cen.x, cen.y, rad, col, col.a != 255);
}

// Integers in floating point position correspond to the pixel center

void imgdraw_dot_smooth(Image* dst, float cx, float cy, float rx, float ry,
	ImgColor col);
	
void imgdraw_ring_smooth(Image* dst, float cx, float cy, float radius,
	float thick, ImgColor col);

/* Draw an image into another */

void imgdraw_blit_inner(Image* dst,
	const uint8_t* sdata, unsigned ssy, ImgFormat sfmt,
	bool blend);

static inline
void imgdraw_blit(Image* dst, ImgPoint pos, const Image* src, bool blend)
{
	Image dv={0};
	img_view_make(&dv, dst, (ImgRect){pos.x, pos.y, src->w, src->h});
	if (dv.w == 0 || dv.h == 0) return;
	const uint8_t * sdata = src->data;
	if (pos.x < 0) sdata -= pos.x * (int)src->bypp;
	if (pos.y < 0) sdata -= pos.y * (int)src->pitch;
	imgdraw_blit_inner(&dv, sdata, src->pitch, src->format, blend);
}

/* Draw a colored gray map. User by text rendering. */

void imgdraw_alpha_mask_inner(Image* dst,
	const uint8_t* sdata, unsigned ssx, unsigned ssy, //alpha mask
	ImgColor col, bool fast);
	
static inline
void imgdraw_alpha_mask(Image* dst, ImgPoint pos, const Image* src,
	ImgColor col, bool fast)
{
	Image dv={0};
	img_view_make(&dv, dst, (ImgRect){pos.x, pos.y, src->w, src->h});
	if (dv.w == 0 || dv.h == 0) return;
	const uint8_t * sdata = src->data;
	if (pos.x < 0) sdata -= pos.x * (int)src->bypp;
	if (pos.y < 0) sdata -= pos.y * (int)src->pitch;
	imgdraw_alpha_mask_inner(&dv, sdata, src->bypp, src->pitch, col, fast);
}
