/* Copyright 2025-2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: Zlib
 *
 * Immediate mode graphical user interface.
 *
 * This is intended to be a simple GUI for use in games or other software with
 * limited interfaces.
 */
#pragma once
#include "image.h"
#include "richtext.h"

typedef ImgPoint ImGuiPoint;
typedef ImgRect ImGuiRect;

typedef enum ImGuiShape {
	IMGUI_SHAPE_RECT	= 0,
	IMGUI_SHAPE_CIRCLE	= 1,
} ImGuiShape;

typedef struct ImGuiRegion {
	ImGuiShape shape;	
	ImGuiPoint p, s;  //Rect: origin (p) and size (s)
	                  //Circle: center (p) and radius (s.x)
} ImGuiRegion;

typedef struct ImGuiStyle {
	ImgColor col_bg, col_bor1, col_bor2;
	StrSlice text_pre;  //Use it to set up rich text format
	int m_bor;  //Border thickness in pixels
	int m_pad;  //Internal padding thickness in pixels
	int sty_hl;  //Style to use on mouse hover or selection
	int sty_on;  //Style to use when in the "on" state.
} ImGuiStyle;

void imgui_style_button_colors_set(ImGuiStyle* sty, ImgColor color);

enum ImGuiStyleId {
	//Default styles, you may define others
	IMGUI_STYLE_NONE = 0,
	IMGUI_STYLE_LABEL,
	IMGUI_STYLE_IREGION,  //Interaction region: black border, centered text
	IMGUI_STYLE_TOOLTIP,
	IMGUI_STYLE_BUTTON,
	IMGUI_STYLE_BUTTON_HL,  //Select, Hover, Hold
	IMGUI_STYLE_BUTTON_ON,
	IMGUI_STYLE_BUTTON_ON_HL,
	IMGUI_STYLE__COUNT
};

enum ImGuiLayoutFlags {
	IMGUI_LAYOUT_HORIZ	= 1,
};

enum ImGuiStateFlag {
	IMGUI_STATE_PRESS	= 0x0001,
	IMGUI_STATE_HOLD	= 0x0002,
	IMGUI_STATE_HOVER	= 0x0004,
	IMGUI_STATE_SELECT	= 0x0008,
	IMGUI_STATE__MASK_INTERACT = 0x000FF,
	IMGUI_STATE_ON		= 0x0100,
};

typedef struct ImGui {
	ImGuiRegion * regions;  //vector
	// Regions interacting, index >= 0 or -1 for none
	int region_select;
	int region_hover;
	int region_hold;
	int region_press;
	unsigned n_region_last;

	ImGuiStyle * styles;  //vector
	
	// Set by the backe
	//Image * fb;  //Set by the backend
	//TODO: replace with fb_get(backend, rect)

	ImGuiPoint mouse_pos;
	
	// Layout manager
	struct {
		ImGuiPoint cur;
		ImGuiPoint once;
		int m_sep;
		int flags;
	} lm;

	// Set by the backend
	struct ImGuiBackendInterface {
		void *backend;
		void (*fb_get)(void*, Image*, ImGuiRect);
	} be;
	
	RichTextEngine * rte;  //Set before use
} ImGui;

void imgui_free(ImGui* S);

void imgui_default_styles_set(ImGui* S);

// Returns and style from its id.
// If the style does not exists, it is allocated.
// Styles are stored sequentially and all intermediate ids will be allocted too.
// The returned pointer may become invalid by new calls that allocate.
ImGuiStyle* imgui_style_get(ImGui* S, int id);

void imgui_reset(ImGui* S);  //Call once before each "screen"

void imgui_frame_begin(ImGui* S);

// Draw rich text in a box using a style.
// Use style=0 for none (text only, no background, no borders).
// If rect != NULL, fills its null elements.
void imgui_textbox(ImGui* S, const char* text, ImGuiRect* rect, int style);

// Draw "tooltip" textbox that does not affect the layout.
// By default, it is drawn at the mouse position.
void imgui_tooltip(ImGui* S, const char* text, ImGuiRect* rect, int style);

// Label textbox. The next one will be drawn besides it.
void imgui_label(ImGui* S, const char* text, ImGuiRect* rect, int style);

// Roughly equivalent to calling imgui_textbox and imgui_rect_interact.
// Return the region state from the later.
// If style == 0, uses the default.
int imgui_button(ImGui* S, const char* text, ImGuiRect* rect, int style);

// Button that remains pushed down, giving it two possible states.
// *pstate stores the current state: 0 or 1.
int imgui_state_button(ImGui* S, int* pstate, const char* text, ImGuiRect* rect,
	int style);

// Set of button to choose between multiple options.
// *psel stores the current option. Use -1 for none.
int imgui_select_buttons(ImGui* S, int* psel, unsigned n_opt,
	const char** texts, ImGuiRect* rect, int style);
#define imgui_select_buttons_va(S, PSEL, N, ...) \
	imgui_select_buttons(S, PSEL, N, (const char*[N]){__VA_ARGS__}, NULL, 0)

// Fills rect without any drawing.
// Updates the layout cursor.
void imgui_textbox_rect_get(ImGui* S, const char* text, ImGuiRect* rect, int style);

// Allows the use to interact with a rectangle-shaped region and
// returns the resulting state (IMGUI_STATE_*).
int imgui_rect_interact(ImGui* S, ImGuiRect rect);

// Allows the use to interact with a circle-shaped region and
// returns the resulting state (IMGUI_STATE_*).
int imgui_circle_interact(ImGui* S, ImGuiPoint cen, int rad);

/* Advanced / internal */

// Interaction with the regions/widgets used by the keyboard navigation.
// Returns 1 of the action had any affect, or 0 otherwise.
int imgui_focus_next(ImGui* S);
int imgui_focus_prev(ImGui* S);
int imgui_focus_press(ImGui* S);

int imgui_region_from_pos(const ImGui* S, int x, int y);

void imgui_draw_debug_regions(ImGui* S, Image* frame, ImgColor col);

void imgui_frame_end(ImGui* S);  //Called by the backend

/* SDL2 Backend */
#ifdef IMGUI_BACKEND_SDL2
#include <SDL2/SDL.h>

enum {
	IMGUI_WF_NAV_KEYBOARD	= 1,
	IMGUI_WF_NAV_GAMEPAD	= 2,
};

typedef struct ImGuiSdl2 {
	ImGui *			gui;  //Set before use

	//You may set the following two before called imgui_sdl2_create().
	SDL_Window * 	window;
	SDL_Renderer * 	renderer;
	
	SDL_Texture * 	texture;
	Image			fb;  //Framebuffer, may be a view set by you

	int				sdl_init_flags;

	struct {
		const char * win_title;
		int win_w, win_h;
		int win_sdl_flags;  //e.g. SDL_WINDOW_FULLSCREEN_DESKTOP
		int wflags;  //IMGUI_WF_*
	} c;
} ImGuiSdl2;

void imgui_sdl2_free(ImGuiSdl2* S);

int imgui_sdl2_create(ImGuiSdl2* S, ImGui* gui);

int imgui_sdl2_present(ImGuiSdl2* S);

// Returns 1 if the event was processed, 0 if ignored.
int imgui_sdl2_event_process(ImGuiSdl2* S, const SDL_Event* ev);

#endif  //IMGUI_BACKEND_SDL2
