/* Copyright 2025-2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: Zlib
 */
#include "im_gui.h"
#include "vector.h"
#include "logging.h"
#include "image_draw.h"

void imgui_style_button_colors_set(ImGuiStyle* sty, ImgColor color)
{
	ImgColorHSV hsv = img_color_rgb2hsv(color);

	sty->col_bg = color;
	IMG_COLOR_TRANSF_HSV_RGB(sty->col_bor1, hsv, v=v*3/4);
	IMG_COLOR_TRANSF_HSV_RGB(sty->col_bor2, hsv, v=v/2);
}

void imgui_free(ImGui* S)
{
	vec_free(S->regions);
}

void imgui_default_styles_set(ImGui* S)
{
	vec_resize_zero(S->styles, IMGUI_STYLE__COUNT);
	S->styles[IMGUI_STYLE_LABEL] = (ImGuiStyle){
		.m_bor		= 0,
		.m_pad		= 3,
		.text_pre   = strsl_static("^vc"),
	};
	S->styles[IMGUI_STYLE_IREGION] = (ImGuiStyle){
		.col_bor1	= {0,0,0,255},
		.col_bor2	= {0,0,0,255},
		.m_bor		= 1,
		.m_pad		= 2,
		.text_pre   = strsl_static("^hc^vc"),  //Center
	};
	S->styles[IMGUI_STYLE_TOOLTIP] = (ImGuiStyle){
		.col_bg		= {223,223,159,255},
		.col_bor1	= {159,159, 95,255},
		.col_bor2	= {159,159, 95,255},
		.m_bor		= 1,
		.m_pad		= 2,
		//Spaces to prevents the cursor obscuring
		.text_pre   = strsl_static("^hc^vc  "),
	};
	S->styles[IMGUI_STYLE_BUTTON] = (ImGuiStyle){
		.col_bg		= {191,191,191,255},
		.col_bor1	= {159,159,159,255},
		.col_bor2	= {127,127,127,255},
		.m_bor		= 2,
		.m_pad		= 1,
		.text_pre   = strsl_static("^hc^vc"),  //Center
		.sty_hl     = IMGUI_STYLE_BUTTON_HL,
		.sty_on		= IMGUI_STYLE_BUTTON_ON,
	};
	S->styles[IMGUI_STYLE_BUTTON_HL] = (ImGuiStyle){
		.col_bg		= {159,159,159,255},
		.col_bor1	= { 95, 95, 95,255},
		.col_bor2	= { 63, 63, 63,255},
		.m_bor		= 2,
		.m_pad		= 1,
		.text_pre   = strsl_static("^hc^vc"),  //Center
		.sty_on		= IMGUI_STYLE_BUTTON_ON_HL,
	};
	S->styles[IMGUI_STYLE_BUTTON_ON] = (ImGuiStyle){
		.col_bg		= {  0,191,  0,255},
		.col_bor1	= {  0,159,  0,255},
		.col_bor2	= {  0, 95,  0,255},
		.m_bor		= 2,
		.m_pad		= 1,
		.text_pre   = strsl_static("^hc^vc"),  //Center
		.sty_hl     = IMGUI_STYLE_BUTTON_ON_HL,
	};
	S->styles[IMGUI_STYLE_BUTTON_ON_HL] = (ImGuiStyle){
		.col_bg		= {  0,159,  0,255},
		.col_bor1	= { 95, 95, 95,255},
		.col_bor2	= { 63, 63, 63,255},
		.m_bor		= 2,	
		.m_pad		= 1,
		.text_pre   = strsl_static("^hc^vc"),  //Center
	};
}

void imgui_reset(ImGui* S)
{
	S->region_select = -1;
	S->region_hover  = -1;
	S->region_hold   = -1;
	S->region_press  = -1;
	S->n_region_last = 0;
}

void imgui_frame_begin(ImGui* S)
{
	if (S->n_region_last == 0)
		imgui_reset(S);

	vec_resize(S->regions, 0);

	IFNPOSSET(S->lm.m_sep, S->rte->c.def_font->metrics.height/2);
	S->lm.cur  = (ImGuiPoint){ S->lm.m_sep, S->lm.m_sep };
	S->lm.once = (ImGuiPoint){0};
}

void imgui_frame_end(ImGui* S)
{
	if (S->n_region_last != vec_count(S->regions)) {
		S->region_select = -1;
		S->region_hover  = -1;
		S->region_hold   = -1;
	}

	S->region_press  = -1;

	S->n_region_last = vec_count(S->regions);
}

static inline
int imgui_region_check(const ImGui* S)
{
	int s=0, id = vec_count(S->regions);  //TODO: check same region?
	if (S->region_press  == id) s |= IMGUI_STATE_PRESS;
	if (S->region_hold   == id) s |= IMGUI_STATE_HOLD;
	if (S->region_hover  == id) s |= IMGUI_STATE_HOVER;
	if (S->region_select == id) s |= IMGUI_STATE_SELECT;
	return s;
}

int imgui_rect_interact(ImGui* S, ImGuiRect rect)
{
	int r = imgui_region_check(S);
	ImGuiRegion item = { .shape=IMGUI_SHAPE_RECT, .p={rect.x, rect.y},
		.s={rect.w, rect.h}};
	vec_push(S->regions, item);
	return r;
}

int imgui_circle_interact(ImGui* S, ImGuiPoint cen, int rad)
{
	int r = imgui_region_check(S);
	ImGuiRegion item = { .shape=IMGUI_SHAPE_CIRCLE, .p=cen, .s={rad} };
	vec_push(S->regions, item);
	return r;
}

int imgui_region_from_pos(const ImGui* S, int x, int y)
{
	//Reversed. In case of overlap, later additions take precedence.
	vec_forr(S->regions, i) {
		const ImGuiRegion * reg = &S->regions[i];
		int xr = x - reg->p.x;
		int yr = y - reg->p.y;
		int w = reg->s.x;
		int h = reg->s.y;

		switch (reg->shape) {
		case IMGUI_SHAPE_RECT:
			if (0 <= xr && xr < w && 0 <= yr && yr < h)
				return i;
			break;
		case IMGUI_SHAPE_CIRCLE:
			if (xr*xr + yr*yr < w*w)
				return i;
			break;
		}
	}
	
	return -1;
}

ImGuiStyle* imgui_style_get(ImGui* S, int id)
{
	if (id < 0) return NULL;
	int n = vec_count(S->styles);
	if (id >= n) {
		assert( id < 1024 );
		vec_append_zero(S->styles, id + 1 - n);
	}
	return &S->styles[id];
}

const ImGuiStyle* imgui_style_get_i(const ImGui* S, int style, int state)
{
	if (!vec_idx_check(S->styles, style)) return NULL;
	const ImGuiStyle * sty = &S->styles[style];
	if (state & IMGUI_STATE_ON && sty->sty_on)
		sty = &S->styles[sty->sty_on];	
	if (state & IMGUI_STATE__MASK_INTERACT && sty->sty_hl)
		sty = &S->styles[sty->sty_hl];
	return sty;
}

void imgui_textbox_rect_get(ImGui* S, const char* text, ImGuiRect* rect, int style)
{
	if (!rect) return;

	const ImGuiStyle * sty = vec_getpd(S->styles, style, NULL);

	ImGuiRect srect;
	if (!(rect->w > 0 && rect->h > 0)) {
		if (text && text[0]) {
			srect = *rect;
			StrSlice texts[2] = { {0}, strsl_fromz(text) };
			if (sty) texts[0] = sty->text_pre;
			richtext_render_m(S->rte, NULL, &srect, 2, texts);
		} else {
			const TextRender * font = S->rte->c.def_font;
			srect.w = font->metrics.m_width * 12;
			srect.h = font->metrics.height;
		}
	}
	
	// Auto-size
	if (rect->w <= 0) rect->w = srect.w + (sty ? (sty->m_bor+sty->m_pad)*2 : 0 );
	if (rect->h <= 0) rect->h = srect.h + (sty ? (sty->m_bor+sty->m_pad)*2 : 0 );

	// Auto-position using the layout manager
	if (S->lm.flags & IMGUI_LAYOUT_HORIZ) {
		if (rect->x <= 0) {
			if (S->lm.once.x) {
				rect->x = S->lm.once.x;
				S->lm.once = (ImGuiPoint){0};
			} else {
				rect->x = S->lm.cur.x;
				S->lm.cur.x = rect->x + rect->w + S->lm.m_sep;
			}
		}
		if (rect->y <= 0)
			rect->y = S->lm.once.y ? S->lm.once.y : S->lm.cur.y;
	} else {
		if (rect->x <= 0)
			rect->x = S->lm.once.x ? S->lm.once.x : S->lm.cur.x;
		if (rect->y <= 0) {
			if (S->lm.once.y) {
				rect->y = S->lm.once.y;
				S->lm.once = (ImGuiPoint){0};
			} else {
				rect->y = S->lm.cur.y;
				S->lm.cur.y = rect->y + rect->h + S->lm.m_sep;
			}
		}
	}
}

static
void imgui_textbox_i(ImGui* S, StrSlice text, ImGuiRect rect, const ImGuiStyle* sty)
{
	if (!S->be.fb_get) return;
	Image fb={0};
	S->be.fb_get(S->be.backend, &fb, rect);
	if (img_empty(&fb)) return;
	rect.x = 0;
	rect.y = 0;

	if (sty) {
		// Background
		if (sty->col_bg.a)
			imgdraw_rect_fill(&fb, rect, sty->col_bg);
		
		// Border
		int lw = sty->m_bor;
		if (lw > 0 && sty->col_bor1.a) {
			imgdraw_rect_fill(&fb, (ImgRect){rect.x, rect.y, rect.w, lw},
				sty->col_bor1);
			imgdraw_rect_fill(&fb, (ImgRect){rect.x, rect.y+lw, lw, rect.h-lw*2},
				sty->col_bor1);
		}
		
		if (lw > 0 && sty->col_bor2.a) {
			int x2 = rect.x + rect.w;
			int y2 = rect.y + rect.h;
			imgdraw_rect_fill(&fb, (ImgRect){x2-lw, rect.y+lw, lw, rect.h-lw*2},
				sty->col_bor2);
			imgdraw_rect_fill(&fb, (ImgRect){rect.x, y2-lw, rect.w, lw},
				sty->col_bor2);
		}
	}

	// Text
	StrSlice texts[2] = { {0}, text };
	if (sty) texts[0] = sty->text_pre;
	richtext_render_m(S->rte, &fb, &rect, 2, texts);
}

void imgui_textbox(ImGui* S, const char* text, ImGuiRect* rect, int style)
{
	ImGuiRect tmp_rect={0};
	if (!rect) rect = &tmp_rect;
	imgui_textbox_rect_get(S, text, rect, style);
	
	const ImGuiStyle * sty = imgui_style_get_i(S, style, 0);
	
	imgui_textbox_i(S, strsl_fromz(text), *rect, sty);
}

void imgui_tooltip(ImGui* S, const char* text, ImGuiRect* rect, int style)
{
	if (!style) style = IMGUI_STYLE_TOOLTIP;
	
	ImGuiRect tmp_rect={0};
	if (!rect) rect = &tmp_rect;
	if (rect->x <= 0) rect->x = S->mouse_pos.x;
	if (rect->y <= 0) rect->y = S->mouse_pos.y;
	
	imgui_textbox(S, text, rect, style);
}

void imgui_label(ImGui* S, const char* text, ImGuiRect* rect, int style)
{
	if (!style) style = IMGUI_STYLE_LABEL;

	ImGuiRect tmp_rect={0};
	if (!rect) rect = &tmp_rect;
	
	imgui_textbox(S, text, rect, style);
	
	S->lm.once = (ImGuiPoint){ rect->x + rect->w, rect->y };
}

int imgui_button(ImGui* S, const char* text, ImGuiRect* rect, int style)
{
	if (!style) style = IMGUI_STYLE_BUTTON;

	ImGuiRect tmp_rect={0};
	if (!rect) rect = &tmp_rect;
	imgui_textbox_rect_get(S, text, rect, style);

	int state = imgui_rect_interact(S, *rect);
	
	const ImGuiStyle * sty = imgui_style_get_i(S, style, state);

	imgui_textbox_i(S, strsl_fromz(text), *rect, sty);

	return state;
}

int imgui_state_button(ImGui* S, int* pstate, const char* text, ImGuiRect* rect,
	int style)
{
	if (!style) style = IMGUI_STYLE_BUTTON;

	ImGuiRect tmp_rect={0};
	if (!rect) rect = &tmp_rect;
	imgui_textbox_rect_get(S, text, rect, style);

	int state = imgui_rect_interact(S, *rect);
	
	if (state & IMGUI_STATE_PRESS) *pstate = !*pstate;
	if (*pstate) state |= IMGUI_STATE_ON;
	
	const ImGuiStyle * sty = imgui_style_get_i(S, style, state);

	imgui_textbox_i(S, strsl_fromz(text), *rect, sty);

	return state;
}

int imgui_select_buttons(ImGui* S, int* psel, unsigned n_opt,
	const char** texts, ImGuiRect* rect, int style)
{
	if (!n_opt) return 0;

	ImGuiRect tmp_rect={0};
	if (!rect) rect = &tmp_rect;

	int x0=0, w=rect ? rect->w/n_opt : 0, result=0;
	for (unsigned i=0; i<n_opt; ++i) {
		if (i) {
			rect->x += rect->w + S->lm.m_sep;
			rect->w = w;
		}
		int on_off = (*psel == (int)i);
		int s = imgui_state_button(S, &on_off, texts[i], rect, style);
		result |= s & (IMGUI_STATE__MASK_INTERACT);
		if (on_off) *psel = i;
		if (!i) x0 = rect->x;
	}
	rect->w = rect->x + rect->w - x0;
	rect->x = x0;

	return result;
}

int imgui_focus_next(ImGui* S)
{
	int n = vec_count(S->regions);
	if (!n) return 0;
	int rs = S->region_select;
	rs = (rs + 1) % n;
	S->region_select = rs;
	return 1;
}

int imgui_focus_prev(ImGui* S)
{
	int n = vec_count(S->regions);
	if (!n) return 0;
	int rs = S->region_select;
	if (rs < 0) rs = 0;
	rs = (rs + n - 1) % n;
	S->region_select = rs;
	return 1;
}

int imgui_focus_press(ImGui* S)
{
	if (S->region_select < 0) return 0;
	S->region_press = S->region_select;
	return 1;
}

void imgui_draw_debug_regions(ImGui* S, Image* frame, ImgColor col)
{
	vec_for(S->regions, i, 0) {
		const ImGuiRegion * reg = &S->regions[i];
		switch (reg->shape) {
		case IMGUI_SHAPE_RECT:
			imgdraw_rect_border(frame,
				(ImgRect){reg->p.x, reg->p.y, reg->s.x, reg->s.y},
				col, 1);
			break;
		case IMGUI_SHAPE_CIRCLE:
			imgdraw_circle_border(frame,
				(ImgPoint){reg->p.x, reg->p.y}, reg->s.x,
				col);
			break;
		}
	}
}

/* SDL2 Backend */
#ifdef IMGUI_BACKEND_SDL2

void imgui_sdl2_free(ImGuiSdl2* S)
{
	img_free(&S->fb);

	if (S->texture) {
		SDL_DestroyTexture(S->texture);
		S->texture = NULL;
	}
	if (S->renderer) {
		SDL_DestroyRenderer(S->renderer);
		S->renderer = NULL;
	}
	if (S->window) {
		SDL_DestroyWindow(S->window);
		S->window = NULL;
	}
	
	if (S->sdl_init_flags) {
		SDL_QuitSubSystem(S->sdl_init_flags);
		S->sdl_init_flags = 0;
	}
}

static
int imgui_sdl2_fb_resize(ImGuiSdl2* S, int w, int h)
{
	// If the image is a view, do not touch it
	if (S->fb.data && !(S->fb.flags & IMG_F_OWN_MEM)) return 0;

	if (S->texture) {
		SDL_DestroyTexture(S->texture);
		S->texture = NULL;
	}

	S->texture = SDL_CreateTexture(S->renderer,
		SDL_PIXELFORMAT_RGB24, SDL_TEXTUREACCESS_STREAMING, w, h);
	if (!S->texture) return -1;
	
	TRYR( img_resize(&S->fb, w, h, IMG_FORMAT_RGB, 0) );

	log_debug("Framebuffer: %dx%d RGB24", w, h);
	return 1;
}

static
void imgui_sdl2_fb_patch_get(ImGuiSdl2* S, Image* view, ImGuiRect rect)
{
	img_view_make(view, &S->fb, rect);
}

int imgui_sdl2_create(ImGuiSdl2* S, ImGui* gui)
{
	int R=1;

	if (!S->sdl_init_flags) {
		S->sdl_init_flags = SDL_INIT_VIDEO;
		if (S->c.wflags & IMGUI_WF_NAV_GAMEPAD)
			S->sdl_init_flags |= SDL_INIT_GAMECONTROLLER;
		
		if (SDL_InitSubSystem(S->sdl_init_flags) < 0)
			ERROR_LOG(-1, "Unable to initialize SDL: %s", SDL_GetError());

		log_debug("Video driver: %s", SDL_GetCurrentVideoDriver());
	}

	if (!S->window) {
		IFFALSESET(S->c.win_title, "ImGui");
		IFFALSESET(S->c.win_w, 640);
		IFFALSESET(S->c.win_h, 480);

		S->window = SDL_CreateWindow(S->c.win_title,
						SDL_WINDOWPOS_UNDEFINED,
						SDL_WINDOWPOS_UNDEFINED,
						S->c.win_w, S->c.win_h,
						S->c.win_sdl_flags);
		if (S->window == NULL)
			ERROR_LOG(-1, "SDL window creation: %s", SDL_GetError());
	}

	if (!S->renderer) {
		S->renderer = SDL_CreateRenderer(S->window, -1, SDL_RENDERER_PRESENTVSYNC);
		if (S->renderer == NULL)
			ERROR_LOG(-1, "SDL renderer creation: %s", SDL_GetError());
	}

	int w, h;
	SDL_GetRendererOutputSize(S->renderer, &w, &h);
	TRY( imgui_sdl2_fb_resize(S, w, h) );

	if (S->c.wflags & IMGUI_WF_NAV_GAMEPAD) {
		// This is required to receive GC events
	    SDL_GameController * gc = SDL_GameControllerOpen(0);
    	if (!gc) log_debug("imgui: could not open gamepad 0");
	}

	if ((S->gui = gui)) {
		//S->gui->fb = &S->fb;
		MEM_ZERO(S->gui->be);
		S->gui->be = (struct ImGuiBackendInterface){
			S,
			(void (*)(void*, Image*, ImGuiRect)) imgui_sdl2_fb_patch_get,
		};
	}

end:
	if (R<0) imgui_sdl2_free(S);
	return R;
}

int imgui_sdl2_present(ImGuiSdl2* S)
{
	imgui_frame_end(S->gui);
	SDL_UpdateTexture(S->texture, NULL, S->fb.data, S->fb.pitch);
	SDL_RenderCopy(S->renderer, S->texture, NULL, NULL);
	SDL_RenderPresent(S->renderer);
	return 1;
}

#define KMOD_CHECK(VAR, FLAGS) \
	((VAR) && ((VAR) & (FLAGS)) == (VAR))

int imgui_sdl2_event_process(ImGuiSdl2* S, const SDL_Event* ev)
{
	int R=0;

	switch (ev->type) {
	case SDL_KEYDOWN:
	 	if (!(S->c.wflags & IMGUI_WF_NAV_KEYBOARD)) break;
		switch (ev->key.keysym.sym) {
		case SDLK_TAB: {
			if (KMOD_CHECK(ev->key.keysym.mod, KMOD_SHIFT)) {
				R = imgui_focus_prev(S->gui);
			} else {
				R = imgui_focus_next(S->gui);
			}
			R=1;
			break;
		}
		case SDLK_RETURN:
		case SDLK_SPACE:
			R = imgui_focus_press(S->gui);
			break;
		}
		break;
	case SDL_CONTROLLERBUTTONDOWN:
	 	if (!(S->c.wflags & IMGUI_WF_NAV_GAMEPAD)) break;
		if (!vec_count(S->gui->regions)) break;
		switch (ev->cbutton.button) {
		case SDL_CONTROLLER_BUTTON_DPAD_UP:
		case SDL_CONTROLLER_BUTTON_DPAD_LEFT:
			R = imgui_focus_prev(S->gui);
			break;
		case SDL_CONTROLLER_BUTTON_DPAD_DOWN:
		case SDL_CONTROLLER_BUTTON_DPAD_RIGHT:
			R = imgui_focus_next(S->gui);
			break;
		case SDL_CONTROLLER_BUTTON_A:
			R = imgui_focus_press(S->gui);
			break;
		}
		break;
	case SDL_MOUSEMOTION: {
		int r = imgui_region_from_pos(S->gui, ev->motion.x, ev->motion.y);
		S->gui->region_hover = r;
		S->gui->mouse_pos = (ImGuiPoint){ ev->motion.x, ev->motion.y };
		if (r >= 0) R=1;
		break;
	}
    case SDL_MOUSEBUTTONDOWN: {
		if (ev->button.button == SDL_BUTTON_LEFT) {
			int r = imgui_region_from_pos(S->gui, ev->motion.x, ev->motion.y);
			S->gui->region_hold  = r;
			S->gui->region_press = -1;
			if (r >= 0) R=1;
		}
		break;
	}
    case SDL_MOUSEBUTTONUP: {
		if (ev->button.button == SDL_BUTTON_LEFT) {
			int r = imgui_region_from_pos(S->gui, ev->motion.x, ev->motion.y);
			S->gui->region_hold  = -1;
			S->gui->region_press = r;
			if (r >= 0) R=1;
		}
		break;
	}
	case SDL_WINDOWEVENT:
		switch (ev->window.event) {
		case SDL_WINDOWEVENT_SIZE_CHANGED: {
			int w = ev->window.data1;
			int h = ev->window.data2;
			imgui_sdl2_fb_resize(S, w, h);
			R=1;
			break;
		}
		}
		break;
	}

	return R;
}

#endif  //IMGUI_BACKEND_SDL2
