/* Copyright 2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: Zlib
 */
#include "audio_sdl.h"
#include "ccommon.h"
#include "logging.h"
#include "vector.h"

static inline
SndFormat snd_format_from_sdl2_size(int sdl2_fmt)
{
	switch (SDL_AUDIO_BITSIZE(sdl2_fmt)) {
	case 8:   return SND_FMT_U8;
	case 16:  return SND_FMT_S16;
	case 32:  return SND_FMT_F32;
	default:  return SND_FMT_NONE;  //should not happen
	}
}

void audio_sdl_free(AudioSdl* S)
{
	snd_free(&S->tmpsnd);
	//vec_free(S->buffer);
	
	if (S->dev) {
		SDL_CloseAudioDevice(S->dev);
		S->dev = 0;
	}

	if (S->sdl_init_flags) {
		SDL_QuitSubSystem(S->sdl_init_flags);
		S->sdl_init_flags = 0;
	}
}

void audio_sdl_stop(AudioSdl* S)
{
	SDL_PauseAudioDevice(S->dev, 1);
	SDL_ClearQueuedAudio(S->dev);
	//vec_resize(S->buffer, 0);
	//S->cursor = 0;
}

/*void SDLCALL audio_sdl_callback(void *userdata, Uint8 * stream, int len)
{
	AudioSdl *S = (AudioSdl*) userdata;

	unsigned bsz = vec_count(S->buffer);
	int rem = (int)bsz - (int)S->cursor;
	ccCLAMP(rem, 0, len);

	if (rem > 0) {
		memcpy(stream, S->buffer + S->cursor, rem);
		S->cursor += rem;
	}

	if (len > rem) {
		memset(stream + rem, 0, len - rem);
	}

	if (S->cursor >= bsz) {
		audio_sdl_stop(S);
	}
}*/

int audio_sdl_init(AudioSdl* S)
{
	int R=1;
	SDL_AudioSpec wanted;

	if (S->dev) audio_sdl_free(S);
	
	if (!S->sdl_init_flags) {
		S->sdl_init_flags = SDL_INIT_AUDIO;
		if (SDL_InitSubSystem(S->sdl_init_flags) < 0)
			ERROR_LOG(-1, "Unable to initialize SDL: %s", SDL_GetError());

		log_debug("Audio driver: %s", SDL_GetCurrentAudioDriver());
	}

	SDL_zero(wanted);
	//TODO: configurable
	wanted.freq = 44100;
	wanted.format = AUDIO_S16SYS;
	wanted.channels = 1;
	//wanted.samples = 4096;
	//wanted.callback = audio_sdl_callback;
	//wanted.userdata = (void*) S;

	SDL_zero(S->spec);

	S->dev = SDL_OpenAudioDevice(NULL, SDL_FALSE, &wanted, &S->spec,
		SDL_AUDIO_ALLOW_ANY_CHANGE);
	if (!S->dev)
		ERROR_LOG(-1, "Couldn't open an audio device for playback: %s",
			SDL_GetError());

	log_debug("Audio spec: freq=%d ch=%d samp=%d size=%d fmt=0x%x",
		S->spec.freq, S->spec.channels, S->spec.samples,
		S->spec.size, S->spec.format);
	
	S->snd_fmt = snd_format_from_sdl2_size(S->spec.format);

	//
	//if (S->cvt.needed)
	//	log_debug("Audio convertion needed");

end:
	return R;
}

int audio_sdl_sound_convert(AudioSdl* S, Sound* dst, const Sound* src)
{
	const SndFormatAttr * fa = snd_format_attr(src->format);

	if (audio_sdl_sound_spec_ready(S, src))
	{	// No convertion needed
		if (dst != src)
			snd_copy(dst, src);
		return 0;
	}

	if (dst == src && !snd_contiguous_is(src)) {
		return -1;
	}

	if (S->cvt_format != (int)fa->sdl2 ||
		S->cvt_freq != (int)src->freq ||
		S->cvt_ch != (int)src->ch)
	{	// Prepare the conversion
		log_debug("SDL_BuildAudioCVT");
		S->cvt_format = fa->sdl2;
		S->cvt_freq = src->freq;
		S->cvt_ch = src->ch;
		int r = SDL_BuildAudioCVT(&S->cvt,
			S->cvt_format, S->cvt_ch, S->cvt_freq,
			S->spec.format, S->spec.channels, S->spec.freq);
		if (r < 0) {
			log_error("SDL_BuildAudioCVT: %s", SDL_GetError());
			S->cvt_format = 0;
			S->cvt_freq = 0;
			S->cvt_ch = 0;
			return -1;
		}
	}
		
	// Conversion
	SndFormat dfmt = snd_format_from_sdl2_size(S->spec.format);
	if (dfmt == SND_FMT_NONE) return -1;

	unsigned n_byte = src->len * src->ch * src->ss;

	snd_copy(dst, src);
	if (S->cvt.len_mult > 1)  //Must reserve memory
		snd_resize(dst, dst->len * S->cvt.len_mult, 0, 0, 0);

	S->cvt.buf = (Uint8*) dst->data;
	S->cvt.len = n_byte;
	if (SDL_ConvertAudio(&S->cvt) < 0) {
		log_error("SDL_ConvertAudio: %s", SDL_GetError());
		return -1;
	}

	unsigned sz_sample = S->spec.channels * snd_format_attr(dfmt)->size;
	unsigned n_sample = S->cvt.len_cvt / sz_sample;
	snd_resize(dst, n_sample, S->spec.channels, S->spec.freq, dfmt);

	return 1;
}

int audio_sdl_sound_queue(AudioSdl* S, const Sound* snd, int flags)
{
	if (!audio_sdl_sound_spec_ready(S, snd)) {
		log_debug("sound queue convert");
		TRYR( audio_sdl_sound_convert(S, &S->tmpsnd, snd) );
		snd = &S->tmpsnd;		
	}

	unsigned n_byte = snd->len * snd->ch * snd->ss;

	//SDL_LockAudioDevice(S->dev);
	//
	//if (flags & AUDIO_QF_TRUNCATE) {
	//	vec_copy(S->buffer, n_byte, snd->data);
	//	S->cursor = 0;
	//} else {
	//	vec_append(S->buffer, n_byte, snd->data);
	//}
	//
	//SDL_UnlockAudioDevice(S->dev);
	//
	//if (S->cursor == 0)
	//	SDL_PauseAudioDevice(S->dev, 0);  //Start
	
	if (flags & AUDIO_QF_TRUNCATE)
		SDL_ClearQueuedAudio(S->dev);
	SDL_QueueAudio(S->dev, snd->data, n_byte);
	SDL_PauseAudioDevice(S->dev, 0);

	return 1;
}

int audio_sdl_beep(AudioSdl* S, SndWaveType type, double freq, double amp,
	double dur)
{
	unsigned srate = S->spec.freq;
	SndFormat fmt = snd_format_from_sdl2_size(S->spec.format);
	snd_resize(&S->tmpsnd, srate*dur, S->spec.channels, srate, fmt);
	snd_zero(&S->tmpsnd);
	snd_wave_add(&S->tmpsnd, type, freq, amp, 0);
	snd_fade(&S->tmpsnd, SND_CT_LINEAR, 1, 0, 0);
	return audio_sdl_sound_queue(S, &S->tmpsnd, 0);
}
