/* Copyright 2025-2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: MIT
 *
 * Hexagonal game board abstraction.
 */
#pragma once
#include <stddef.h>
#include <stdbool.h>
#include "ccommon/vector.h"

/*    Edges          Nodes
	  __1__         1_____2
	0/     \2       /     \
	/       \     0/       \3
	\       /      \       /
	5\_____/3       \_____/ 
	    4           5     4

	Positions mesured with respecto the root tile center.
	Edge length = 1
 */
typedef struct {
	int tiles[6];  //tl, top, tr, br, bottom, bl
	int edges[6];  //tl, top, tr, br, bottom, bl
	int nodes[6];  //left, tl, tr, right, br, bl
	float x, y;  //Center position
} BoardHexTile;

typedef struct {
	int tiles[2];  //below, above
	int edges[4];
	int nodes[2];
	float x1,y1, x2,y2;  //Extremes positions
} BoardHexEdge;

typedef struct {
	int tiles[3];
	int edges[3];
	int nodes[3];
	float x, y;  //Position
} BoardHexNode;

typedef struct {
	BoardHexTile * tiles;  //vector
	BoardHexEdge * edges;  //vector
	BoardHexNode * nodes;  //vector
} BoardHex;

void board_hex_free(BoardHex* S);

void board_hex_reserve(BoardHex* S, unsigned n_tile, unsigned n_edge,
	unsigned n_node);

// Returns tile id (0)
int board_tile_root_add(BoardHex* S);

// Returns tile id (>0) or negative in case of error
int board_tile_add(BoardHex* S, int tile, int edge);

// Drawing / export

// Coordinate origin at the center of the root tile
// Edge length = 1
typedef struct {
	int (*tile_draw)(void* uesr, int id, float x, float y);
	int (*edge_draw)(void* user, int id, float x1, float y1, float x2, float y2);
	int (*node_draw)(void* user, int id, float x, float y);
} BoardHexDrawClass;

int board_draw(const BoardHex* S, void* user, const BoardHexDrawClass* C);

//enum BoardHexExportType {
//	BOARD_ET_NONE,
//	BOARD_ET_SVG,
//	BOARD_ET__COUNT,
//};
//
//int board_export(const BoardHex* S, DynStr* out, int type);

void board_debug_dump(const BoardHex* S, DynStr* out);

// Info

static inline
unsigned board_tile_count(const BoardHex* S) { return vec_count(S->tiles); }
static inline
unsigned board_edge_count(const BoardHex* S) { return vec_count(S->edges); }
static inline
unsigned board_node_count(const BoardHex* S) { return vec_count(S->nodes); }

// Geometric informacion

static inline
BoardHexTile board_hex_tile_get(const BoardHex* S, int id) {
	if (0 <= id && (unsigned)id < board_tile_count(S))
		return S->tiles[id];
	else
		return (BoardHexTile){0};
}

static inline
BoardHexEdge board_hex_edge_get(const BoardHex* S, int id) {
	if (0 <= id && (unsigned)id < board_edge_count(S))
		return S->edges[id];
	else
		return (BoardHexEdge){0};
}

static inline
BoardHexNode board_hex_node_get(const BoardHex* S, int id) {
	if (0 <= id && (unsigned)id < board_node_count(S))
		return S->nodes[id];
	else
		return (BoardHexNode){0};
}
