/* Copyright 2025-2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: MIT
 */
#include "board_hex.h"

typedef BoardHexTile Tile;
typedef BoardHexEdge Edge;
typedef BoardHexNode Node;

// Tile half height
#define HH  0.86602540378443864676
static const float xnode[6] = {-1, -0.5, +0.5, +1, +0.5, -0.5};
static const float ynode[6] = {0, HH, HH, 0, -HH, -HH};

static const float xtile[6] = {-1.5, 0, +1.5, +1.5, 0, -1.5};
static const float ytile[6] = {HH, HH*2, HH, -HH, -HH*2, -HH};

void board_hex_free(BoardHex* S)
{
	vec_free(S->nodes);
	vec_free(S->edges);
	vec_free(S->tiles);
}

void board_hex_reserve(BoardHex* S, unsigned n_tile, unsigned n_edge,
	unsigned n_node)
{
	vec_realloc(S->tiles, n_tile);
	vec_realloc(S->edges, n_edge);
	vec_realloc(S->nodes, n_node);
}

static inline
int board_tile_new(BoardHex* S)
{
	int idx = vec_count(S->tiles);
	Tile tile = { {-1,-1,-1,-1,-1,-1}, {-1,-1,-1,-1,-1,-1}, {-1,-1,-1,-1,-1,-1} };
	vec_push(S->tiles, tile);
	return idx;
}

static inline
int board_edge_new(BoardHex* S)
{
	int idx = vec_count(S->edges);
	Edge edge = { {-1,-1}, {-1,-1,-1,-1}, {-1,-1} };
	vec_push(S->edges, edge);
	return idx;
}

static inline
int board_node_new(BoardHex* S)
{
	int idx = vec_count(S->nodes);
	Node node = { {-1,-1,-1}, {-1,-1,-1}, {-1,-1,-1} };
	vec_push(S->nodes, node);
	return idx;
}

// Set an index.
// Asserts that no overwrite should happen.
#define SET_IDX(X, Y) do { \
	assert( (X) == -1 || (X) == (Y) ); \
	(X) = (Y); \
} while(0)

/* Sets edge e to be connected to a new tile t at position j.
 */
static inline
void edge_tile_connect(BoardHex* S, int e, int t, int j)
{
	assert( 0 <= j && j < 6 );
	assert( vec_idx_check(S->tiles, t) );
	assert( vec_idx_check(S->edges, e) );

	const Tile * tile = &S->tiles[t];
	Edge * edge = &S->edges[e];

	int k = j/3;  //0 = top of tile, 1 = bottom of tile
	SET_IDX( edge->tiles[k], t );
	SET_IDX( edge->nodes[0+k], tile->nodes[j] );
	SET_IDX( edge->nodes[1-k], tile->nodes[(j+1)%6] );
	SET_IDX( edge->edges[0+k*2], tile->edges[(j+5)%6] );
	SET_IDX( edge->edges[1+k*2], tile->edges[(j+1)%6] );
}

/* Sets node n to be connected to a new tile t at position j.
 */
static inline
void node_tile_connect(BoardHex* S, int n, int t, int j)
{
	assert( 0 <= j && j < 6 );
	assert( vec_idx_check(S->tiles, t) );
	assert( vec_idx_check(S->nodes, n) );

	const Tile * tile = &S->tiles[t];
	Node * node = &S->nodes[n];

	SET_IDX( node->tiles[j%3], t );
	SET_IDX( node->edges[j%3    ], tile->edges[j] );
	SET_IDX( node->edges[(j+2)%3], tile->edges[(j+5)%6] );
	SET_IDX( node->nodes[j%3    ], tile->nodes[(j+1)%6] );
	SET_IDX( node->nodes[(j+2)%3], tile->nodes[(j+5)%6] );
}

/* Create missing edges and nodes.
 * Connect all edges and nodes with the new tile.
 */
static inline
void tile_complete(BoardHex* S, int t)
{
	Tile * tile = &S->tiles[t];

	for (int j=0; j<6; ++j) {
		if (tile->edges[j] == -1) {
			int e = tile->edges[j] = board_edge_new(S);
			Edge * edge = &S->edges[e];
			edge->x1 = tile->x + xnode[j];
			edge->y1 = tile->y + ynode[j];
			edge->x2 = tile->x + xnode[(j+1)%6];
			edge->y2 = tile->y + ynode[(j+1)%6];
		}
		if (tile->nodes[j] == -1) {
			int n = tile->nodes[j] = board_node_new(S);
			Node * node = &S->nodes[n];
			node->x = tile->x + xnode[j];
			node->y = tile->y + ynode[j];
		}
	}

	// Connections must be set up after all edges and nodes have been created.
	for (unsigned j=0; j<6; ++j) {
		edge_tile_connect(S, tile->edges[j], t, j);
		node_tile_connect(S, tile->nodes[j], t, j);
	}
}

int board_tile_root_add(BoardHex* S)
{
	if (board_tile_count(S) >= 1) return 0;
	int i_tile = board_tile_new(S);
	tile_complete(S, i_tile);
	return i_tile;
}

// Connect a new tile t1 to an already-setup one t2
// at edge position j2 in the second one.
static inline
int tiles_connect(BoardHex* S, int t1, int t2, int j2)
{
	assert( t1 != t2 );
	assert( 0 <= j2 && j2 < 6 );
	assert( vec_idx_check(S->tiles, t1) );
	assert( vec_idx_check(S->tiles, t2) );

	Tile * tile1 = &S->tiles[t1];
	Tile * tile2 = &S->tiles[t2];
	int j1 = (j2+3)%6;  //Reciprocal

	SET_IDX( tile2->tiles[j2], t1 );
	SET_IDX( tile1->tiles[j1], t2 );
	SET_IDX( tile1->edges[j1], tile2->edges[j2] );
	SET_IDX( tile1->nodes[j1], tile2->nodes[(j2+1)%6] );
	SET_IDX( tile1->nodes[(j1+1)%6], tile2->nodes[j2] );

	return j1;
}

/*	Example situation:
      _____
	 /     \
	/   1   \__1__
	\       /     \2
	 \_____/0 new  \
	 /     \5      /
	/   2   \_____/3
	\       /  4
	 \_____/

    The new tile's edge 0 is connected with tile 1's edge 3.
	The new tile's edge 5 is connected with tile 2's edge 2.
	The other edges of the new tile are create.
 */
int board_tile_add(BoardHex* S, int i_tile2, int j_edge2)
{
	if (!(0 <= i_tile2 && (unsigned)i_tile2 < board_tile_count(S))) return -1;
	if (!(0 <= j_edge2 && j_edge2 < 6)) return -1;
	if (S->tiles[i_tile2].tiles[j_edge2] != -1)
		return -1;  //Space already occupied

	// Create new tile
	int i_tile = board_tile_new(S);
	Tile * tile = &S->tiles[i_tile];

	int j_edge = tiles_connect(S, i_tile, i_tile2, j_edge2);
	
	tile->x = S->tiles[i_tile2].x + xtile[j_edge2];
	tile->y = S->tiles[i_tile2].y + ytile[j_edge2];
	
	// Connect to neighboring tiles
	// Clockwise
	for (int dj=1; dj<=4; ++dj) {
		int j = (j_edge+dj)%6;
		assert( tile->tiles[j] == -1 );
		
		int t2 = tile->tiles[(j+5)%6];
		assert( t2 != -1 );
		
		int t3 = S->tiles[t2].tiles[(j+1)%6];
		if (t3 == -1) break;
		tiles_connect(S, i_tile, t3, (j+3)%6);
	}
	// CCW
	for (int dj=1; dj<=4; ++dj) {
		int j = (j_edge+6-dj)%6;
		if (tile->tiles[j] != -1) break;
		
		int t2 = tile->tiles[(j+1)%6];
		assert( t2 != -1 );
		
		int t3 = S->tiles[t2].tiles[(j+5)%6];
		if (t3 == -1) break;
		tiles_connect(S, i_tile, t3, (j+3)%6);
	}

	// Create missing edges and nodes
	tile_complete(S, i_tile);

	return i_tile;
}

int board_draw(const BoardHex* S, void* user, const BoardHexDrawClass* C)
{
	// Tiles
	vec_for(S->tiles, t, 0) {
		const Tile * tile = &S->tiles[t];
		TRYR( C->tile_draw(user, t, tile->x, tile->y) );
	}
	
	// Edges
	vec_for(S->edges, t, 0) {
		const Edge * edge = &S->edges[t];
		TRYR( C->edge_draw(user, t, edge->x1, edge->y1, edge->x2, edge->y2) );
	}
	
	// Nodes
	vec_for(S->nodes, t, 0) {
		const Node * node = &S->nodes[t];
		TRYR( C->node_draw(user, t, node->x, node->y) );
	}

	return 0;
}

//const BoardHexDrawClass board_dcls_debug = {
//};

int board_export(const BoardHex* S, DynStr* out, int type)
{
	return -1; //TODO
}

static inline
void int_list_to_str(DynStr* out, unsigned count, int list[count])
{
	dstr_appendz(*out, " {");
	for (unsigned j=0; j<count; ++j) {
		if (j) dstr_push(*out, ',');
		if (list[j] == -1)
			dstr_appendz(*out, " *");
		else
			dstr_printfa(*out, "%2d", list[j]);
	}
	dstr_appendz(*out, "}");
}

void board_debug_dump(const BoardHex* S, DynStr* out)
{
	dstr_resize(*out, 0);
	vec_for(S->tiles, i, 0) {
		dstr_printfa(*out, "tile %3d:", i);
		int_list_to_str(out, 6, S->tiles[i].tiles);
		int_list_to_str(out, 6, S->tiles[i].edges);
		int_list_to_str(out, 6, S->tiles[i].nodes);
		dstr_appendz(*out, "\n");
	}
}

/*BoardHexEdge board_hex_edge_get(const BoardHex* S, int id)
{
	BoardHexEdge edge = {{-1,-1}, {-1,-1,-1,-1}, {-1,-1}};
	int jt=0, jn=0, je=0;
	vec_for(S->tiles, i, 0) {
		for (int j=0; j<6; ++j) {
			if (S->tiles[i].edges[j] == id) {
				assert(jt < 2);
				assert(je < 4);
				edge.tiles[jt++] = i;
				edge.edges[je++] = S->tiles[i].edges[(j+5)%6];
				edge.edges[je++] = S->tiles[i].edges[(j+1)%6];
				if (jn == 0) {
					edge.nodes[jn++] = S->tiles[i].nodes[j];
					edge.nodes[jn++] = S->tiles[i].nodes[(j+1)%6];
				}
			}
		}
	}
	return edge;
}*/

/*BoardHexNode board_hex_node_get(const BoardHex* S, int id)
{
	BoardHexNode node = {{-1,-1,-1}, {-1,-1,-1}};
	int jt=0, je=0, jn=0;
	vec_for(S->tiles, i, 0) {
		for (int j=0; j<6; ++j) {
			if (S->tiles[i].nodes[j] == id) {
				assert(jt < 3);
				assert(je < 3);
				node.tiles[jt++] = i;
				node.edges[je++] = S->tiles[i].edges[j];
				node.nodes[jn++] = S->tiles[i].nodes[(j+1)%6];
			}
		}
	}
	return node;
}*/
