/* Copyright 2025-2026, Alejandro A. García <aag@zorzal.net>
 * SPDX-License-Identifier: MIT
 */
#include "ai.h"
#include "ccommon/ccommon.h"
#include "ccommon/bisect.h"
#include "ccommon/vector.h"
#include "ccommon/logging.h"

static inline
void sorted_index_make_sta(unsigned n, int out[n], const int vec[n])
{
	for (unsigned i=0; i<n; ++i) {
		BISECT_RIGHT_DECL(found, idx, 0, i, vec[i] - vec[out[i_]] );
		if (idx < i) memmove(out+(idx+1), out+idx, (i-idx)*sizeof(*out));
		out[idx] = i;
	}
}

static inline
void sorted_index_make_vec(int** out, const int* vec)
{
	vec_realloc(*out, vec_count(vec));
	vec_for(vec, i, 0) {
		BISECT_RIGHT_DECL(found, idx, 0, i, vec[i] - vec[(*out)[i_]] );
		vec_insert(*out, idx, 1, &i);
	}
}

typedef struct {
	int *tile_uty;
	int *edge_uty;
	int *node_uty;
	int *node_uty_tr;
	int *node_conn;
	int *tile_index;
	int *edge_index;
	int *node_index;
	int res_uty[GAME_RES__COUNT];
	int res_index[GAME_RES__COUNT];
	int n_node_new;
	unsigned thief_affecting:1;
} GameAiState;

void game_ai_state_free(GameAiState* A)
{
	vec_free(A->node_index);
	vec_free(A->edge_index);
	vec_free(A->tile_index);
	vec_free(A->node_conn);
	vec_free(A->node_uty);
	vec_free(A->edge_uty);
	vec_free(A->tile_uty);
}

void game_ai_analyze(Game* S, int pl, GameAiState* A)
{
	const GamePlayer * player = &S->pls[pl];
	const float * num_prob = S->rule.num_prob;
	
	// During setup we place first a settlement anywhere and then,
	// a road connected to the last settlement.
	bool is_setup = (S->state == GAME_STATE_SETUP);

	// Maximum cost for each resource
	int res_cost_max[GAME_RES__COUNT] = {0};
	for (int r=0; r<GAME_RES__COUNT; ++r) {
		MAXSET(res_cost_max[r], S->rule.cost_edge[r]);
		MAXSET(res_cost_max[r], S->rule.cost_node[0][r]);
		MAXSET(res_cost_max[r], S->rule.cost_node[1][r]);
	}

	// Generic trade rate for this player
	int res_tq_min = 255;
	for (int r=0; r<GAME_RES__COUNT; ++r)
		MINSET(res_tq_min, player->res_tq[r]);

	// Mean resource production in N turns
	int res_prod_turns = 20;
	int res_prod[GAME_RES__COUNT] = {0};
	vec_for(S->nodes, n, 0) {
		if (S->nodes[n].pl != pl) continue;
		int mult = (S->nodes[n].lvl + 1) * res_prod_turns;
		for (int j=0; j<3; ++j) {
			int t = S->brd.nodes[n].tiles[j];
			if (t == -1) continue;
			int r = S->tiles[t].res;
			float p = num_prob[ S->tiles[t].num ];
			res_prod[r] += p * mult + 0.5;
		}
	}

	// Resource utility / weight / priority
	MEM_ZERO(A->res_uty);

	int e_n_rel = player->n_edge - player->n_node*2;
	for (int i=e_n_rel; i<=0; ++i)  // More edges
		game_res_add(A->res_uty, S->rule.cost_edge);
	for (int i=e_n_rel; i>=0; --i)  // More nodes
		game_res_add(A->res_uty, S->rule.cost_node[0]);
	if (player->n_node_lvl[0] > 2)  // More node upgrades
		game_res_add(A->res_uty, S->rule.cost_node[1]);

	{
		A->thief_affecting = 0;
		int t = S->thief_tile;
		for (int j=0; j<6; ++j) {
			int n = S->brd.tiles[t].nodes[j];
			if (S->nodes[n].pl == pl) { // Need a devcard to move the thief
				game_res_add(A->res_uty, S->rule.cost_devcard);
				A->thief_affecting = 1;
			}
		}
	}

	// - Substract current resources and estimated production for the next 8 turns
	for (int r=0; r<GAME_RES__COUNT; ++r) {
		A->res_uty[r] -= player->res[r] + res_prod[r];
		// Even lower utility for accumulating a resource in excess
		int ex = player->res[r] - res_cost_max[r];
		if (ex > 0) A->res_uty[r] -= ex;
	}
	
	// - Shift to value to >= 1
	{
		int min=255;
		for (int r=0; r<GAME_RES__COUNT; ++r) MINSET(min, A->res_uty[r]);
		for (int r=0; r<GAME_RES__COUNT; ++r) A->res_uty[r] += 1 - min;
	}
	
	if (is_setup) {
		for (int r=0; r<GAME_RES__COUNT; ++r) A->res_uty[r] = 1;
		A->res_uty[GAME_RES_WOOD]++;
		A->res_uty[GAME_RES_BRICK]++;
	}

	// Store resource list ordered by descending utility
	sorted_index_make_sta(GAME_RES__COUNT, A->res_index, A->res_uty);

	// Rate positions
	vec_resize_zero(A->tile_uty, vec_count(S->tiles));
	vec_resize_zero(A->edge_uty, vec_count(S->edges));
	vec_resize_zero(A->node_uty, vec_count(S->nodes));
	vec_resize_zero(A->node_uty_tr, vec_count(S->nodes));
	vec_resize_zero(A->node_conn, vec_count(S->nodes));
	vec_resize_zero(A->node_index, vec_count(S->nodes));

	vec_for(S->tiles, t, 0) {
		int res = S->tiles[t].res;
		if (res == -1) continue;
		int prob = num_prob[ S->tiles[t].num ] * 100;
		for (int j=0; j<6; ++j) {
			int n = S->brd.tiles[t].nodes[j];

			// Rate tiles according to the enemy settlements production
			// Substract own settlements
			if (S->nodes[n].pl == pl)
				A->tile_uty[t] -= (S->nodes[n].lvl+1) * prob;
			else if (S->nodes[n].pl != -1)
				A->tile_uty[t] += (S->nodes[n].lvl+1) * prob;
		}
	}

	vec_for(S->nodes, n, 0) {
		// Is this node connected?
		bool is_conn = false;
		for (int j=0; j<3; ++j) {
			int e = S->brd.nodes[n].edges[j];
			if (e != -1 && S->edges[e].pl == pl) {
				is_conn = true;
				break;
			}
		}
		// During setup we must built an edge connected to the last node
		if (is_setup)  is_conn = ((int)n == player->last_node);
		
		A->node_conn[n] = is_conn;

		// Can be built?
		bool is_avail = (S->nodes[n].pl == -1 ||
			(S->nodes[n].pl == pl && S->nodes[n].lvl+1 < GAME_SETTLE_LEVEL_COUNT));

		// Is it available under the farness condition?
		for (int j=0; j<3; ++j) {
			int n2 = S->brd.nodes[n].nodes[j];
			if (n2 != -1 && S->nodes[n2].pl != -1)
				is_avail = false;
		}

		if (!is_avail) continue;

		// Resource utility
		int uty_res=0;
		for (int j=0; j<3; ++j) {
			int t = S->brd.nodes[n].tiles[j];
			if (t == -1) continue;
			int res = S->tiles[t].res;
			if (res == -1) continue;
			int prob = num_prob[ S->tiles[t].num ] * 100;
			uty_res += (A->res_uty[res] + 2) * prob;
		}
		A->node_uty[n] += uty_res;

		// Trade route utility
		// Only for new node, not for upgrade
		int tr = S->nodes[n].tr;
		if (tr >= 0 && S->nodes[n].pl == -1) {
			int uty_tr = 0;
			int tr_res = S->troutes[tr].res;
			if (tr_res == -1) {
				if (res_tq_min > S->troutes[tr].qty)
					uty_tr += 30;
			} else {
				if (player->res_tq[tr_res] > S->troutes[tr].qty) {
					int n = player->res[tr_res] + res_prod[tr_res];
					uty_tr += n * 3;
				}
			}
			if (S->n_round == 0) uty_tr *= 3;
			A->node_uty_tr[n] += uty_tr;
		}
	}
	
	vec_for(S->nodes, n, 0) {
		// Accumulate neighboring nodes utilities
		A->node_index[n] = 0;  // Used as a temporal variable
		for (int j=0; j<3; ++j) {
			int n2 = S->brd.nodes[n].nodes[j];
			if (n2 == -1) continue;
			A->node_index[n] += A->node_uty[n2];
		}
	}
	
	vec_for(S->nodes, n, 0) {
		// Substract from this node's utility part of the neighboring one
		// because if one buils here, one can not built in the neighbors.
		A->node_uty[n] -= A->node_index[n] / 4;
		if (A->node_uty[n] < 0) A->node_uty[n] = 0;
	}

	vec_for(S->edges, e, 0) {
		if (S->edges[e].pl != -1) continue;
		int n1 = S->brd.edges[e].nodes[0];
		int n2 = S->brd.edges[e].nodes[1];
		if (!(A->node_conn[n1] || A->node_conn[n2])) continue;
		
		// Rate edges according to their unoccupied unconnected neighboring nodes
		int uty_res=0, uty_tr=0;
		for (int j=0; j<2; ++j) {
			int n = S->brd.edges[e].nodes[j];
			if (!(A->node_conn[n]==0 && S->nodes[n].pl == -1)) continue;
			MAXSET(uty_res, A->node_uty[n]);
			MAXSET(uty_tr, A->node_uty_tr[n]);
		}
		// Consider also nodes two steps away with half weight
		for (int j1=0; j1<2; ++j1) {
			int n = S->brd.edges[e].nodes[j1];
			for (int j2=0; j2<3; ++j2) {
				int n_ = S->brd.nodes[n].nodes[j2];
				if (n_ == -1 || n_ == n1 || n_ == n2) continue;
				if (!(A->node_conn[n_]==0 && S->nodes[n_].pl == -1)) continue;
				uty_res += A->node_uty[n_] / 4;
				MAXSET(uty_tr, A->node_uty_tr[n_] / 2);
			}
		}
		
		int uty = uty_res + uty_tr;

		// Bonus for bridging other edges
		// During setup it is best to choose separated edges to have space to grow
		if (!is_setup && A->node_conn[n1] && A->node_conn[n2])
			uty += 100;
		
		A->edge_uty[e] += uty;
	}

	vec_for(S->nodes, n, 0) {
		A->node_uty[n] += A->node_uty_tr[n];
		if (A->node_uty[n] <= 0) continue;
		if (!A->node_conn[n]) {
			// Rate zero nodes not connected
			if (!is_setup) A->node_uty[n] = 0;
		}
		else if (S->nodes[n].pl == -1) {
			// Count ready to use nodes spaces
			A->n_node_new++;
		}
		else if (is_setup) A->node_uty[n] = 0;  // No upgrade in setup
	}

	// Make sorted indices
	sorted_index_make_vec(&A->tile_index, A->tile_uty);
	sorted_index_make_vec(&A->edge_index, A->edge_uty);
	sorted_index_make_vec(&A->node_index, A->node_uty);

	if (log_line_begin(LOG_LVL_DEBUG2)) {
		log_line_strf("ai %d res utility:", pl);
		for (int r=0; r<GAME_RES__COUNT; ++r)
			log_line_strf(" %d", A->res_uty[r]);
		log_line_end();
	}
	if (log_line_begin(LOG_LVL_DEBUG2)) {
		log_line_strf("ai %d tile ranks:", pl);
		for (int i=0; i<5; ++i)
			log_line_strf(" %d:%d",
				A->tile_index[i], A->tile_uty[A->tile_index[i]]);
		log_line_end();
	}
	if (log_line_begin(LOG_LVL_DEBUG2)) {
		log_line_strf("ai %d edge ranks:", pl);
		for (int i=0; i<5; ++i)
			log_line_strf(" %d:%d",
				A->edge_index[i], A->edge_uty[A->edge_index[i]]);
		log_line_end();
	}
	if (log_line_begin(LOG_LVL_DEBUG2)) {
		log_line_strf("ai %d node ranks:", pl);
		for (int i=0; i<5; ++i)
			log_line_strf(" %d:%d",
				A->node_index[i], A->node_uty[A->node_index[i]]);
		log_line_end();
	}
}

int game_player_action_ai(Game* S, int pl)
{
	if (pl != S->pl_curr) return 0;
	GamePlayer * player = &S->pls[pl];

	// Analize game state
	GameAiState ais={0};

	// Special case: initial setup
	if (S->state == GAME_STATE_SETUP) {
		// Build one road and one settlement
		game_ai_analyze(S, pl, &ais);
		game_player_action(S, pl, GAME_PA_BUILD_NODE, ais.node_index[0], 0);
		game_ai_analyze(S, pl, &ais);
		game_player_action(S, pl, GAME_PA_BUILD_EDGE, ais.edge_index[0], 0);
		goto end;
	}

	// Action loop
	while (1) {
		game_ai_analyze(S, pl, &ais);
		int n_act=0, r;

		if (S->state == GAME_STATE_THIEF_MOVE) {
			// Move thief to the highest rated tile
			int t = ais.tile_index[0];
			if (t == S->thief_tile) t = ais.tile_index[1];
			game_player_action(S, pl, GAME_PA_THIEF_MOVE, t, 0);
			continue;
		}

		// Dev.card use
		for (unsigned j=0; j<COUNTOF(player->devcards); ++j) {
			if (player->devcards[j] == -1) continue;
			if (player->devcards_lock[j]) continue;
			if (player->devcards[j] == 0 && ais.thief_affecting) {
				r = game_player_action(S, pl, GAME_PA_DEVCARD_USE, j, 0);
				if (r > 0) n_act++;
			}
		}
		if (n_act) continue;

		// Build settlements
		for (int i=0; i<5; ++i) {
			log_debug("ai %d node %d", pl, ais.node_index[i]);
			r = game_player_action(S, pl, GAME_PA_BUILD_NODE, ais.node_index[i], 0);
			if (r > 0) n_act++;
		}
		if (n_act) continue;
		
		// Build roads
		for (int i=0; i<3 && ais.n_node_new <= 1; ++i) {
			log_debug("ai %d edge %d", pl, ais.edge_index[i]);
			r = game_player_action(S, pl, GAME_PA_BUILD_EDGE, ais.edge_index[i], 0);
			if (r > 0) n_act++;
		}
		if (n_act) continue;

		// Dev.card buy
		if (player->n_devcard < 1 || ais.thief_affecting) {
			log_debug("ai %d dev.card buy", pl);
			r = game_player_action(S, pl, GAME_PA_DEVCARD_BUY, 0, 0);
			if (r > 0) n_act++;
		}
		if (n_act) continue;

		// Try to trade the less valuable resources for the more valuables
		for (int i1=0; i1<GAME_RES__COUNT && n_act==0; ++i1) {
			int r1 = ais.res_index[GAME_RES__COUNT-1-i1];
			int q1 = player->res_tq[r1];
			if (!(player->res[r1] >= q1)) continue;
			for (int i2=0; i2<GAME_RES__COUNT; ++i2) {
				int r2 = ais.res_index[i2];
				
				// Check if the trade will improve the utility
				int du = ais.res_uty[r2] - ais.res_uty[r1] * q1;
				int ex = game_res_count(player->res) - S->rule.thief_res_min + 1;
				if (ex > 0 ) du += ex;
				if (du < 0) continue;
				
				r = game_player_action(S, pl, GAME_PA_TRADE_RES, r1, r2);
				if (r > 0) n_act++;
				break;
			}
		}
		if (n_act) continue;

		break;
	}

end:
	game_ai_state_free(&ais);
	return 1;
}

void game_ai_thief_res(Game* S, int pl, int res[GAME_RES__COUNT], int n_remove)
{
	GamePlayer * player = &S->pls[pl];
	GameAiState ais={0};
	
	for (int i=0; i<GAME_RES__COUNT; ++i) res[i] = 0;

	game_ai_analyze(S, pl, &ais);
	
	for (int pass=0; pass<=3 && n_remove>0; ++pass) {
		// Maximum cost for each resource
		int res_cost_max[GAME_RES__COUNT] = {0};
		for (int r=0; r<GAME_RES__COUNT; ++r) {
			if (pass <= 2)
				MAXSET(res_cost_max[r], S->rule.cost_edge[r]);
			if (pass <= 1)
				MAXSET(res_cost_max[r], S->rule.cost_node[0][r]);
			if (pass <= 0)
				MAXSET(res_cost_max[r], S->rule.cost_node[1][r]);
		}

		for (int i=0; i<GAME_RES__COUNT && n_remove>0; ++i) {
			int r = ais.res_index[i];
			assert( 0 <= r && r < GAME_RES__COUNT );
			int ex = player->res[r] - res[r] - res_cost_max[r];
			if (ex <= 0) continue;
			MINSET(ex, n_remove);
			res[r] += ex;
			n_remove -= ex;
		}
	}

	assert( n_remove == 0 );
	
	game_ai_state_free(&ais);
}
